/*
 * WebWork, Web Application Framework
 *
 * Distributable under Apache license.
 * See terms of license at opensource.org
 */
package webwork.dispatcher;

import org.apache.commons.logging.*;
import webwork.action.Action;
import webwork.action.ServletActionContext;
import webwork.action.client.ActionResult;
import webwork.action.factory.ActionFactory;
import webwork.action.factory.ContextActionFactoryProxy;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.*;

/**
 * Dispatcher servlet that can be invoked by remote clients, such as applets.
 *
 * @author Rickard \u00D6berg (rickard@dreambean.com)
 * @version $Revision: 1.14 $
 */
public class ClientServletDispatcher extends HttpServlet
{
    // Attributes ----------------------------------------------------
    protected static final Log log = LogFactory.getLog(ClientServletDispatcher.class);

    // HttpServlet overrides -----------------------------------------
    /**
     * Initialize dispatcher servlet
     *
     * @throws ServletException
     */
    public void init(ServletConfig config)
            throws ServletException
    {
        super.init(config);
        log.info("Remote Action dispatcher initialized");
    }

    /**
     * Service a request
     *
     * @throws ServletException
     */
    public void service(HttpServletRequest aRequest, HttpServletResponse aResponse)
            throws ServletException
    {
        try
        {
            ObjectInputStream in = createObjectInputStream(aRequest.getInputStream());
            final Action action = (Action) in.readObject();

            ServletActionContext.setContext(aRequest, aResponse, getServletContext(), null);

            // Give the action its proper context
            new ContextActionFactoryProxy(new ActionFactory()
            {
                public Action getActionImpl(String aName) throws Exception
                {
                    return action;
                }
            }).getActionImpl(null);

            // Execute action
            ActionResult actionResult;

            try
            {
                String result = action.execute();

                actionResult = new ActionResult(action, result);
            }
            catch (Exception e)
            {
                actionResult = new ActionResult(action, null);
                actionResult.setException(e);
            }

            ObjectOutputStream out = createObjectOutputStream(aResponse.getOutputStream());

            out.writeObject(actionResult);
            out.flush();
        }
        catch (Exception e)
        {
            e.printStackTrace();
            throw new ServletException(e);
        }
    }

    // Protected -----------------------------------------------------
    /**
     * Create an object input stream that wraps the URL connection stream.
     * <p/>
     * This method can be overriden in order to create custom streams.
     *
     * @param in the underlying stream. It is buffered
     *
     * @return an object input stream
     *
     * @throws IOException
     */
    protected ObjectInputStream createObjectInputStream(InputStream in)
            throws IOException
    {
        return new ObjectInputStream(in);
    }

    /**
     * Create an object input stream that wraps the URL connection stream.
     * <p/>
     * This method can be overriden in order to create custom streams.
     *
     * @param out The OutputStream to wrap
     *
     * @return an ObjectOutputStream wrapping the specified OutputStream
     *
     * @throws IOException
     */
    protected ObjectOutputStream createObjectOutputStream(OutputStream out)
            throws IOException
    {
        return new ObjectOutputStream(out);
    }
}
