/*
 * WebWork, Web Application Framework
 *
 * Distributable under Apache license.
 * See terms of license at opensource.org
 */
package webwork.action;

import webwork.multipart.MultiPartRequestWrapper;
import webwork.util.ValueStack;

import java.security.Principal;
import java.util.Collections;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * Handles the context of each invoked action while providing an abstraction layer for both servlet and non servlet
 * based applications.
 * <p/>
 * Information is associated with the action's thread so it can be accessed from anywhere by calling a static method.
 * <p/>
 * The context is initially set by the dispatcher.
 *
 * @author Rickard \u00D6berg (rickard@middleware-company.com)
 * @author Matt Baldree (matt@smallleap.com)
 * @author Maurice C. Parker (maurice@vineyardenterprise.com)
 * @version $Revision: 1.14 $
 * @see webwork.dispatcher.ServletDispatcher
 * @see webwork.dispatcher.ClientServletDispatcher
 */
public class ActionContext
{
    private static final ThreadLocal<ActionContext> actionContext = new ThreadLocal<ActionContext>()
    {
        @Override
        protected ActionContext initialValue()
        {
            return new ActionContext();
        }
    };

    public static final String ACTION_NAME = "webwork.action.ActionContext.name";
    public static final String SESSION = "webwork.action.ActionContext.action";
    public static final String APPLICATION = "webwork.action.ActionContext.application";
    public static final String PARAMETERS = "webwork.action.ActionContext.parameters";
    public static final String SINGLE_VALUE_PARAMETERS = "webwork.action.ActionContext.singleValueParameters";
    public static final String LOCALE = "webwork.action.ActionContext.locale";
    public static final String PRINCIPAL = "webwork.action.ActionContext.principal";

    private static final String REQUEST = ServletActionContext.REQUEST;
    private static final String RESPONSE = ServletActionContext.RESPONSE;
    private static final String SERVLET_CONTEXT = ServletActionContext.SERVLET_CONTEXT;

    public static void setContext(final ActionContext aContext)
    {
        actionContext.set(aContext);
    }

    /**
     * Returns the ActionContext specific to the current thread.
     *
     * @return ActionContext for the current thread
     */
    public static ActionContext getContext()
    {
        final ActionContext context = actionContext.get();
        return context;
    }

    private Map lookup;

    public Map getTable()
    {
        return lookup;
    }

    public void setTable(final Map lookup)
    {
        this.lookup = lookup;
    }

    /**
     * Stores a value in the current ActionContext.  The value can be looked up using the key.
     *
     * @param key   The key of the value.
     * @param value The value to be stored.
     */
    public void put(final Object key, final Object value)
    {
        if (lookup == null)
        {
            lookup = new HashMap();
        }
        lookup.put(key, value);
    }

    /**
     * Returns a value that is stored in the current ActionContext buy doing a lookup using the value's key.
     *
     * @param key The key used to find the value.
     *
     * @return The value that was found using the key.
     */
    public Object get(final Object key)
    {
        if (lookup == null)
        {
            return null;
        }
        return lookup.get(key);
    }

    /**
     * Returns the name of the current Action.
     *
     * @return The current Action name.
     */
    public static String getName()
    {
        return getContext().getNameImpl();
    }

    public String getNameImpl()
    {
        return (String) get(ACTION_NAME);
    }

    /**
     * Stores the name of the current Action in the ActionContext.
     *
     * @param name The name of the current action.
     */
    public static void setName(final String name)
    {
        getContext().setNameImpl(name);
    }

    public void setNameImpl(final String name)
    {
        put(ACTION_NAME, name);
    }

    /**
     * Returns the Locale of the current request in a servlet environment or the default Locale in other environments.
     *
     * @return current locale
     */
    public static Locale getLocale()
    {
        return getContext().getLocaleImpl();
    }

    public Locale getLocaleImpl()
    {
        Locale locale = (Locale) get(LOCALE);
        if (locale == null)
        {
            locale = Locale.getDefault();
            put(LOCALE, locale);
        }
        return locale;
    }

    /**
     * Set the current locale.
     *
     * @param locale current locale
     */
    public static void setLocale(final Locale locale)
    {
        getContext().setLocaleImpl(locale);
    }

    public void setLocaleImpl(final Locale locale)
    {
        put(LOCALE, locale);
    }

    /**
     * Returns the HttpSession when in a servlet environment or a generic session map otherwise.
     *
     * @return a map of HttpSession or a generic session map
     */
    public static Map getSession()
    {
        return getContext().getSessionImpl();
    }

    public Map getSessionImpl()
    {
        Map session = (Map) get(SESSION);
        if (session == null)
        {
            session = new HashMap();
            put(SESSION, session);
        }
        return session;
    }

    /**
     * Set a session Map.
     */
    public static void setSession(final Map session)
    {
        getContext().setSessionImpl(session);
    }

    public void setSessionImpl(final Map session)
    {
        put(SESSION, session);
    }

    /**
     * Returns a Map of the ServletContext when in a servlet environment or a generic application level Map otherwise.
     *
     * @return Map of ServletContext or generic application level Map
     */
    public static Map getApplication()
    {
        return getContext().getApplicationImpl();
    }

    public Map getApplicationImpl()
    {
        Map application = (Map) get(APPLICATION);
        if (application == null)
        {
            application = new HashMap();
            put(APPLICATION, application);
        }
        return application;
    }

    /**
     * Set an application level Map.
     */
    public static void setApplication(final Map application)
    {
        getContext().setApplicationImpl(application);
    }

    public void setApplicationImpl(final Map application)
    {
        put(APPLICATION, application);
    }

    /**
     * Returns a Map of the HttpServletRequest parameters when in a servlet environment or a generic Map of parameters
     * otherwise.
     *
     * @return Map of HttpServletRequest parameters, generic Map of parameters, or multipart Map.
     */
    public static Map getParameters()
    {
        return getContext().getParametersImpl();
    }

    public Map getParametersImpl()
    {
        Map parameters = (Map) get(PARAMETERS);
        if (parameters == null)
        {
            parameters = Collections.EMPTY_MAP;
            put(PARAMETERS, parameters);
        }
        return parameters;
    }

    /**
     * Set a Map of parameters.
     *
     * @param parameters The parameters for the current action context.
     */
    public static void setParameters(final Map parameters)
    {
        getContext().setParametersImpl(parameters);
    }

    public void setParametersImpl(final Map parameters)
    {
        put(PARAMETERS, parameters);
    }

    /**
     * Get the current ActionContext parameters.  Each Map entry is a String.
     *
     * @return The parameters for the current action context.
     */
    public static Map getSingleValueParameters()
    {
        return getContext().getSingleValueParametersImpl();
    }

    public Map getSingleValueParametersImpl()
    {
        Map parameters = (Map) get(SINGLE_VALUE_PARAMETERS);
        if (parameters == null)
        {
            parameters = Collections.EMPTY_MAP;
            put(SINGLE_VALUE_PARAMETERS, parameters);
        }
        return parameters;
    }

    /**
     * Set a Map of single value parameters.
     *
     * @param parameters for the current action context.
     */
    public static void setSingleValueParameters(final Map parameters)
    {
        getContext().setSingleValueParametersImpl(parameters);
    }

    public void setSingleValueParametersImpl(final Map parameters)
    {
        put(SINGLE_VALUE_PARAMETERS, parameters);
    }

    /**
     * Returns the current user's security Principal.
     *
     * @return the current user's security Principal
     */
    public static Principal getPrincipal()
    {
        return getContext().getPrincipalImpl();
    }

    public Principal getPrincipalImpl()
    {
        return (Principal) get(PRINCIPAL);
    }

    /**
     * Set  the current user's security Principal.
     *
     * @param principal the current user's security Principal
     */
    public static void setPrincipal(final Principal principal)
    {
        getContext().setPrincipalImpl(principal);
    }

    public void setPrincipalImpl(final Principal principal)
    {
        put(PRINCIPAL, principal);
    }

    /**
     * Return multipart request for HttpServletRequest.
     *
     * @return MultiPartRequestWrapper
     */
    public static MultiPartRequestWrapper getMultiPartRequest()
    {
        return getContext().getMultiPartRequestImpl();
    }

    // add wrapRequest code here and throw IOException!!!
    public MultiPartRequestWrapper getMultiPartRequestImpl()
    {
        final HttpServletRequest request = getRequestImpl();
        if (request instanceof MultiPartRequestWrapper)
        {
            return (MultiPartRequestWrapper) request;
        }
        else
        {
            return null;
        }
    }

    /**
     * Returns the HttpServletRequest object when in a servlet environment.
     *
     * @return HttpServletRequest in a servlet environment or null otherwise
     */
    public static HttpServletRequest getRequest()
    {
        return getContext().getRequestImpl();
    }

    public HttpServletRequest getRequestImpl()
    {
        return (HttpServletRequest) get(REQUEST);
    }

    /**
     * Set the HttpServletRequest.
     */
    public static void setRequest(final HttpServletRequest request)
    {
        getContext().setRequestImpl(request);
    }

    public void setRequestImpl(final HttpServletRequest request)
    {
        put(REQUEST, request);
    }

    /**
     * Returns the HttpServletResponse when in a servlet environment.
     *
     * @return HttpServletResponse or null
     */
    public static HttpServletResponse getResponse()
    {
        return getContext().getResponseImpl();
    }

    public HttpServletResponse getResponseImpl()
    {
        return (HttpServletResponse) get(RESPONSE);
    }

    /**
     * Set the HttpServletResponse.
     */
    public static void setResponse(final HttpServletResponse response)
    {
        getContext().setResponseImpl(response);
    }

    public void setResponseImpl(final HttpServletResponse response)
    {
        put(RESPONSE, response);
    }

    /**
     * Returns the ServletContext when in a servlet environment.
     *
     * @return ServletContext or null.
     */
    public static ServletContext getServletContext()
    {
        return getContext().getServletContextImpl();
    }

    public ServletContext getServletContextImpl()
    {
        return (ServletContext) get(SERVLET_CONTEXT);
    }

    /**
     * Set the ServletContext.
     */
    public static void setServletContext(final ServletContext context)
    {
        getContext().setServletContextImpl(context);
    }

    public void setServletContextImpl(final ServletContext context)
    {
        put(SERVLET_CONTEXT, context);
    }

    /**
     * Returns the ValueStack specific to the current thread.
     *
     * @return ActionContext for the current thread
     */
    public static ValueStack getValueStack()
    {
        return getContext().getValueStackImpl();
    }

    public ValueStack getValueStackImpl()
    {
        ValueStack valueStack = (ValueStack) get(ValueStack.STACK_NAME);
        if (valueStack == null)
        {
            valueStack = new ValueStack();
            put(ValueStack.STACK_NAME, valueStack);
        }
        return valueStack;
    }

    /**
     * Set the ValueStack.
     *
     * @param valueStack
     */
    public static void setValueStack(final ValueStack valueStack)
    {
        getContext().setValueStackImpl(valueStack);
    }

    public void setValueStackImpl(final ValueStack valueStack)
    {
        put(ValueStack.STACK_NAME, valueStack);
    }
}
