/*
 * WebWork, Web Application Framework
 *
 * Distributable under Apache license.
 * See terms of license at opensource.org
 */
package webwork.view.taglib.ui;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.velocity.context.Context;
import webwork.config.Configuration;
import webwork.util.BeanUtil;
import webwork.util.ContainUtil;
import webwork.view.taglib.IncludeTag;
import webwork.view.taglib.ParamTag;
import webwork.view.taglib.WebWorkBodyTagSupport;
import webwork.view.velocity.VelocityHelper;

import java.util.HashMap;
import java.util.Map;
import javax.servlet.jsp.JspException;
import javax.servlet.jsp.JspTagException;

/**
 * Base class for UI tags.
 *
 * @author Rickard \u00D6berg (rickard@dreambean.com)
 * @version $Revision: 1.16 $
 */
public class ComponentTag
        extends WebWorkBodyTagSupport
        implements ParamTag.Parametric
{
    // Attributes ----------------------------------------------------
    protected static Log log = LogFactory.getLog(ComponentTag.class);
    protected String themeAttr;
    protected String templateAttr;
    protected String templateSuffix;
    protected String nameAttr;
    protected String valueAttr;
    protected String labelAttr;
    protected String labelPositionAttr;
    protected String theme;
    protected String tabindex;
    protected String title;
    protected String mandatory;
    protected String accesskey;
    protected Map params = new HashMap();
    protected String templateDir;

    // Public --------------------------------------------------------

    public String getTemplate()
    {
        // Should never get here, but if we do, then just show something
        return "text";
    }

    public String getTheme()
    {
        // If theme set is not explicitly given,
        // try to find attribute which states the theme set to use
        if ((theme == null) || (theme == ""))
        {
            theme = (String) pageContext.findAttribute("theme");
        }

        // Default template set
        if ((theme == null) || (theme == ""))
        {
            theme = Configuration.getString("webwork.ui.theme");
        }

        return theme;
    }

    public String getTemplateSuffix()
    {
        if (templateSuffix == null || templateSuffix == "")
        {
            templateSuffix = Configuration.getString("webwork.ui.templateSuffix");
        }
        return templateSuffix;
    }

    public String getTemplateDir()
    {
        // If templateDir is not explicitly given,
        // try to find attribute which states the dir set to use
        if ((templateDir == null) || (templateDir == ""))
        {
            templateDir = (String) pageContext.findAttribute("templateDir");
        }

        // Default template set
        if ((templateDir == null) || (templateDir == ""))
        {
            templateDir = Configuration.getString("webwork.ui.templateDir");
        }

        if ((templateDir == null) || (templateDir == ""))
        {
            templateDir = "template";
        }

        return templateDir;
    }

    public void setTheme(String aName)
    {
        themeAttr = aName;
    }

    public void setTemplate(String aName)
    {
        templateAttr = aName;
    }

    public void setLabel(String aLabel)
    {
        labelAttr = aLabel;
    }

    public void setName(String aName)
    {
        nameAttr = aName;
    }

    public void setValue(String aName)
    {
        valueAttr = aName;
    }

    public void setLabelposition(String aLabelPosition)
    {
        labelPositionAttr = aLabelPosition;
    }

    public void setTabindex(final String tabindex)
    {
        this.tabindex = tabindex;
    }

    public void setTitle(final String title)
    {
        this.title = title;
    }

    public void setAccesskey(final String accesskey)
    {
        this.accesskey = accesskey;
    }

    public void setMandatory(final String mandatory)
    {
        this.mandatory = mandatory;
    }

    public void addParameter(String name, Object value)
    {
        addParameterInternal(name, value);
    }

    private void addParameterInternal(String name, Object value)
    {
        params.put(name, value);
    }

    public Map getParameters()
    {
        return params;
    }

    public boolean memberOf(Object obj1, Object obj2)
    {
        return ContainUtil.contains(obj1, obj2);
    }

    // IncludeTag overrides ------------------------------------------
    public int doEndTag()
            throws JspException
    {
        if (themeAttr != null)
        {
            theme = (String) findValue(themeAttr);
        }

        // Get name of JSP fragment
        String template = templateAttr;
        if (template == null)
        {
            template = getTemplate() + "." + getTemplateSuffix();
        }

        // Put params on the stack
        addParameterInternal("label", findValue(labelAttr));
        Object value = findValue(nameAttr);
        if (value != null)
        {
            addParameterInternal("name", value);
        }

        // If value attribute is not given, use the name
        if (valueAttr == null)
        {
            addParameterInternal("nameValue", findValue(BeanUtil.toStringValue(value)));
        }
        else
        {
            if (log.isDebugEnabled())
            {
                log.debug("valueAttr:" + valueAttr);
                log.debug("Setting value to:" + findValue(valueAttr));
            }
            addParameterInternal("nameValue", findValue(valueAttr));
        }

        if (labelPositionAttr == null)
        {
            labelPositionAttr = "'center'";
        }
        addParameterInternal("labelposition", findValue(labelPositionAttr));

        if (getId() != null)
        {
            addParameterInternal("id", removeSingleQuotes(getId()));
        }
        if (tabindex != null)
        {
            addParameterInternal("tabindex", findValue(tabindex));
        }
        if (title != null)
        {
            addParameterInternal("title", findValue(title));
        }
        if (accesskey != null)
        {
            addParameterInternal("accesskey", findValue(accesskey));
        }
        if (mandatory != null)
        {
            addParameterInternal("mandatory", findValue(mandatory));
        }

        getStack().pushValue(this);

        try
        {
            String templatePath = "/" + getTemplateDir() + "/" + getTheme() + "/" + template;
            if (log.isDebugEnabled())
            {
                log.debug("Using template:" + templatePath);
            }
            if (templatePath.endsWith(".vm"))
            {
                Context c = VelocityHelper.getContext(pageContext.getServletContext(), pageContext.getRequest(), pageContext.getResponse());
                VelocityHelper.merge(c, templatePath, pageContext.getOut());
            }
            else
            {
                IncludeTag.include(templatePath, pageContext);
            }
            return EVAL_PAGE;
        }
        catch (Exception e)
        {
            throw new JspTagException("Including component failed:" + toString(e));
        }
        finally
        {
            getStack().popValue();
            params = new HashMap();
        }
    }

    /**
     * Removes the prefix and suffix single quotes ' from the value.  This is so that non evaluated attributes like id
     * can be specified as "'myid'" and still have the value "myid"
     *
     * @param s the string in play
     *
     * @return s without quotes
     */
    private String removeSingleQuotes(final String s)
    {
        return QuoteRemover.removeSingleQuotes(s);
    }
}
