/*
 * WebWork, Web Application Framework
 *
 * Distributable under Apache license.
 * See terms of license at opensource.org
 */
package webwork.dispatcher;

import java.beans.Introspector;
import java.io.*;

import javax.servlet.RequestDispatcher;
import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import webwork.action.Action;
import webwork.action.ActionContext;
import webwork.action.ResultException;
import webwork.action.ServletActionContext;
import webwork.config.Configuration;
import webwork.multipart.MultiPartRequest;
import webwork.multipart.MultiPartRequestWrapper;
import webwork.util.ServletValueStack;
import webwork.util.ValueStack;

/**
 * Main dispatcher servlet. It works in three phases: first propagate all parameters to the command JavaBean. Second,
 * call execute() to let the JavaBean create the result data. Third, delegate to the JSP that corresponds to the result
 * state that was chosen by the JavaBean.
 * <p/>
 * The command JavaBeans can be found in a package prefixed with either of the package names in the comma-separated
 * "packages" servlet init parameter.
 * <p/>
 * Modified by Raymond Lai (alpha2_valen@yahoo.com) on 1 Nov 2003: modified wrapRequest() to set the character encoding
 * of HttpServletRequest using the parameter "webwork.i18n.encoding" in webwork.properties.
 *
 * @author Rickard \u00D6berg (rickard@middleware-company.com)
 * @author Matt Baldree (matt@smallleap.com)
 * @version $Revision: 1.73 $
 */
public class ServletDispatcher extends HttpServlet
{
    // Attributes ----------------------------------------------------

    /**
     * After a view is processed the value of the stack's head is put into the request attributes with this key.
     */
    public static final String STACK_HEAD = "webwork.valuestack.head";

    protected static Log log = LogFactory.getLog(ServletDispatcher.class);

    private static String saveDir;
    private static Integer maxSize;
    private static String actionExtension;
    private String encoding;
    public static final String GENERIC_DISPATCHER_FOR_CLEANUP = "webwork.generic.dispatcher";

    // HttpServlet overrides -----------------------------------------
    /**
     * Initialize dispatcher servlet
     *
     * @param config
     *
     * @throws ServletException
     */
    public void init(ServletConfig config)
            throws ServletException
    {
        super.init(config);

        // Clear caches
        // RO: If not, then it will contain garbage after a couple of redeployments
        Introspector.flushCaches();

        // Clear ValueStack method cache
        // RO: If not, then it will contain garbage after a couple of redeployments
        ValueStack.clearMethods();

        //load multipart configuration
        //saveDir
        try
        {
            saveDir = Configuration.getString("webwork.multipart.saveDir");
            if (saveDir.compareTo("") == 0)
            {
                throw new IllegalArgumentException("saveDir=null");
            }
        }
        catch (IllegalArgumentException e)
        {
            //use default
            File tempdir = (File) config.getServletContext().getAttribute("javax.servlet.context.tempdir");
            log.info("Unable to find 'webwork.multipart.saveDir' property setting. Defaulting to javax.servlet.context.tempdir");
            if (tempdir != null)
            {
                saveDir = tempdir.toString();
            }
        }
        log.debug("saveDir=" + saveDir);

        //maxSize
        try
        {
            String maxSizeStr = Configuration.getString("webwork.multipart.maxSize");
            if (maxSizeStr != null)
            {
                try
                {
                    maxSize = new Integer(maxSizeStr);
                }
                catch (NumberFormatException e)
                {
                    maxSize = new Integer(Integer.MAX_VALUE);
                    log.info("Unable to format 'webwork.multipart.maxSize' property setting. Defaulting to Integer.MAX_VALUE");
                }
            }
            else
            {
                maxSize = new Integer(Integer.MAX_VALUE);
                log.info("Unable to format 'webwork.multipart.maxSize' property setting. Defaulting to Integer.MAX_VALUE");
            }
        }
        catch (IllegalArgumentException e1)
        {
            maxSize = new Integer(Integer.MAX_VALUE);
            log.info("Unable to format 'webwork.multipart.maxSize' property setting. Defaulting to Integer.MAX_VALUE");
        }
        log.debug("maxSize=" + maxSize);
        try
        {
            actionExtension = "." + Configuration.getString("webwork.action.extension");
        }
        catch (IllegalArgumentException iae)
        {
            actionExtension = ".action";
            log.info("Unable to find \'webwork.action.extension\' property setting. Defaulting to \'action\'");
        }

        try
        {
            encoding = Configuration.getString("webwork.i18n.encoding");
        }
        catch (IllegalArgumentException iae)
        {
        }

        log.debug("action extension=" + actionExtension);
        log.info("Action dispatcher initialized");
    }

    /**
     * Service a request. The request is first checked to see if it is a multi-part. If it is, then the request is
     * wrapped so WW will be able to work with the multi-part as if it was a normal request. Next, we will process all
     * actions until an action returns a non-action which is usually a view. For each action in a chain, the action's
     * context will be first set and then the action will be instantiated. Next, the previous action if this action
     * isn't the first in the chain will have its attributes copied to the current action.
     *
     * @param aRequest
     * @param aResponse
     *
     * @throws ServletException
     */
    public void service(HttpServletRequest aRequest, HttpServletResponse aResponse)
            throws ServletException
    {
        //wrap request if needed
        aRequest = wrapRequest(aRequest);

        // Get action
        String servletPath = (String) aRequest.getAttribute("javax.servlet.include.servlet_path");
        if (servletPath == null)
        {
            servletPath = aRequest.getServletPath();
        }

        String actionName = getActionName(servletPath);
        GenericDispatcher gd = new GenericDispatcher(actionName, false);
        ActionContext context = gd.prepareContext();
        ServletActionContext.setContext(aRequest, aResponse, getServletContext(), actionName);
        gd.prepareValueStack();
        ActionResult ar = null;
        try
        {
            gd.executeAction();
            ar = gd.finish();
        }
        catch (Throwable e)
        {
            log.error("Could not execute action", e);
            try
            {
                aResponse.sendError(404, "Could not execute action [" + actionName + "]:" + e.getMessage() + getHTMLErrorMessage(e));
            }
            catch (IOException e1)
            {
            }
        }

        if (ar != null && ar.getActionException() != null)
        {
            log.error("Could not execute action", ar.getActionException());
            try
            {
                aResponse.sendError(500, ar.getActionException().getMessage() + getHTMLErrorMessage(ar.getActionException()));
            }
            catch (IOException e1)
            {
            }
        }

        // check if no view exists
        if (ar != null && ar.getResult() != null && ar.getView() == null && !ar.getResult().equals(Action.NONE))
        {
            try
            {
                aResponse.sendError(404, "No view for result [" + ar.getResult() + "] exists for action [" + actionName + "]");
            }
            catch (IOException e)
            {
            }
        }

        if (ar != null && ar.getView() != null && ar.getActionException() == null)
        {
            String view = ar.getView().toString();
            log.debug("Result:" + view);

            RequestDispatcher dispatcher = null;
            try
            {
                dispatcher = aRequest.getRequestDispatcher(view);
            }
            catch (Throwable e)
            {
                // Ignore
            }

            if (dispatcher == null)
            {
                throw new ServletException("No presentation file with name '" + view + "' found!");
            }

            try
            {
                // If we're included, then include the view
                // Otherwise do forward
                // This allow the page to, for example, set content type
                if (aRequest.getAttribute("javax.servlet.include.servlet_path") == null)
                {
                    aRequest.setAttribute("webwork.view_uri", view);
                    aRequest.setAttribute("webwork.request_uri", aRequest.getRequestURI());
                    //               aRequest.setAttribute("webwork.contextPath",aRequest.getContextPath());
                    dispatcher.forward(aRequest, aResponse);
                }
                else
                {
                    //               aRequest.setAttribute("webwork.request_uri",aRequest.getAttribute("javax.servlet.include.request_uri"));
                    //               aRequest.setAttribute("webwork.contextPath",aRequest.getAttribute("javax.servlet.include.context_path"));
                    dispatcher.include(aRequest, aResponse);
                }
            }
            catch (IOException e)
            {
                throw new ServletException(e);
            }
            finally
            {
                // Get last action from stack and and store it in request attribute STACK_HEAD
                // It is then popped from the stack.
                aRequest.setAttribute(STACK_HEAD, ServletValueStack.getStack(aRequest).peek());
            }
        }

        Boolean skipCleaning = (Boolean) aRequest.getAttribute(CleanupFilter.DO_NOT_CLEAN);
        if ((skipCleaning != null) && (skipCleaning.booleanValue()))
        {
            aRequest.setAttribute(GENERIC_DISPATCHER_FOR_CLEANUP, gd);
        }
        else
        {
            gd.finalizeContext();
        }
    }

    /**
     * Determine action name by extracting last string and removing extension. (/.../.../Foo.action -> Foo)
     */
    protected String getActionName(String name)
    {
        // Get action name ("Foo.action" -> "Foo" action)
        int beginIdx = name.lastIndexOf("/");
        int endIdx = name.lastIndexOf(actionExtension);
        return name.substring((beginIdx == -1 ? 0 : beginIdx + 1),
                endIdx == -1 ? name.length() : endIdx);
    }

    /**
     * Wrap servlet request with the appropriate request. It will check to see if request is a multipart request and
     * wrap in appropriately.
     *
     * @param request
     *
     * @return wrapped request or original request
     */
    private HttpServletRequest wrapRequest(HttpServletRequest request)
    {
        if (encoding != null)
        {
            try
            {
                request.setCharacterEncoding(encoding);
            }
            catch (Exception e) {}
        }
        // don't wrap more than once
        if (request instanceof MultiPartRequestWrapper)
        {
            return request;
        }

        if (MultiPartRequest.isMultiPart(request))
        {
            try
            {
                request = new MultiPartRequestWrapper(request, saveDir, maxSize);
            }
            catch (IOException e)
            {
                request.setAttribute("webwork.action.ResultException",
                        new ResultException(Action.ERROR, e.getLocalizedMessage()));
            }
        }
        return request;
    }

    /**
     * Return an HTML error message for the throwable
     *
     * @param t Throwable
     *
     * @return A String with an error message
     */
    protected String getHTMLErrorMessage(Throwable t)
    {
        return "<p><small><small><pre>" + getStackTrace(t) + "</pre></small></small></p>";
    }

    /**
     * Get the stack trace of the throwable and return as a String
     *
     * @param t Throwable
     *
     * @return Stack trace of the Throwable
     */
    private static String getStackTrace(Throwable t)
    {
        CharArrayWriter sink = new CharArrayWriter(400);
        PrintWriter exceptionWriter = new PrintWriter(sink);
        t.printStackTrace(exceptionWriter);
        return sink.toString();
    }
}
