/*
 * WebWork, Web Application Framework
 *
 * Distributable under Apache license.
 * See terms of license at opensource.org
 */
package webwork.dispatcher;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import webwork.action.Action;
import webwork.action.ActionContext;
import webwork.action.CleanupAware;
import webwork.action.factory.ActionFactory;
import webwork.config.Configuration;
import webwork.interceptor.ChainedInterceptorFactory;
import webwork.interceptor.DefaultInterceptorChain;
import webwork.interceptor.Interceptor;
import webwork.interceptor.InterceptorChain;
import webwork.util.BeanUtil;
import webwork.util.ValueStack;
import webwork.util.injection.ObjectFactory;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

public class GenericDispatcher
{
    protected static Log log = LogFactory.getLog(GenericDispatcher.class);
    private static ViewMapping mapping;

    static
    {
        // Choose classloader
        final ClassLoader classLoader = Thread.currentThread().getContextClassLoader();

        // Initialize view mapping
        String mappingName;
        try
        {
            mappingName = Configuration.getString("webwork.viewmapping");
        }
        catch (final IllegalArgumentException e)
        {
            // Default
            mappingName = DefaultViewMapping.class.getName();
        }

        try
        {
            mapping = (ViewMapping) ObjectFactory.instantiateBean(classLoader, mappingName);
        }
        catch (final Throwable t)
        {
            log.error("GenericDispatcher could not create ViewMapping object of class: " + mappingName, t);
        }
    }

    private boolean prepared = false;
    private Map oldContextTable;
    private ActionContext context;
    private final List<Action> actions = new ArrayList<Action>();
    private String result;
    private String actionName;
    private LazyValueHolder lazyValueHolder;
    private boolean lazy = false;
    private Object view;
    private Exception actionException;
    private int initialStackSize;

    public GenericDispatcher(final String actionName)
    {
        this.actionName = actionName;
    }

    public GenericDispatcher(final String actionName, final boolean lazy)
    {
        this.actionName = actionName;
        this.lazy = lazy;
    }

    /**
     * Prepare a new ActionContext and return it prepareValueStack should be called after this when the new Valuestack
     * has been set in the action context
     */
    public ActionContext prepareContext()
    {
        // Get oldContext and save
        if (ActionContext.getContext() != null)
        {
            oldContextTable = ActionContext.getContext().getTable();
        }
        // Create new context
        context = new ActionContext();
        ActionContext.setContext(context);
        context.put("action.chain", actions);
        return context;
    }

    /**
     * Prepare the dispatcher by saving the initial size of the current value stack
     */
    public void prepareValueStack()
    {
        initialStackSize = ActionContext.getValueStack().size();
        prepared = true;
    }

    public ActionContext getOldContext()
    {
        final ActionContext oldContext = new ActionContext();
        oldContext.setTable(oldContextTable);
        return oldContext;
    }

    public List<Action> getActions()
    {
        return actions;
    }

    public void executeAction() throws Exception
    {
        if (!prepared)
        {
            throw new Exception("You must prepare the dispatcher first");
        }

        Action action = ActionFactory.getAction(actionName);

        if (lazy)
        {
            lazyValueHolder = new LazyValueHolder(action);
            ActionContext.getValueStack().pushValue(lazyValueHolder);
            // lazy evaluation can't chain
            return;
        }
        else
        {
            try
            {
                result = executeAction(action);
            }
            catch (final Exception e)
            {
                actionException = e;
                return;
            }

            ActionContext.getValueStack().pushValue(action);
        }

        if (result != null)
        {
            try
            {
                view = mapping.getView(actionName, result);
            }
            catch (final Exception e)
            {
                view = null;
            }
        }

        while (view instanceof ViewActionWrapper)
        {
            final ViewActionWrapper viewActionWrapper = (ViewActionWrapper) view;
            actionName = viewActionWrapper.getActionName();
            action = ActionFactory.getAction(actionName);
            if (viewActionWrapper.hasParams())
            {
                BeanUtil.setProperties(viewActionWrapper.getParams(), action);
            }

            try
            {
                result = executeAction(action);
            }
            catch (final Exception e)
            {
                actionException = e;
                return;
            }

            ActionContext.getValueStack().pushValue(action);
            if (result != null)
            {
                try
                {
                    view = mapping.getView(actionName, result);
                }
                catch (final Exception e)
                {
                    view = null;
                }
            }
        }
    }

    private String executeAction(Action action) throws Exception
    {
        Interceptor interceptor = ChainedInterceptorFactory.getInstance().createInterceptorFor(action);
        InterceptorChain chain = new DefaultInterceptorChain(action, interceptor);
        String result = chain.proceed();
        return result;
    }

    public ActionResult finish()
    {
        if (lazy)
        {
            lazyValueHolder.getValue();
            if (result != null)
            {
                try
                {
                    view = mapping.getView(actionName, result);
                }
                catch (final Exception e)
                {
                    view = null;
                }
            }
        }

        return new ActionResult(result, view, actions, actionException);
    }

    public void finalizeContext()
    {
        // Take all the actions off the stack of the old context
        final ValueStack vs = ActionContext.getValueStack();
        final int finalStackSize = vs.size();
        final int count = finalStackSize - initialStackSize;

        for (int i = 0; i < count; i++)
        {
            Object o = vs.popValue();
            if (o instanceof ValueStack.ValueHolder)
            {
                o = ((ValueStack.ValueHolder) o).getValue();
            }
            if (o instanceof CleanupAware)
            {
                ((CleanupAware) o).cleanup();
            }
        }

        // Create a new context
        final ActionContext newContext = new ActionContext();
        // Set old values in it
        newContext.setTable(oldContextTable);
        oldContextTable = null;
        ActionContext.setContext(newContext);
    }

    public class LazyValueHolder implements ValueStack.ValueHolder
    {
        private final Action action;
        private boolean hasExecuted = false;

        public LazyValueHolder(final Action action)
        {
            this.action = action;
        }

        public Object getValue()
        {
            if (!hasExecuted)
            {
                hasExecuted = true;
                try
                {
                    result = executeAction(action);
                }
                catch (final Exception e)
                {
                    actionException = e;
                    result = null;
                    return null;
                }
            }
            return action;
        }
    }

}
