package nz.co.nativemobile.cameramodule.helper

import android.app.Activity
import android.app.Activity.RESULT_OK
import android.app.AlertDialog
import android.content.ContentResolver
import android.content.Context
import android.content.Intent
import android.content.res.Resources
import android.net.Uri
import android.provider.MediaStore
import androidx.core.app.ActivityCompat
import androidx.core.content.FileProvider
import com.theartofdev.edmodo.cropper.CropImage
import com.theartofdev.edmodo.cropper.CropImageView
import nz.co.nativemobile.cameramodule.R
import nz.co.nativemobile.cameramodule.extensions.createImageFile
import nz.co.nativemobile.cameramodule.extensions.createVideoFile
import java.io.BufferedInputStream
import java.io.File


/**
 * Created by wadereweti on 13/10/17.
 */

enum class CameraViewShape {
    OVAL, RECTANGLE
}

typealias CancelActionClosure = () -> Unit


/**
 * Users of this module must create the following dependencies
 *
 *
 * implementation 'com.theartofdev.edmodo:android-image-cropper:2.5.1'
 *
 * the following file_paths.xml file under xml directory in resources
 * <paths xmlns:android="http://schemas.android.com/apk/res/android">
 * <external-path name="images" path="Android/data/com.nativemobile.teamview/files/Pictures" />
 * </paths>
 *
 * The following provider must be declared in the manifest
 *
 * <provider
 *      android:name="android.support.v4.content.FileProvider"
 *      android:authorities="${applicationId}.fileprovider"
 *      android:exported="false"
 *      android:grantUriPermissions="true">

 *  <meta-data
 *      android:name="android.support.FILE_PROVIDER_PATHS"
 *      android:resource="@xml/file_paths" />
 * </provider>
 */
interface CameraView {
    /**
     * Authority is made up of the following string "${BuildConfig.APPLICATION_ID}${BuildConfig.FILE_PROVIDER}"
     */
    fun configureWithCameraContext(context: Context, resources: Resources, viewModel: CameraViewModel, authority: String)

    fun displayImagePicker(imageType: CameraViewDelegate.ImageType = CameraViewDelegate.ImageType.PRIMARY, cropShape: CameraViewShape = CameraViewShape.OVAL, negativeActionClosure: CancelActionClosure = {})
    fun displayVideoPicker()
    fun handleImageRequestResult(requestCode: Int, resultCode: Int, data: Intent?)
    fun currentImages(): List<Uri>
}

interface CameraViewModel {
    interface ImageViewModel : CameraViewModel {
        fun onImageCaptured(selectedMedia: List<BitmapMedia>, imageType: CameraViewDelegate.ImageType)
    }

    interface VideoViewModel : CameraViewModel {
        fun onVideoCaptured(selectedMedia: Pair<List<BitmapMedia>, BitmapMedia>)
    }

    interface ImageAndVideoViewModel : CameraViewModel, VideoViewModel, ImageViewModel
}

class CameraViewDelegate : CameraView {
    enum class Media(val type: String) {

        PHOTO(MediaStore.ACTION_IMAGE_CAPTURE),

        VIDEO(MediaStore.ACTION_VIDEO_CAPTURE)
    }

    enum class Gallery(val uri: Uri, val type: String) {
        PHOTO(MediaStore.Images.Media.EXTERNAL_CONTENT_URI, IMAGE_TYPE),

        VIDEO(MediaStore.Video.Media.EXTERNAL_CONTENT_URI, VIDEO_TYPE)
    }

    enum class ImageType {
        PRIMARY, SECONDARY
    }

    private var images: MutableList<Uri> = mutableListOf()
    private var mediaUri: Uri? = null
    private var mediaFile: File? = null
    private var videoMedia: BitmapMedia? = null

    private lateinit var context: Context
    private lateinit var resources: Resources
    private lateinit var contentResolver: ContentResolver
    private lateinit var viewModel: CameraViewModel
    private lateinit var imageType: ImageType
    private lateinit var cropShape: CropImageView.CropShape
    private lateinit var authority: String

    override fun currentImages(): List<Uri> = images

    override fun configureWithCameraContext(context: Context, resources: Resources, viewModel: CameraViewModel, authority: String) {
        this.context = context
        this.resources = resources
        this.contentResolver = context.contentResolver
        this.viewModel = viewModel
        this.authority = authority
    }

    override fun displayImagePicker(imageType: ImageType, cropShape: CameraViewShape, negativeActionClosure: CancelActionClosure) {
        this.imageType = imageType
        this.cropShape = when (cropShape) {
            CameraViewShape.OVAL -> CropImageView.CropShape.OVAL
            CameraViewShape.RECTANGLE -> CropImageView.CropShape.RECTANGLE
        }

        val items = resources.getStringArray(R.array.photo_dialog_options)

        val builder = AlertDialog.Builder(context)
        builder.setTitle(resources.getString(R.string.title_picker))

        builder.setItems(items) { dialog, item ->
            when (item) {
                0 -> dispatchMediaCaptureIntent(Media.PHOTO)
                1 -> dispatchGalleryIntent(Gallery.PHOTO)
                2 -> dialog.dismiss()
            }
        }

        builder.setOnCancelListener { negativeActionClosure.invoke() }

        builder.show()
    }

    override fun displayVideoPicker() {
        val items = resources.getStringArray(R.array.video_dialog_options)

        val builder = AlertDialog.Builder(context)
        builder.setTitle(resources.getString(R.string.title_picker))

        builder.setItems(items) { dialog, item ->
            when (item) {
                0 -> dispatchMediaCaptureIntent(Media.VIDEO)
                1 -> dispatchGalleryIntent(Gallery.VIDEO)
                2 -> dialog.dismiss()
            }
        }

        builder.show()
    }

    override fun handleImageRequestResult(requestCode: Int, resultCode: Int, data: Intent?) {
        if (resultCode != RESULT_OK) {
            return
        }

        if (requestCode == CropImage.CROP_IMAGE_ACTIVITY_REQUEST_CODE) {
            val result = CropImage.getActivityResult(data)
            if (resultCode == RESULT_OK) {
                images.clear()
                images.add(result.uri)
                (viewModel as CameraViewModel.ImageViewModel).onImageCaptured(buildMedia(), imageType)
            } else if (resultCode == CropImage.CROP_IMAGE_ACTIVITY_RESULT_ERROR_CODE) {
                // Do something here with the error
                // val error = result.error
            }
        } else {
            data?.let {
                when (requestCode) {
                    REQUEST_VIDEO_GALLERY, REQUEST_CAMERA_VIDEO -> buildVideoMedia(it, requestCode)
                    REQUEST_IMAGE_GALLERY, REQUEST_CAMERA_IMAGE -> buildImageMedia(it, requestCode)
                }
            }
        }
    }

    private fun buildImageMedia(data: Intent, requestCode: Int) {
        val imageUri: Uri = when (requestCode) {
            REQUEST_IMAGE_GALLERY -> data.data!!
            REQUEST_CAMERA_IMAGE -> mediaUri!!
            else -> data.data!!
        }

        CropImage.activity(imageUri)
                .setCropShape(cropShape)
                .start(context as Activity)
    }

    private fun buildVideoMedia(intent: Intent, requestCode: Int) {
        val videoUri: Uri = when (requestCode) {
            REQUEST_VIDEO_GALLERY -> intent.data!!
            REQUEST_CAMERA_VIDEO -> mediaUri!!
            else -> intent.data!!
        }

        val media = buildMedia()
        videoMedia = BitmapMedia(uri = videoUri)
        (viewModel as CameraViewModel.VideoViewModel).onVideoCaptured(Pair(media, videoMedia!!))
    }

    private fun buildMedia(): List<BitmapMedia> {
        /*var bitmapMedia: List<BitmapMedia> = listOf()
        launchAsync {
            asyncAwait {
                bitmapMedia = imageUris.map {
                    val inputStream = contentResolver.openInputStream(it)
                    val bufferedInputStream = BufferedInputStream(contentResolver.openInputStream(it))
                    val dimension = 150
                    val bitmap = BitmapHelper.decodeSampledBitmapFromInputStream(inputStream, bufferedInputStream, dimension, dimension)
                    inputStream.close()
                    bufferedInputStream.close()

                    BitmapMedia(it, bitmap)
                }
            }
        }

        return bitmapMedia*/


        return images.map {
            val inputStream = context.contentResolver.openInputStream(it)
            val bufferedInputStream = BufferedInputStream(context.contentResolver.openInputStream(it)!!)
            val dimension = 150
            val bitmap = BitmapHelper.decodeSampledBitmapFromInputStream(inputStream!!, bufferedInputStream, dimension, dimension)
            inputStream.close()
            bufferedInputStream.close()

            BitmapMedia(it, bitmap)
        }
    }

    private fun dispatchMediaCaptureIntent(media: Media) {
        val intent = Intent(media.type)

        if (intent.resolveActivity(context.packageManager) != null) {
            mediaFile = when (media) {
                Media.PHOTO -> context.createImageFile()
                Media.VIDEO -> context.createVideoFile()
            }

            val requestType = when (media) {
                Media.PHOTO -> REQUEST_CAMERA_IMAGE
                Media.VIDEO -> REQUEST_CAMERA_VIDEO
            }

            mediaUri = FileProvider.getUriForFile(context, authority, mediaFile!!)
            intent.putExtra(MediaStore.EXTRA_OUTPUT, mediaUri)

            ActivityCompat.startActivityForResult(context as Activity, intent, requestType, null)
        }
    }

    private fun dispatchGalleryIntent(gallery: Gallery) {
        val intent = Intent(Intent.ACTION_PICK, gallery.uri)
        intent.type = gallery.type

        val requestType = when (gallery) {
            Gallery.PHOTO -> REQUEST_IMAGE_GALLERY
            Gallery.VIDEO -> REQUEST_VIDEO_GALLERY
        }

        val chooserIntent = Intent.createChooser(intent, resources.getString(R.string.title_select_file))
        ActivityCompat.startActivityForResult(context as Activity, chooserIntent, requestType, null)
    }

    companion object {
        const val IMAGE_TYPE = "image/*"
        const val VIDEO_TYPE = "video/*"

        const val REQUEST_CAMERA_IMAGE = 10
        const val REQUEST_CAMERA_VIDEO = 20

        const val REQUEST_IMAGE_GALLERY = 30
        const val REQUEST_VIDEO_GALLERY = 40
    }
}