/*
	* Copyright (C) 2020-2021 Sebastiano Vigna
	*
	* Licensed under the Apache License, Version 2.0 (the "License");
	* you may not use this file except in compliance with the License.
	* You may obtain a copy of the License at
	*
	*     http://www.apache.org/licenses/LICENSE-2.0
	*
	* Unless required by applicable law or agreed to in writing, software
	* distributed under the License is distributed on an "AS IS" BASIS,
	* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
	* See the License for the specific language governing permissions and
	* limitations under the License.
	*/
package it.unimi.dsi.fastutil.objects;
/**
 * A type-specific {@link it.unimi.dsi.fastutil.Pair Pair}; provides some
 * additional methods that use polymorphism to avoid (un)boxing.
 */
public interface ObjectBooleanPair<K> extends it.unimi.dsi.fastutil.Pair<K, Boolean> {
	/**
	 * Returns the right element of this pair.
	 *
	 * @return the right element of this pair.
	 */
	public boolean rightBoolean();
	/**
	 * {@inheritDoc}
	 * 
	 * @deprecated Please use the corresponding type-specific method instead.
	 */
	@SuppressWarnings("deprecation")
	@Deprecated
	@Override
	public default Boolean right() {
		return Boolean.valueOf(rightBoolean());
	}
	/**
	 * Sets the right element of this pair (optional operation).
	 *
	 * @param r
	 *            a new value for the right element.
	 *
	 * @implSpec This implementation throws an
	 *           {@link UnsupportedOperationException}.
	 */
	public default ObjectBooleanPair<K> right(final boolean r) {
		throw new UnsupportedOperationException();
	}
	/**
	 * {@inheritDoc}
	 * 
	 * @deprecated Please use the corresponding type-specific method instead.
	 */
	@SuppressWarnings("deprecation")
	@Deprecated
	@Override
	public default ObjectBooleanPair<K> right(final Boolean l) {
		return right((l).booleanValue());
	}
	/**
	 * Returns the right element of this pair.
	 *
	 * @return the right element of this pair.
	 *
	 * @implSpec This implementation delegates to {@link #right()}.
	 *
	 */
	public default boolean secondBoolean() {
		return rightBoolean();
	}
	/**
	 * {@inheritDoc}
	 * 
	 * @deprecated Please use the corresponding type-specific method instead.
	 */
	@SuppressWarnings("deprecation")
	@Deprecated
	@Override
	public default Boolean second() {
		return Boolean.valueOf(secondBoolean());
	}
	/**
	 * Sets the right element of this pair (optional operation).
	 *
	 * @param r
	 *            a new value for the right element.
	 *
	 * @implSpec This implementation delegates to {@link #right(Object)}.
	 */
	public default ObjectBooleanPair<K> second(final boolean r) {
		return right(r);
	}
	/**
	 * {@inheritDoc}
	 * 
	 * @deprecated Please use the corresponding type-specific method instead.
	 */
	@SuppressWarnings("deprecation")
	@Deprecated
	@Override
	public default ObjectBooleanPair<K> second(final Boolean l) {
		return second((l).booleanValue());
	}
	/**
	 * Returns the right element of this pair.
	 *
	 * @return the right element of this pair.
	 *
	 * @implSpec This implementation delegates to {@link #right()}.
	 *
	 */
	public default boolean valueBoolean() {
		return rightBoolean();
	}
	/**
	 * {@inheritDoc}
	 * 
	 * @deprecated Please use the corresponding type-specific method instead.
	 */
	@SuppressWarnings("deprecation")
	@Deprecated
	@Override
	public default Boolean value() {
		return Boolean.valueOf(valueBoolean());
	}
	/**
	 * Sets the right element of this pair (optional operation).
	 *
	 * @param r
	 *            a new value for the right element.
	 *
	 * @implSpec This implementation delegates to {@link #right(Object)}.
	 */
	public default ObjectBooleanPair<K> value(final boolean r) {
		return right(r);
	}
	/**
	 * {@inheritDoc}
	 * 
	 * @deprecated Please use the corresponding type-specific method instead.
	 */
	@SuppressWarnings("deprecation")
	@Deprecated
	@Override
	public default ObjectBooleanPair<K> value(final Boolean l) {
		return value((l).booleanValue());
	}
	/**
	 * Returns a new type-specific immutable {@link it.unimi.dsi.fastutil.Pair Pair}
	 * with given left and right value.
	 * 
	 * @param left
	 *            the left value.
	 * @param right
	 *            the right value.
	 */
	public static <K> ObjectBooleanPair<K> of(final K left, final boolean right) {
		return new ObjectBooleanImmutablePair<K>(left, right);
	}
	/**
	 * Returns a lexicographical comparator for pairs.
	 *
	 * <p>
	 * The comparator returned by this method implements lexicographical order. It
	 * compares first the left elements: if the result of the comparison is nonzero,
	 * it returns said result. Otherwise, this comparator returns the result of the
	 * comparison of the right elements.
	 *
	 * @return a lexicographical comparator for pairs.
	 */
	@SuppressWarnings("unchecked")
	public static <K> java.util.Comparator<ObjectBooleanPair<K>> lexComparator() {
		return (x, y) -> {
			final int t = ((Comparable<K>) x.left()).compareTo(y.left());
			if (t != 0)
				return t;
			return Boolean.compare(x.rightBoolean(), y.rightBoolean());
		};
	}
}
