package ir.map.sdk_services.network;

import android.content.Context;
import android.graphics.Bitmap;
import android.view.View;

import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;
import com.nostra13.universalimageloader.core.ImageLoader;
import com.nostra13.universalimageloader.core.assist.FailReason;
import com.nostra13.universalimageloader.core.listener.ImageLoadingListener;

import org.json.JSONArray;
import org.json.JSONException;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import ir.map.sdk_common.MapirLatLng;
import ir.map.sdk_services.MapUtils;
import ir.map.sdk_services.PublicTransportMode;
import ir.map.sdk_services.R;
import ir.map.sdk_services.RouteMode;
import ir.map.sdk_services.ServiceSDK;
import ir.map.sdk_services.URLHelper;
import ir.map.sdk_services.models.MaptexAuthenticateResponse;
import ir.map.sdk_services.models.MaptexBookmark;
import ir.map.sdk_services.models.MaptexForgotEntity;
import ir.map.sdk_services.models.MaptexLoginResponse;
import ir.map.sdk_services.models.MaptexMedia;
import ir.map.sdk_services.models.MaptexNetworkEntity;
import ir.map.sdk_services.models.MaptexProfileResponse;
import ir.map.sdk_services.models.MaptexPublicTransportResponse;
import ir.map.sdk_services.models.MaptexResendVerification;
import ir.map.sdk_services.models.MaptexReverse;
import ir.map.sdk_services.models.MaptexRouteResponse;
import ir.map.sdk_services.models.MaptexSearchPostCode;
import ir.map.sdk_services.models.MaptexSearchResponse;
import ir.map.sdk_services.models.MaptexSimpleNetworkEntity;
import ir.map.sdk_services.models.MaptexTag;
import ir.map.sdk_services.models.MaptexTicket;
import ir.map.sdk_services.models.MaptexWeatherItem;
import okhttp3.RequestBody;
import rx.Observable;
import rx.Subscriber;

import static ir.map.sdk_services.URLHelper.FORGOT_PASSWORD_URL;
import static ir.map.sdk_services.URLHelper.GET_PROFILE;
import static ir.map.sdk_services.URLHelper.LOCAL_URL_GMAIL_LOGIN;
import static ir.map.sdk_services.URLHelper.SIGN_IN_OUT_URL;
import static ir.map.sdk_services.URLHelper.SIGN_UP_URL;
import static ir.map.sdk_services.URLHelper.STATIC_MAP;
import static ir.map.sdk_services.URLHelper.UPDATE_PROFILE;
import static ir.map.sdk_services.URLHelper.VERIFICATION_CODE_URL;

/**
 * @author Haniyeh Khaksar
 */

public class RestApiImpl implements RestApi {

    //    private PreferenceService preferenceService;
//    private  ;
    private Gson gsonMapper;

    public RestApiImpl() {
        gsonMapper = new Gson();
    }

    @Override
    public Observable<MaptexReverse> getReverseGeoInfo(final String params) {

        return Observable.create(new Observable.OnSubscribe<MaptexReverse>() {
            @Override
            public void call(Subscriber<? super MaptexReverse> subscriber) {
                try {

                    String responseString = getReverseGeoInfoApi(params);

                    if (params == null || "" == params) {
                        subscriber.onError(new Throwable("Post Method Need Param"));
                    } else if (responseString == null) {
                        subscriber.onError(new Throwable("Error From Server"));
                    } else {
                        MaptexReverse reverseGeoInfo = gsonMapper.fromJson(responseString, MaptexReverse.class);
                        subscriber.onNext(reverseGeoInfo);
                        subscriber.onCompleted();
                    }

                } catch (Exception e) {
                    subscriber.onError(e);

                }
            }
        });

    }

    private String getReverseGeoInfoApi(String params) throws Exception {
        return ApiConnection.createGET(new String(URLHelper.MAP_REVERSE_GEO_URL) + params).call();
    }

//    @Override
//    public Observable<List<MaptexSearch>> getPointOfInterests() {
//        return Observable.create(new Observable.OnSubscribe<List<MaptexSearch>>() {
//            @Override
//            public void call(Subscriber<? super List<MaptexSearch>> subscriber) {
//                try {
//                    InputStream is = context.getAssets().open("poi.json");
//                    int size = is.available();
//                    byte[] buffer = new byte[size];
//                    is.read(buffer);
//                    is.close();
//                    String responseString = new String(buffer);
//
//                    if (responseString == null) {
//                        subscriber.onError(new Throwable("Error From Server"));
//                    } else {
//
//
//                        maptexSearchList odataModel = gsonMapper.fromJson(responseString, maptexSearchList.class);
//
//                        subscriber.onNext(odataModel.getValues());
//                        subscriber.onCompleted();
//                    }
//
//                } catch (Exception e) {
//                    subscriber.onError(new Throwable("Error From Server"));
//
//                }
//            }
//        });
//    }

    @Override
    public Observable<MaptexSimpleNetworkEntity> sendTicket(final String token, final String params) {
        return Observable.create(new Observable.OnSubscribe<MaptexSimpleNetworkEntity>() {
            @Override
            public void call(Subscriber<? super MaptexSimpleNetworkEntity> subscriber) {
                try {

                    String responseString = sendTicketApi(token, params);

                    if (params == null || "" == params) {
                        subscriber.onError(new Throwable("Post Method Need Param"));
                    } else if (responseString == null) {
                        subscriber.onError(new Throwable("Error From Server"));
                    } else {
                        MaptexSimpleNetworkEntity ticket = gsonMapper.fromJson(responseString, MaptexSimpleNetworkEntity.class);
                        subscriber.onNext(ticket);
                        subscriber.onCompleted();
                    }

                } catch (Exception e) {
                    subscriber.onError(e);

                }
            }
        });
    }

    private String sendTicketApi(String token, String params) throws Exception {
        return ApiConnection.createPOST(new String(URLHelper.MAP_CREATE_TICKET_URL), params, token).call();
    }

    @Override
    public Observable<MaptexSimpleNetworkEntity> sendReportTicket(final String token, final String params) {
        return Observable.create(new Observable.OnSubscribe<MaptexSimpleNetworkEntity>() {
            @Override
            public void call(Subscriber<? super MaptexSimpleNetworkEntity> subscriber) {
                try {

                    String responseString = sendReportTicketApi(token, params);

                    if (params == null || "" == params) {
                        subscriber.onError(new Throwable("Post Method Need Param"));
                    } else if (responseString == null) {
                        subscriber.onError(new Throwable("Error From Server"));
                    } else {
                        MaptexSimpleNetworkEntity ticket = gsonMapper.fromJson(responseString, MaptexSimpleNetworkEntity.class);
                        subscriber.onNext(ticket);
                        subscriber.onCompleted();
                    }

                } catch (Exception e) {
                    subscriber.onError(e);

                }
            }
        });
    }

    private String sendReportTicketApi(String token, String params) throws Exception {
        return ApiConnection.createPOST(new String(URLHelper.MAP_CREATE_REPORT_TICKET_URL), params, token).call();
    }

    @Override
    public Observable<List<MaptexTicket>> getTicketList(final String token) {
        return Observable.create(new Observable.OnSubscribe<List<MaptexTicket>>() {
            @Override
            public void call(Subscriber<? super List<MaptexTicket>> subscriber) {
                try {

                    String responseString = getTicketListApi(token);

                    if (responseString == null) {
                        subscriber.onError(new Throwable("Error From Server"));
                    } else {
                        List<MaptexTicket> ticketList = gsonMapper.fromJson(responseString, new TypeToken<List<MaptexTicket>>() {
                        }.getType());
                        subscriber.onNext(ticketList);
                        subscriber.onCompleted();
                    }

                } catch (Exception e) {
                    subscriber.onError(e);

                }
            }
        });
    }

    private String getTicketListApi(String token) throws Exception {
        return ApiConnection.createGET(new String(URLHelper.MAP_TICKET_URL), token).call();
    }

    @Override
    public Observable<MaptexMedia> uploadMediaByPath(final String token, final String path) {
        return Observable.create(new Observable.OnSubscribe<MaptexMedia>() {
            @Override
            public void call(Subscriber<? super MaptexMedia> subscriber) {
                try {

                    String responseString = uploadMediaByPathApi(token, path);

                    if (responseString == null) {
                        subscriber.onError(new Throwable("Error From Server"));
                    } else {
                        MaptexMedia entity = gsonMapper.fromJson(responseString, MaptexMedia.class);
                        subscriber.onNext(entity);
                        subscriber.onCompleted();
                    }

                } catch (Exception e) {
                    subscriber.onError(e);

                }
            }
        });
    }

    private String uploadMediaByPathApi(String token, String path) throws Exception {

        return ApiConnection.createPOST(new String(URLHelper.UPLOAD_MEDIA_URL), new File(path), token).call();
    }

    @Override
    public Observable<MaptexWeatherItem> getWeatherInfo(final String path) {
        return Observable.create(new Observable.OnSubscribe<MaptexWeatherItem>() {
            @Override
            public void call(Subscriber<? super MaptexWeatherItem> subscriber) {
                try {

                    String responseString = getWeatherInfoApi(path);

                    if (responseString == null) {
                        subscriber.onError(new Throwable("Error From Server"));
                    } else {
                        MaptexWeatherItem entity = gsonMapper.fromJson(responseString, MaptexWeatherItem.class);
                        subscriber.onNext(entity);
                        subscriber.onCompleted();
                    }

                } catch (Exception e) {
                    subscriber.onError(e);
                }
            }
        });
    }

    @Override
    public Observable<List<MaptexWeatherItem>> getAllWeatherInfo() {
        return Observable.create(new Observable.OnSubscribe<List<MaptexWeatherItem>>() {
            @Override
            public void call(Subscriber<? super List<MaptexWeatherItem>> subscriber) {
                try {

                    String responseString = getAllWeatherInfoApi();

                    if (responseString == null) {
                        subscriber.onError(new Throwable("Error From Server"));
                    } else {
                        List<MaptexWeatherItem> entity = gsonMapper.fromJson(responseString, new TypeToken<List<MaptexWeatherItem>>() {
                        }.getType());
                        subscriber.onNext(entity);
                        subscriber.onCompleted();
                    }

                } catch (Exception e) {
                    subscriber.onError(e);
                }
            }
        });
    }

    private String getWeatherInfoApi(String param) throws Exception {
        String apiUrl = new String(URLHelper.WEATHER_URL) + param;
        return ApiConnection.createGET(apiUrl).call();
    }

    private String getAllWeatherInfoApi() throws Exception {
        String apiUrl = new String(URLHelper.WEATHER_ALL_URL);
        return ApiConnection.createGET(apiUrl).call();
    }

    @Override
    public Observable<MaptexMedia> uploadMediaByFile(final String token, final File file) {
        return Observable.create(new Observable.OnSubscribe<MaptexMedia>() {
            @Override
            public void call(Subscriber<? super MaptexMedia> subscriber) {
                try {

                    String responseString = uploadMediaByFileApi(token, file);

                    if (responseString == null) {
                        subscriber.onError(new Throwable("Error From Server"));
                    } else {
                        MaptexMedia entity = gsonMapper.fromJson(responseString, MaptexMedia.class);
                        subscriber.onNext(entity);
                        subscriber.onCompleted();
                    }

                } catch (Exception e) {
                    subscriber.onError(e);

                }
            }
        });
    }

    private String uploadMediaByFileApi(String token, File file) throws Exception {
        return ApiConnection.createPOST(new String(URLHelper.UPLOAD_MEDIA_URL), file, token).call();
    }

    @Override
    public Observable<MaptexSimpleNetworkEntity> createOrModifyTag(final String token, final String params) {
        return Observable.create(new Observable.OnSubscribe<MaptexSimpleNetworkEntity>() {
            @Override
            public void call(Subscriber<? super MaptexSimpleNetworkEntity> subscriber) {
                try {

                    String responseString = createOrModifyTagApi(token, params);

                    if (responseString == null) {
                        subscriber.onError(new Throwable("Error From Server"));
                    } else {
                        MaptexSimpleNetworkEntity entity = gsonMapper.fromJson(responseString, MaptexSimpleNetworkEntity.class);
                        subscriber.onNext(entity);
                        subscriber.onCompleted();
                    }

                } catch (Exception e) {
                    subscriber.onError(e);

                }
            }
        });
    }

    private String createOrModifyTagApi(String token, String params) throws Exception {
        return ApiConnection.createPOST(new String(URLHelper.TAG_BASE_URL), params, token).call();
    }

    @Override
    public Observable<ArrayList<MaptexTag>> getTagList(final int type, final String token, final int skip) {
        return Observable.create(new Observable.OnSubscribe<ArrayList<MaptexTag>>() {
            @Override
            public void call(Subscriber<? super ArrayList<MaptexTag>> subscriber) {
                try {

                    String responseString = getTagListApi(type, token, skip);

                    if (responseString == null) {
                        subscriber.onError(new Throwable("Error From Server"));
                    } else {
                        ArrayList<MaptexTag> entity = gsonMapper.fromJson(responseString, new TypeToken<ArrayList<MaptexTag>>() {
                        }.getType());
                        subscriber.onNext(entity);
                        subscriber.onCompleted();
                    }

                } catch (Exception e) {
                    subscriber.onError(e);

                }
            }
        });
    }

    private String getTagListApi(int type, String token, int skip) throws Exception {
        String apiUrl = URLHelper.getLoadMoreTag(type, skip);

        return ApiConnection.createGET(apiUrl, token).call();
    }

    @Override
    public Observable<MaptexSimpleNetworkEntity> deleteTag(final String token, final String id) {
        return Observable.create(new Observable.OnSubscribe<MaptexSimpleNetworkEntity>() {
            @Override
            public void call(Subscriber<? super MaptexSimpleNetworkEntity> subscriber) {
                try {

                    String responseString = deleteTagApi(token, id);

                    if (responseString == null) {
                        subscriber.onError(new Throwable("Error From Server"));
                    } else {
                        MaptexSimpleNetworkEntity entity = gsonMapper.fromJson(responseString, MaptexSimpleNetworkEntity.class);
                        subscriber.onNext(entity);
                        subscriber.onCompleted();
                    }

                } catch (Exception e) {
                    subscriber.onError(e);

                }
            }
        });
    }

    private String deleteTagApi(String token, String params) throws Exception {
        String apiUrl = URLHelper.deleteTagURL(params);

        return ApiConnection.createDelete(apiUrl, token).call();
    }

    @Override
    public Observable<MaptexSearchResponse> search(final String params) {
        return Observable.create(new Observable.OnSubscribe<MaptexSearchResponse>() {
            @Override
            public void call(Subscriber<? super MaptexSearchResponse> subscriber) {
                try {

                    String responseString = searchPoiApi(params);

                    if (responseString == null) {
                        subscriber.onError(new Throwable("Error From Server"));
                    } else {
                        MaptexSearchResponse entity = gsonMapper.fromJson(responseString, MaptexSearchResponse.class);
                        subscriber.onNext(entity);
                        subscriber.onCompleted();
                    }

                } catch (Exception e) {
                    subscriber.onError(e);

                }
            }
        });
    }


    private String searchPoiApi(String params) throws Exception {

        return ApiConnection.createPOST(new String(URLHelper.SEARCH_ADDRESS_API), params).call();
    }

    @Override
    public Observable<MaptexSearchPostCode> searchPostCode(final String params) {
        return Observable.create(new Observable.OnSubscribe<MaptexSearchPostCode>() {
            @Override
            public void call(Subscriber<? super MaptexSearchPostCode> subscriber) {
                try {

                    String responseString = searchPostCodeApi(params);

                    if (responseString == null) {
                        subscriber.onError(new Throwable("Error From Server"));
                    } else {
                        MaptexSearchPostCode entity = gsonMapper.fromJson(responseString, MaptexSearchPostCode.class);
                        subscriber.onNext(entity);
                        subscriber.onCompleted();
                    }

                } catch (Exception e) {
                    subscriber.onError(e);

                }
            }
        });
    }

    private String searchPostCodeApi(String params) throws Exception {

        return ApiConnection.createGET(new String(URLHelper.SEARCH_POST_CODE_API) + params).call();
    }

    @Override
    public Observable<MaptexSimpleNetworkEntity> createBookmark(final String token, final String params) {
        return Observable.create(new Observable.OnSubscribe<MaptexSimpleNetworkEntity>() {
            @Override
            public void call(Subscriber<? super MaptexSimpleNetworkEntity> subscriber) {
                try {
                    String responseString = createBookmarkApi(token, params);
                    if (responseString == null) {
                        subscriber.onError(new Throwable("Error From Server"));
                    } else {
                        MaptexSimpleNetworkEntity entity = gsonMapper.fromJson(responseString, MaptexSimpleNetworkEntity.class);
                        subscriber.onNext(entity);
                        subscriber.onCompleted();
                    }

                } catch (Exception e) {
                    subscriber.onError(e);
                }
            }
        });
    }

    private String createBookmarkApi(String token, String params) throws Exception {
        return ApiConnection.createPOST(new String(URLHelper.MAP_BOOKMARK_URL), params, token).call();
    }

    @Override
    public Observable<List<MaptexBookmark>> getBookmarkList(final String token, final int skip) {
        return Observable.create(new Observable.OnSubscribe<List<MaptexBookmark>>() {
            @Override
            public void call(Subscriber<? super List<MaptexBookmark>> subscriber) {
                try {
                    String responseString = getBookmarkListApi(token, skip);

                    if (responseString == null) {
                        subscriber.onError(new Throwable("Error From Server"));
                    } else {
                        List<MaptexBookmark> entity = gsonMapper.fromJson(responseString, new TypeToken<List<MaptexBookmark>>() {
                        }.getType());
                        subscriber.onNext(entity);
                        subscriber.onCompleted();
                    }
                } catch (Exception e) {
                    subscriber.onError(e);
                }
            }
        });
    }

    private String getBookmarkListApi(String token, int skip) throws Exception {
        String apiUrl = URLHelper.getBookmarkListURL(skip);
        return ApiConnection.createGET(apiUrl, token).call();
    }

    @Override
    public Observable<MaptexSimpleNetworkEntity> deleteBookmark(final String token, final String id) {
        return Observable.create(new Observable.OnSubscribe<MaptexSimpleNetworkEntity>() {
            @Override
            public void call(Subscriber<? super MaptexSimpleNetworkEntity> subscriber) {
                try {

                    String responseString = deleteBookmarkApi(token, id);

                    if (responseString == null) {
                        subscriber.onError(new Throwable("Error From Server"));
                    } else {
                        MaptexSimpleNetworkEntity entity = gsonMapper.fromJson(responseString, MaptexSimpleNetworkEntity.class);
                        subscriber.onNext(entity);
                        subscriber.onCompleted();
                    }

                } catch (Exception e) {
                    subscriber.onError(e);

                }
            }
        });
    }

    private String deleteBookmarkApi(String token, String id) throws Exception {
        String apiUrl = URLHelper.deleteBookmarkURL(id);

        return ApiConnection.createDelete(apiUrl, token).call();
    }


    @Override
    public Observable<MaptexRouteResponse> getRouteInfo(final MapirLatLng startPosition, final MapirLatLng endPosition, final RouteMode routeMode) {
        return Observable.create(new Observable.OnSubscribe<MaptexRouteResponse>() {
            @Override
            public void call(Subscriber<? super MaptexRouteResponse> subscriber) {
                try {
                    boolean isEndInEvenOdd = false;
                    boolean isEndInTraffic = false;
                    boolean isStartInEvenOdd = false;
                    boolean isStartInTraffic = false;
                    String params = null;
                    List<MapirLatLng> evenOddPolygon = getEvenOddPolygonPoses();
                    List<MapirLatLng> trafficPolygon = getTrafficPolygonPoses();

                    if (MapUtils.containsLocation(startPosition, evenOddPolygon, true))
                        isStartInEvenOdd = true;
                    if (MapUtils.containsLocation(startPosition, trafficPolygon, true))
                        isStartInTraffic = true;
                    if (MapUtils.containsLocation(endPosition, evenOddPolygon, true))
                        isEndInEvenOdd = true;
                    if (MapUtils.containsLocation(endPosition, trafficPolygon, true))
                        isEndInTraffic = true;

                    switch (routeMode) {
                        case BASIC:
                            params = startPosition.longitude + "," + startPosition.latitude + ";"
                                    + endPosition.longitude + "," + endPosition.latitude
                                    + "?alternatives=true&steps=true&overview=false&hints=%3B";
                            break;
                        case EVENODD:
                            if (!(isStartInEvenOdd && isEndInEvenOdd) && !(isEndInEvenOdd || isStartInEvenOdd)) {
                                params = startPosition.longitude + "," + startPosition.latitude + ";"
                                        + endPosition.longitude + "," + endPosition.latitude
                                        + "?alternatives=true&steps=true&overview=false&hints=%3B";
                            } else if (isStartInEvenOdd && isEndInEvenOdd) {
                                subscriber.onError(new Throwable(ServiceSDK.getContext().getResources().getString(R.string.route_evenodd_both_poses_inside)));
                                return;
                            } else if (isEndInEvenOdd || isStartInEvenOdd) {
                                subscriber.onError(new Throwable(ServiceSDK.getContext().getResources().getString(R.string.route_evenodd_one_pos_inside)));
                                return;
                            }
                            break;
                        case TRAFFIC:
                            if (!(isStartInTraffic && isEndInTraffic) && !(isStartInTraffic || isEndInTraffic)) {
                                params = startPosition.longitude + "," + startPosition.latitude + ";"
                                        + endPosition.longitude + "," + endPosition.latitude
                                        + "?alternatives=true&steps=true&overview=false&hints=%3B";
                            } else if (isStartInTraffic && isEndInTraffic) {
                                subscriber.onError(new Throwable(ServiceSDK.getContext().getResources().getString(R.string.route_traffic_both_poses_inside)));
                                return;
                            } else if (isStartInTraffic || isEndInTraffic) {
                                subscriber.onError(new Throwable(ServiceSDK.getContext().getResources().getString(R.string.route_traffic_one_pos_inside)));
                                return;
                            }
                            break;
                        case BOTH:
                            if (!(isStartInEvenOdd && isEndInEvenOdd) && !(isEndInEvenOdd || isStartInEvenOdd)) {
                                params = startPosition.longitude + "," + startPosition.latitude + ";"
                                        + endPosition.longitude + "," + endPosition.latitude
                                        + "?alternatives=true&steps=true&overview=false&hints=%3B";
                            } else if (isStartInEvenOdd && isEndInEvenOdd) {
                                subscriber.onError(new Throwable(ServiceSDK.getContext().getResources().getString(R.string.route_evenodd_both_poses_inside)));
                                return;
                            } else if (isEndInEvenOdd || isStartInEvenOdd) {
                                subscriber.onError(new Throwable(ServiceSDK.getContext().getResources().getString(R.string.route_evenodd_one_pos_inside)));
                                return;
                            }
                            break;
                        case WALK:
                            params = startPosition.longitude + "," + startPosition.latitude + ";"
                                    + endPosition.longitude + "," + endPosition.latitude
                                    + "?alternatives=true&steps=true&overview=false&hints=%3B";
                            break;

                        case BICYCLE:
                            params = startPosition.longitude + "," + startPosition.latitude + ";"
                                    + endPosition.longitude + "," + endPosition.latitude
                                    + "?alternatives=true&steps=true&overview=false&hints=%3B";
                            break;
                    }
                    String responseString = getRouteInfoApi(routeMode, params);

                    if (responseString == null) {
                        subscriber.onError(new Throwable("Error From Server"));
                    } else {
                        MaptexRouteResponse entity = gsonMapper.fromJson(responseString, MaptexRouteResponse.class);
                        subscriber.onNext(entity);
                        subscriber.onCompleted();
                    }

                } catch (Exception e) {
                    subscriber.onError(e);

                }
            }
        });
    }

    private String getRouteInfoApi(RouteMode routeMode, String params) throws Exception {
        String apiUrl = "";
        switch (routeMode) {
            case BICYCLE:
                apiUrl = new String(URLHelper.BICYCLE_ROUTE_DRIVING_URL) + params;
                break;
            case WALK:
                apiUrl = new String(URLHelper.WALK_ROUTE_DRIVING_URL) + params;
                break;
            case BOTH:
                apiUrl = new String(URLHelper.EVENODD_ROUTE_DRIVING_URL) + params;
                break;
            case BASIC:
                apiUrl = new String(URLHelper.DEFAULT_ROUTE_DRIVING_URL) + params;
                break;
            case EVENODD:
                apiUrl = new String(URLHelper.EVENODD_ROUTE_DRIVING_URL) + params;
                break;
            case TRAFFIC:
                apiUrl = new String(URLHelper.TRAFFIC_ROUTE_DRIVING_URL) + params;
                break;
        }


        return ApiConnection.createGET(apiUrl).call();
    }


    private List<MapirLatLng> getEvenOddPolygonPoses() {
        MapUtils mapUtils = new MapUtils();
        return mapUtils.getEvenOddPoses();
    }

    private List<MapirLatLng> getTrafficPolygonPoses() {
        MapUtils mapUtils = new MapUtils();
        return mapUtils.getTrafficPoses();
    }


    @Override
    public Observable<MaptexRouteResponse> getGlobalRouteInfo(final MapirLatLng startPosition, final MapirLatLng endPosition, final RouteMode routeMode) {
        return Observable.create(new Observable.OnSubscribe<MaptexRouteResponse>() {
            @Override
            public void call(Subscriber<? super MaptexRouteResponse> subscriber) {
                try {
                    boolean isEndInEvenOdd = false;
                    boolean isEndInTraffic = false;
                    boolean isStartInEvenOdd = false;
                    boolean isStartInTraffic = false;
                    String params = null;
                    List<MapirLatLng> evenOddPolygon = getEvenOddPolygonPoses();
                    List<MapirLatLng> trafficPolygon = getTrafficPolygonPoses();

                    if (MapUtils.containsLocation(startPosition, evenOddPolygon, true))
                        isStartInEvenOdd = true;
                    if (MapUtils.containsLocation(startPosition, trafficPolygon, true))
                        isStartInTraffic = true;
                    if (MapUtils.containsLocation(endPosition, evenOddPolygon, true))
                        isEndInEvenOdd = true;
                    if (MapUtils.containsLocation(endPosition, trafficPolygon, true))
                        isEndInTraffic = true;

                    switch (routeMode) {
                        case BASIC:
                            params = startPosition.longitude + "," + startPosition.latitude + ";"
                                    + endPosition.longitude + "," + endPosition.latitude
                                    + "?alternatives=true&steps=true&overview=false&hints=%3B";
                            break;
                        case EVENODD:
                            if (!(isStartInEvenOdd && isEndInEvenOdd) && !(isEndInEvenOdd || isStartInEvenOdd)) {
                                params = startPosition.longitude + "," + startPosition.latitude + ";"
                                        + endPosition.longitude + "," + endPosition.latitude
                                        + "?alternatives=true&steps=true&overview=false&hints=%3B";
                            } else if (isStartInEvenOdd && isEndInEvenOdd) {
                                subscriber.onError(new Throwable(ServiceSDK.getContext().getResources().getString(R.string.route_evenodd_both_poses_inside)));
                                return;
                            } else if (isEndInEvenOdd || isStartInEvenOdd) {
                                subscriber.onError(new Throwable(ServiceSDK.getContext().getResources().getString(R.string.route_evenodd_one_pos_inside)));
                                return;
                            }
                            break;
                        case TRAFFIC:
                            if (!(isStartInTraffic && isEndInTraffic) && !(isStartInTraffic || isEndInTraffic)) {
                                params = startPosition.longitude + "," + startPosition.latitude + ";"
                                        + endPosition.longitude + "," + endPosition.latitude
                                        + "?alternatives=true&steps=true&overview=false&hints=%3B";
                            } else if (isStartInTraffic && isEndInTraffic) {
                                subscriber.onError(new Throwable(ServiceSDK.getContext().getResources().getString(R.string.route_traffic_both_poses_inside)));
                                return;
                            } else if (isStartInTraffic || isEndInTraffic) {
                                subscriber.onError(new Throwable(ServiceSDK.getContext().getResources().getString(R.string.route_traffic_one_pos_inside)));
                                return;
                            }
                            break;
                        case BOTH:
                            if (!(isStartInEvenOdd && isEndInEvenOdd) && !(isEndInEvenOdd || isStartInEvenOdd)) {
                                params = startPosition.longitude + "," + startPosition.latitude + ";"
                                        + endPosition.longitude + "," + endPosition.latitude
                                        + "?alternatives=true&steps=true&overview=false&hints=%3B";
                            } else if (isStartInEvenOdd && isEndInEvenOdd) {
                                subscriber.onError(new Throwable(ServiceSDK.getContext().getResources().getString(R.string.route_evenodd_both_poses_inside)));
                                return;
                            } else if (isEndInEvenOdd || isStartInEvenOdd) {
                                subscriber.onError(new Throwable(ServiceSDK.getContext().getResources().getString(R.string.route_evenodd_one_pos_inside)));
                                return;
                            }
                            break;
                        case WALK:
                            params = startPosition.longitude + "," + startPosition.latitude + ";"
                                    + endPosition.longitude + "," + endPosition.latitude
                                    + "?alternatives=true&steps=true&overview=false&hints=%3B";
                            break;

                        case BICYCLE:
                            params = startPosition.longitude + "," + startPosition.latitude + ";"
                                    + endPosition.longitude + "," + endPosition.latitude
                                    + "?alternatives=true&steps=true&overview=false&hints=%3B";
                            break;
                    }
                    String responseString = getGlobalRouteInfoApi(routeMode, params);

                    if (responseString == null) {
                        subscriber.onError(new Throwable("Error From Server"));
                    } else {
                        MaptexRouteResponse entity = gsonMapper.fromJson(responseString, MaptexRouteResponse.class);
                        subscriber.onNext(entity);
                        subscriber.onCompleted();
                    }

                } catch (Exception e) {
                    subscriber.onError(e);

                }
            }
        });
    }

    private String getGlobalRouteInfoApi(RouteMode routeMode, String params) throws Exception {
        String apiUrl = "";
        switch (routeMode) {
            case BICYCLE:
                apiUrl = new String(URLHelper.BICYCLE_GLOBAL_ROUTE_DRIVING_URL) + params;
                break;
            case WALK:
                apiUrl = new String(URLHelper.WALK_GLOBAL_ROUTE_DRIVING_URL) + params;
                break;
            case BOTH:
                apiUrl = new String(URLHelper.EVENODD_GLOBAL_ROUTE_DRIVING_URL) + params;
                break;
            case BASIC:
                apiUrl = new String(URLHelper.DEFAULT_GLOBAL_ROUTE_DRIVING_URL) + params;
                break;
            case EVENODD:
                apiUrl = new String(URLHelper.EVENODD_GLOBAL_ROUTE_DRIVING_URL) + params;
                break;
            case TRAFFIC:
                apiUrl = new String(URLHelper.TRAFFIC_GLOBAL_ROUTE_DRIVING_URL) + params;
                break;
        }


        return ApiConnection.createGET(apiUrl).call();
    }

    @Override
    public Observable getStaticMap(final double latitude, final double longitude, int zoom) {
        return Observable.create(new Observable.OnSubscribe<Bitmap>() {
            @Override
            public void call(final Subscriber<? super Bitmap> subscriber) {
                try {

                    String url = String.format(STATIC_MAP,
                            450, 450,
                            longitude, latitude, 17);

                    ImageLoader.getInstance().loadImage(url, new ImageLoadingListener() {
                        @Override
                        public void onLoadingStarted(String imageUri, View view) {

                        }

                        @Override
                        public void onLoadingFailed(String imageUri, View view, FailReason failReason) {
                            subscriber.onError(failReason.getCause());
                        }

                        @Override
                        public void onLoadingComplete(String imageUri, View view, Bitmap loadedImage) {
                            subscriber.onNext(loadedImage);
                        }

                        @Override
                        public void onLoadingCancelled(String imageUri, View view) {
                            subscriber.onError(new Throwable("Loading Image Cancelled"));
                        }
                    });

                } catch (Exception e) {
                    subscriber.onError(e);

                }
            }
        });
    }

    @Override
    public Observable<MaptexNetworkEntity> sendFeedback(final RequestBody body) {
        return Observable.create(new Observable.OnSubscribe<MaptexNetworkEntity>() {
            @Override
            public void call(Subscriber<? super MaptexNetworkEntity> subscriber) {
                try {

                    String responseString = sendFeedbackApi(body);

                    if (responseString == null) {
                        subscriber.onError(new Throwable("Error From Server"));
                    } else {
                        MaptexNetworkEntity entity = new MaptexNetworkEntity();
                        entity.dispMessage = "success";
                        subscriber.onNext(entity);
                        subscriber.onCompleted();
                    }

                } catch (Exception e) {
                    subscriber.onError(e);

                }
            }
        });
    }

    private String sendFeedbackApi(RequestBody body) throws Exception {
        return ApiConnection.createPOST(new String(URLHelper.SUPPORT_MAP_URL), body).call();
    }

    @Override
    public Observable<MaptexLoginResponse> loginUser(final String params) {
        return Observable.create(new Observable.OnSubscribe<MaptexLoginResponse>() {
            @Override
            public void call(Subscriber<? super MaptexLoginResponse> subscriber) {
                try {

                    String responseString = loginUserApi(params);

                    if (responseString == null) {
                        subscriber.onError(new Throwable("Error From Server"));
                    } else {
                        MaptexLoginResponse entity = gsonMapper.fromJson(responseString, MaptexLoginResponse.class);
                        subscriber.onNext(entity);
                        subscriber.onCompleted();
                    }

                } catch (Exception e) {
                    subscriber.onError(e);

                }
            }
        });
    }

    private String loginUserApi(String params) throws Exception {
        return ApiConnection.createPOST(new String(SIGN_IN_OUT_URL), params).call();
    }

    @Override
    public Observable<MaptexForgotEntity> forgetPassword(final String params) {
        return Observable.create(new Observable.OnSubscribe<MaptexForgotEntity>() {
            @Override
            public void call(Subscriber<? super MaptexForgotEntity> subscriber) {
                try {
                    String responseString = forgetPasswordApi(params);
                    if (responseString == null) {
                        subscriber.onError(new Throwable("Error From Server"));
                    } else {
                        MaptexForgotEntity entity = gsonMapper.fromJson(responseString, MaptexForgotEntity.class);
                        subscriber.onNext(entity);
                        subscriber.onCompleted();
                    }

                } catch (Exception e) {
                    subscriber.onError(e);
                }
            }
        });
    }

    private String forgetPasswordApi(String params) throws Exception {
        String apiUrl = new String(FORGOT_PASSWORD_URL);
        return ApiConnection.createGET(apiUrl + "user=" + params).call();
    }

    @Override
    public Observable<MaptexAuthenticateResponse> signupUser(final String params) {
        return Observable.create(new Observable.OnSubscribe<MaptexAuthenticateResponse>() {
            @Override
            public void call(Subscriber<? super MaptexAuthenticateResponse> subscriber) {
                try {

                    String responseString = signupUserApi(params);

                    if (responseString == null) {
                        subscriber.onError(new Throwable("Error From Server"));
                    } else {
                        MaptexAuthenticateResponse entity = gsonMapper.fromJson(responseString, MaptexAuthenticateResponse.class);
                        subscriber.onNext(entity);
                        subscriber.onCompleted();
                    }

                } catch (Exception e) {
                    subscriber.onError(e);

                }
            }
        });
    }

    private String signupUserApi(String params) throws Exception {
        return ApiConnection.createPOST(new String(SIGN_UP_URL), params).call();
    }

    @Override
    public Observable<MaptexSimpleNetworkEntity> signoutUser(final String token) {
        return Observable.create(new Observable.OnSubscribe<MaptexSimpleNetworkEntity>() {
            @Override
            public void call(Subscriber<? super MaptexSimpleNetworkEntity> subscriber) {
                try {

                    String responseString = signoutUserApi(token);

                    if (responseString == null) {
                        subscriber.onError(new Throwable("Error From Server"));
                    } else {
                        MaptexSimpleNetworkEntity entity = gsonMapper.fromJson(responseString, MaptexSimpleNetworkEntity.class);
                        subscriber.onNext(entity);
                        subscriber.onCompleted();
                    }

                } catch (Exception e) {
                    subscriber.onError(e);

                }
            }
        });
    }

    private String signoutUserApi(String token) throws Exception {
        return ApiConnection.createDelete(new String(SIGN_IN_OUT_URL), token).call();
    }

    @Override
    public Observable<MaptexLoginResponse> loginUserByGmail(final String token) {
        return Observable.create(new Observable.OnSubscribe<MaptexLoginResponse>() {
            @Override
            public void call(Subscriber<? super MaptexLoginResponse> subscriber) {
                try {

                    String responseString = loginUserByGmailApi(token);

                    if (responseString == null) {
                        subscriber.onError(new Throwable("Error From Server"));
                    } else {
                        MaptexLoginResponse entity = gsonMapper.fromJson(responseString, MaptexLoginResponse.class);
                        subscriber.onNext(entity);
                        subscriber.onCompleted();
                    }

                } catch (Exception e) {
                    subscriber.onError(e);

                }
            }
        });
    }

    private String loginUserByGmailApi(String token) throws Exception {
        String apiUrl = new String(LOCAL_URL_GMAIL_LOGIN) + token;
        return ApiConnection.createGET(apiUrl).call();
    }

    @Override
    public Observable<MaptexAuthenticateResponse> verifyUser(final String params) {
        return Observable.create(new Observable.OnSubscribe<MaptexAuthenticateResponse>() {
            @Override
            public void call(Subscriber<? super MaptexAuthenticateResponse> subscriber) {
                try {

                    String responseString = verifyUserApi(params);

                    if (responseString == null) {
                        subscriber.onError(new Throwable("Error From Server"));
                    } else {
                        MaptexAuthenticateResponse entity = gsonMapper.fromJson(responseString, MaptexAuthenticateResponse.class);
                        subscriber.onNext(entity);
                        subscriber.onCompleted();
                    }

                } catch (Exception e) {
                    subscriber.onError(e);

                }
            }
        });
    }

    private String verifyUserApi(String params) throws Exception {
        String apiUrl = new String(VERIFICATION_CODE_URL) + params;
        return ApiConnection.createPOST(apiUrl).call();
    }

    @Override
    public Observable<MaptexResendVerification> resendVerificationCode(final String params) {
        return Observable.create(new Observable.OnSubscribe<MaptexResendVerification>() {
            @Override
            public void call(Subscriber<? super MaptexResendVerification> subscriber) {
                try {

                    String responseString = resendVerificationCodeApi(params);

                    if (responseString == null) {
                        subscriber.onError(new Throwable("Error From Server"));
                    } else {
                        MaptexResendVerification entity = gsonMapper.fromJson(responseString, MaptexResendVerification.class);
                        subscriber.onNext(entity);
                        subscriber.onCompleted();
                    }

                } catch (Exception e) {
                    subscriber.onError(e);

                }
            }
        });
    }

    private String resendVerificationCodeApi(String params) throws Exception {
        String apiUrl = new String(VERIFICATION_CODE_URL) + params;
        return ApiConnection.createPOST(apiUrl).call();
    }

    @Override
    public Observable<List<MaptexTag>> getTag(final String token, final String params) {
        return Observable.create(new Observable.OnSubscribe<List<MaptexTag>>() {
            @Override
            public void call(Subscriber<? super List<MaptexTag>> subscriber) {
                try {

                    String responseString = getTagApi(token, params);

                    if (responseString == null) {
                        subscriber.onError(new Throwable("Error From Server"));
                    } else {
                        List<MaptexTag> entity = gsonMapper.fromJson(responseString, new TypeToken<List<MaptexTag>>() {
                        }.getType());
//                        List<MaptexTag> entity = gsonMapper.fromJson(responseString, List<MaptexTag>.class);
                        subscriber.onNext(entity);
                        subscriber.onCompleted();
                    }

                } catch (Exception e) {
                    subscriber.onError(e);

                }
            }
        });
    }

    private String getTagApi(String token, String body) throws Exception {
        String apiUrl = URLHelper.getTagURL(body);

        return ApiConnection.createGET(apiUrl, token).call();
    }

    @Override
    public Observable<MaptexProfileResponse> getProfile(final String userName, final String token) {

        return Observable.create(new Observable.OnSubscribe<MaptexProfileResponse>() {
            @Override
            public void call(Subscriber<? super MaptexProfileResponse> subscriber) {
                try {

                    String responseString = getProfileApi(userName, token);

                    if (responseString == null) {
                        subscriber.onError(new Throwable("Error From Server"));
                    } else {
                        MaptexProfileResponse entity = gsonMapper.fromJson(responseString, MaptexProfileResponse.class);
                        subscriber.onNext(entity);
                        subscriber.onCompleted();
                    }

                } catch (Exception e) {
                    subscriber.onError(e);

                }
            }
        });
    }

    private String getProfileApi(String userName, String token) throws Exception {
        String apiUrl = new String(GET_PROFILE) + userName;

        return ApiConnection.createGET(apiUrl, token).call();
    }


    @Override
    public Observable<MaptexProfileResponse> updateProfile(final String userName, final String token, final String body) {
        return Observable.create(new Observable.OnSubscribe<MaptexProfileResponse>() {
            @Override
            public void call(Subscriber<? super MaptexProfileResponse> subscriber) {
                try {

                    String responseString = updateProfileApi(userName, token, body);

                    if (responseString == null) {
                        subscriber.onError(new Throwable("Error From Server"));
                    } else {
                        MaptexProfileResponse entity = gsonMapper.fromJson(responseString, MaptexProfileResponse.class);
                        subscriber.onNext(entity);
                        subscriber.onCompleted();
                    }

                } catch (Exception e) {
                    subscriber.onError(e);

                }
            }
        });
    }

    @Override
    public Observable<MaptexPublicTransportResponse> getPublicTrnasportRoute(final MapirLatLng startPosition,
                                                                             final MapirLatLng endPosition,
                                                                             final Double walkDistance,
                                                                             final PublicTransportMode routeMode) {
        return Observable.create(new Observable.OnSubscribe<MaptexPublicTransportResponse>() {
            @Override
            public void call(Subscriber<? super MaptexPublicTransportResponse> subscriber) {
                try {

                    String currentTime = getCurrentTime();
                    String currentDate = getCurrentDate();

                    String responseString = getPublicTrnasportRouteApi(startPosition, endPosition,
                            currentTime, currentDate, walkDistance,
                            routeMode.params());

                    if (responseString == null) {
                        subscriber.onError(new Throwable("Error From Server"));
                    } else {
                        MaptexPublicTransportResponse entity = gsonMapper.fromJson(responseString, MaptexPublicTransportResponse.class);
                        subscriber.onNext(entity);
                        subscriber.onCompleted();
                    }

                } catch (Exception e) {
                    subscriber.onError(e);

                }
            }
        });
    }

    private String getPublicTrnasportRouteApi(MapirLatLng startPosition, MapirLatLng endPosition, String currentTime,
                                              String currentDate, Double walkDistance, String routeMode) throws Exception {
        String apiUrl = URLHelper.getPublicTransportURL(startPosition, endPosition, currentTime, currentDate, walkDistance, routeMode);

        return ApiConnection.createGET(apiUrl).call();
    }

    private String getCurrentDate() {
        Calendar calendar = Calendar.getInstance();
        return (calendar.get(Calendar.MONTH) + 1 > 9 ? String.valueOf(calendar.get(Calendar.MONTH) + 1) : "0" + String.valueOf(calendar.get(Calendar.MONTH) + 1))
                + "-" + (calendar.get(Calendar.DAY_OF_MONTH) > 9 ? String.valueOf(calendar.get(Calendar.DAY_OF_MONTH)) : ("0" + calendar.get(Calendar.DAY_OF_MONTH)))
                + "-" + calendar.get(Calendar.YEAR);
    }

    private String getCurrentTime() {
        Calendar calendar = Calendar.getInstance();
        return calendar.get(Calendar.HOUR_OF_DAY) + ":" + calendar.get(Calendar.MINUTE) +
                (calendar.get(Calendar.HOUR_OF_DAY) > 11 ? "p" : "a");
    }

    private String updateProfileApi(String userName, String token, String body) throws Exception {
        String apiUrl = new String(UPDATE_PROFILE) + userName;

        return ApiConnection.createPUT(apiUrl, token, body).call();
    }

    public ArrayList<MapirLatLng> getEvenOddPoses(Context context) {
        return getMapPoses(context, "even-odd");
    }

    public ArrayList<MapirLatLng> getTrafficPoses(Context context) {
        return getMapPoses(context, "traffic");
    }

    private ArrayList<MapirLatLng> getMapPoses(Context context, String type) {
        ArrayList<MapirLatLng> latLngs = new ArrayList<>();
        String json = loadJSONFromAsset(context, type);
        if (json != null) {
            try {
                JSONArray jsonArray = new JSONArray(json);
                for (int i = 0; i < jsonArray.length(); i++) {
                    latLngs.add(new MapirLatLng(jsonArray.getJSONArray(i).getDouble(1), jsonArray.getJSONArray(i).getDouble(0)));
                }

            } catch (JSONException e) {
                e.printStackTrace();
            }
        }
        return latLngs;
    }

    private String loadJSONFromAsset(Context context, String type) {
        String json = null;
        try {
            InputStream is = null;
            switch (type) {
                case "traffic":
                    is = context.getAssets().open("traffic-json.json");
                    break;
                case "even-odd":
                    is = context.getAssets().open("even-odd.json");
                    break;
                case "earthqueke":
                    is = context.getAssets().open("earthqueke.geojson");
                    break;
                case "metro_stations_isfahan":
                    is = context.getAssets().open("metro_stations_isfahan.geojson");
                    break;
                case "metro_lines_isfahan":
                    is = context.getAssets().open("metro_lines_isfahan.geojson");
                    break;
                case "metro_stations_mashhad":
                    is = context.getAssets().open("metro_stations_mashhad.geojson");
                    break;
                case "metro_lines_mashhad":
                    is = context.getAssets().open("metro_lines_mashhad.geojson");
                    break;
                case "metro_stations_shiraz":
                    is = context.getAssets().open("metro_stations_shiraz.geojson");
                    break;
                case "metro_lines_shiraz":
                    is = context.getAssets().open("metro_lines_shiraz.geojson");
                    break;
                case "metro_stations_tabriz":
                    is = context.getAssets().open("metro_stations_tabriz.geojson");
                    break;
                case "metro_lines_tabriz":
                    is = context.getAssets().open("metro_lines_tabriz.geojson");
                    break;
                case "metro_stations_tehran":
                    is = context.getAssets().open("metro_stations_tehran.geojson");
                    break;
                case "metro_lines_tehran":
                    is = context.getAssets().open("metro_lines_tehran.geojson");
                    break;
                case "bus_lines":
                    is = context.getAssets().open("bus_lines.geojson");
                    break;
                case "bus_stations":
                    is = context.getAssets().open("bus_stations.geojson");
                    break;
            }
            if (is != null) {
                int size = is.available();
                byte[] buffer = new byte[size];
                is.read(buffer);
                is.close();
                json = new String(buffer, "UTF-8");
            }
        } catch (IOException ex) {
            ex.printStackTrace();
            return null;
        }

        return json;
    }
}
