package ir.map.sdk_map.maps;

import android.app.Activity;
import android.app.AlertDialog;
import android.content.ActivityNotFoundException;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.net.Uri;
import android.support.annotation.NonNull;
import android.view.View;
import android.widget.ArrayAdapter;
import android.widget.Toast;

import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Set;

import ir.map.sdk_map.MapStrictMode;
import ir.map.sdk_map.Mapir;
import ir.map.sdk_map.R;
import ir.map.sdk_map.attribution.Attribution;
import ir.map.sdk_map.attribution.AttributionParser;
import ir.map.sdk_map.camera.CameraPosition;
import ir.map.sdk_map.style.sources.Source;

/**
 * Responsible for managing attribution interactions on the map.
 * <p>
 * When the user clicks the attribution icon, {@link AttributionDialogManager#onClick(View)} will be invoked.
 * An attribution dialog will be shown to the user with contents based on the attributions found in the map style.
 * Additionally an telemetry option item is shown to configure telemetry settings.
 * </p>
 */
public class AttributionDialogManager implements View.OnClickListener, DialogInterface.OnClickListener {

  private static final String MAP_FEEDBACK_URL = "https://www.mapbox.com/map-feedback";
  private static final String MAP_FEEDBACK_LOCATION_FORMAT = MAP_FEEDBACK_URL + "/#/%f/%f/%d";

  private final Context context;
  private final MapirMap mapirMap;
  private Set<Attribution> attributionSet;

  public AttributionDialogManager(@NonNull Context context, @NonNull MapirMap mapirMap) {
    this.context = context;
    this.mapirMap = mapirMap;
  }

  // Called when someone presses the attribution icon on the map
  @Override
  public void onClick(View view) {
    attributionSet = new AttributionBuilder(mapirMap, view.getContext()).build();

    boolean isActivityFinishing = false;
    if (context instanceof Activity) {
      isActivityFinishing = ((Activity) context).isFinishing();
    }

    // check is hosting activity isn't finishing
    // https://github.com/mapbox/mapbox-gl-native/issues/11238
    if (!isActivityFinishing) {
      showAttributionDialog(getAttributionTitles());
    }
  }

  protected void showAttributionDialog(String[] attributionTitles) {
    AlertDialog.Builder builder = new AlertDialog.Builder(context);
    builder.setTitle(R.string.mapir_attributionsDialogTitle);
    builder.setAdapter(new ArrayAdapter<>(context, R.layout.mapir_attribution_list_item, attributionTitles), this);
    builder.show();
  }

  private String[] getAttributionTitles() {
    List<String> titles = new ArrayList<>();
    for (Attribution attribution : attributionSet) {
      titles.add(attribution.getTitle());
    }
    return titles.toArray(new String[titles.size()]);
  }

  // Called when someone selects an attribution or telemetry settings from the dialog
  @Override
  public void onClick(DialogInterface dialog, int which) {
    if (isLatestEntry(which)) {
      showTelemetryDialog();
    } else {
      showMapFeedbackWebPage(which);
    }
  }

  private boolean isLatestEntry(int attributionKeyIndex) {
    return attributionKeyIndex == getAttributionTitles().length - 1;
  }

  private void showTelemetryDialog() {
    AlertDialog.Builder builder = new AlertDialog.Builder(context);
    builder.setTitle(R.string.mapir_attributionTelemetryTitle);
    builder.setMessage(R.string.mapir_attributionTelemetryMessage);
    builder.setPositiveButton(R.string.mapir_attributionTelemetryPositive, new DialogInterface.OnClickListener() {
      @Override
      public void onClick(DialogInterface dialog, int which) {
        TelemetryDefinition telemetry = Mapir.getTelemetry();
        if (telemetry != null) {
          telemetry.setUserTelemetryRequestState(true);
        }
        dialog.cancel();
      }
    });
    builder.setNeutralButton(R.string.mapir_attributionTelemetryNeutral, new DialogInterface.OnClickListener() {
      @Override
      public void onClick(DialogInterface dialog, int which) {
        showWebPage(context.getResources().getString(R.string.mapir_telemetryLink));
        dialog.cancel();
      }
    });
    builder.setNegativeButton(R.string.mapir_attributionTelemetryNegative, new DialogInterface.OnClickListener() {
      @Override
      public void onClick(DialogInterface dialog, int which) {
        TelemetryDefinition telemetry = Mapir.getTelemetry();
        if (telemetry != null) {
          telemetry.setUserTelemetryRequestState(false);
        }
        dialog.cancel();
      }
    });
    builder.show();
  }

  private void showMapFeedbackWebPage(int which) {
    Attribution[] attributions = attributionSet.toArray(new Attribution[attributionSet.size()]);
    String url = attributions[which].getUrl();
    if (url.contains(MAP_FEEDBACK_URL)) {
      url = buildMapFeedbackMapUrl(mapirMap.getCameraPosition());
    }
    showWebPage(url);
  }

  private String buildMapFeedbackMapUrl(CameraPosition cameraPosition) {
    // appends current location to the map feedback url if available
    return cameraPosition != null ? String.format(Locale.getDefault(),
      MAP_FEEDBACK_LOCATION_FORMAT, cameraPosition.target.getLongitude(), cameraPosition.target.getLatitude(),
      (int) cameraPosition.zoom) : MAP_FEEDBACK_URL;
  }

  private void showWebPage(@NonNull String url) {
    try {
      Intent intent = new Intent(Intent.ACTION_VIEW);
      intent.setData(Uri.parse(url));
      context.startActivity(intent);
    } catch (ActivityNotFoundException exception) {
      // explicitly handling if the device hasn't have a web browser installed. #8899
      Toast.makeText(context, R.string.mapir_attributionErrorNoBrowser, Toast.LENGTH_LONG).show();
      MapStrictMode.strictModeViolation(exception);
    }
  }

  private static class AttributionBuilder {

    private final MapirMap mapirMap;
    private final WeakReference<Context> context;

    AttributionBuilder(MapirMap mapirMap, Context context) {
      this.mapirMap = mapirMap;
      this.context = new WeakReference<>(context);
    }

    private Set<Attribution> build() {
      Context context = this.context.get();
      if (context == null) {
        return Collections.emptySet();
      }

      List<String> attributions = new ArrayList<>();
      String attribution;
      for (Source source : mapirMap.getSources()) {
        attribution = source.getAttribution();
        if (!attribution.isEmpty()) {
          attributions.add(source.getAttribution());
        }
      }

      return new AttributionParser.Options()
        .withCopyrightSign(true)
        .withImproveMap(true)
        .withContext(context)
        .withTelemetryAttribution(true)
        .withAttributionData(attributions.toArray(new String[attributions.size()]))
        .build().getAttributions();
    }
  }
}
