package io.intercom.android.nexus;

import androidx.annotation.NonNull;
import android.util.LruCache;

import com.intercom.twig.Twig;

import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

class NexusEventPropagator implements NexusListener {
    private final LruCache<String, Boolean> cache = new LruCache<>(100);
    private final List<NexusListener> listeners = new CopyOnWriteArrayList<>();
    private final Twig twig;

    public NexusEventPropagator(Twig twig) {
        this.twig = twig;
    }

    void addListener(@NonNull NexusListener listener) {
        listeners.add(listener);
    }

    void removeListener(@NonNull NexusListener listener) {
        listeners.remove(listener);
    }

    void cacheEvent(NexusEvent event) {
        cache.put(event.getGuid(), true);
    }

    @Override public void notifyEvent(@NonNull NexusEvent event) {
        if (event.getEventType() != NexusEventType.UNKNOWN) {
            synchronized (cache) {
                if (cache.get(event.getGuid()) == null) {
                    cache.put(event.getGuid(), true);
                    twig.internal("notifying listeners of event: " +  event.getEventType());
                    for (NexusListener listener : listeners) {
                        listener.notifyEvent(event);
                    }
                } else {
                    twig.internal("dropping event, already in cache: " + event.toStringEncodedJsonObject());
                }
            }
        }
    }

    @Override public void onConnect() {
        twig.d("notifying listeners that a connection opened");
        for (NexusListener listener : listeners) {
            listener.onConnect();
        }
    }

    @Override public void onConnectFailed() {
        twig.d("notifying listeners that a connection failed to open");
        for (NexusListener listener : listeners) {
            listener.onConnectFailed();
        }
    }

    @Override public void onShutdown() {
        twig.d("notifying listeners that a connection has shutdown");
        for (NexusListener listener : listeners) {
            listener.onShutdown();
        }
    }
}
