package io.intercom.android.sdk.push;

import android.app.Notification;
import android.app.NotificationManager;
import android.content.Context;
import android.support.annotation.VisibleForTesting;

import com.intercom.twig.Twig;

import java.util.ArrayList;
import java.util.List;

import io.intercom.android.sdk.logger.LumberMill;

import static java.util.Collections.synchronizedList;

public class SystemNotificationManager {
    private static final int NOTIFICATION_ID = 9999997;

    @VisibleForTesting final List<PushPayload> receivedPayloads = synchronizedList(new ArrayList<PushPayload>());

    private final Twig twig = LumberMill.getLogger();
    private final SystemNotificationFactory systemNotificationFactory;
    private final NotificationManager androidNotificationManager;

    public SystemNotificationManager(NotificationManager androidNotificationManager) {
        this(androidNotificationManager, new SystemNotificationFactory());
    }

    @VisibleForTesting SystemNotificationManager(NotificationManager androidNotificationManager,
                                                 SystemNotificationFactory systemNotificationFactory) {
        this.androidNotificationManager = androidNotificationManager;
        this.systemNotificationFactory = systemNotificationFactory;
    }

    public void clear() {
        if (!receivedPayloads.isEmpty()) {
            twig.i("Removing Intercom push notifications.");
        }

        androidNotificationManager.cancel(NOTIFICATION_ID);
        receivedPayloads.clear();
    }

    void createPushOnlyNotification(PushPayload payload, Context context) {
        Notification notification = systemNotificationFactory.createPushOnlyNotification(payload, context);
        String pushOnlyConversationId = payload.getPushOnlyConversationId();
        androidNotificationManager.notify(pushOnlyConversationId, pushOnlyConversationId.hashCode(), notification);
    }

    void createNotification(PushPayload payload, Context context) {
        // Hacky way to check for duplicate pushes.
        // This is caused by the device having multiple working tokens registered
        if (receivedPayloads.contains(payload)) {
            return;
        }

        receivedPayloads.add(payload);

        Notification notification = receivedPayloads.size() == 1
                ? systemNotificationFactory.createSingleNotification(payload, context)
                : systemNotificationFactory.createGroupedNotification(receivedPayloads, context);

        androidNotificationManager.notify(NOTIFICATION_ID, notification);
    }
}
