package im.zego.zegoexpress.callback;

import im.zego.zegoexpress.constants.*;
import im.zego.zegoexpress.callback.*;
import im.zego.zegoexpress.entity.*;
import java.nio.*;
import java.util.*;
import org.json.*;
import im.zego.zegoexpress.*;
import android.app.Application;
import android.graphics.*;

public abstract class IZegoEventHandler {

    /**
     * The callback for obtaining debugging error information.
     *
     * When the APIs are not used correctly, the callback prompts for detailed error information, which is controlled by the [setDebugVerbose] interface
     *
     * @param errorCode Error code, please refer to the common error code document [https://doc-en.zego.im/en/308.html] for details
     * @param funcName Function name
     * @param info Detailed error information
     */
    public void onDebugError(int errorCode, String funcName, String info){

    }

    /**
     * The callback triggered when the engine state changes.
     *
     * When developers call [startPreview] ,[startPublishing] or [startPlaying] for the first time, the engine will start.
     * When developers have [loginRoom], they must call [logoutRoom], and the audio and video engine will stop.
     *
     * @param state engine state
     */
    public void onEngineStateUpdate(ZegoEngineState state){

    }

    /**
     * The callback triggered when the room connection state changes.
     *
     * This callback is triggered when the connection status of the room changes, and the reason for the change is notified. Developers can use this callback to determine the status of the current user in the room. If the connection is being requested for a long time, the general probability is that the user's network is unstable.
     *
     * @param roomID Room ID, a string of up to 128 bytes in length.
     * @param state Changed room state
     * @param errorCode Error code, please refer to the [common error code document](https://doc-en.zego.im/en/308.html) for details
     * @param extendedData Extended Information with state updates. As the standby, only an empty json table is currently returned
     */
    public void onRoomStateUpdate(String roomID, ZegoRoomState state, int errorCode, JSONObject extendedData){

    }

    /**
     * The callback triggered when the number of other users in the room increases or decreases.
     *
     * Note that the callback is only triggered when the isUserStatusNotify parameter in the ZegoRoomConfig passed loginRoom function is true. Developers can use this callback to determine the situation of users in the room.
     * If developers need to use ZEGO room users notifications, please make sure that each login user sets isUserStatusNotify to true
     * When a user logs in to a room for the first time, other users already exist in this room, and a user list of the type of addition is received.
     * When the user is already in the room, other users in this room will trigger this callback to notify the changed users when they enter or exit the room.
     *
     * @param roomID Room ID where the user is logged in, a string of up to 128 bytes in length.
     * @param updateType Update type (add/delete)
     * @param userList List of users changed in the current room
     */
    public void onRoomUserUpdate(String roomID, ZegoUpdateType updateType, ArrayList<ZegoUser> userList){

    }

    /**
     * The callback triggered every 30 seconds to report the current number of online users.
     *
     * This interface is called back every 30 seconds.
     * Developers can use this callback to show the number of user online in the current room.
     *
     * @param roomID Room ID where the user is logged in, a string of up to 128 bytes in length.
     * @param count Count of online users
     */
    public void onRoomOnlineUserCountUpdate(String roomID, int count){

    }

    /**
     * The callback triggered when the number of streams published by the other users in the same room increases or decreases.
     *
     * When a user logs in to a room for the first time, there are other users in the room who are publishing streams, and will receive a stream list of the added type.
     * When the user is already in the room, other users in this room will trigger this callback to notify the changed stream list when adding or deleting streams.
     * Developers can use this callback to determine if there are other users in the same room who have added or stopped streaming, in order to implement active play stream [startPlayingStream] or active stop playing stream [stopPlayingStream], and use simultaneous Changes to Streaming render UI widget;
     *
     * @param roomID Room ID where the user is logged in, a string of up to 128 bytes in length.
     * @param updateType Update type (add/delete)
     * @param streamList Updated stream list
     */
    public void onRoomStreamUpdate(String roomID, ZegoUpdateType updateType, ArrayList<ZegoStream> streamList){

    }

    /**
     * The callback triggered when there is an update on the extra information of the streams published by other users in the same room.
     *
     * When a user publishing the stream update the extra information of the stream in the same room, other users in the same room will receive the callback.
     * The stream extra information is an extra information identifier of the stream ID. Unlike the stream ID, which cannot be modified during the publishing process, the stream extra information can be modified midway through the stream corresponding to the stream ID.
     * Developers can synchronize variable content related to stream IDs based on stream additional information.
     *
     * @param roomID Room ID where the user is logged in, a string of up to 128 bytes in length.
     * @param streamList List of streams that the extra info was updated.
     */
    public void onRoomStreamExtraInfoUpdate(String roomID, ArrayList<ZegoStream> streamList){

    }

    /**
     * @deprecated This API is invalid in this SDK package and should not be used. If you need to use this API, please contact ZEGO technical support.
     */
    @Deprecated
    public void onRoomRecvReliableMessage(String roomID, ZegoReliableMessage message){

    }

    /**
     * @deprecated This API is invalid in this SDK package and should not be used. If you need to use this API, please contact ZEGO technical support.
     */
    @Deprecated
    public void onRoomUpdateReliableMessageInfo(String roomID, String msgType, int latestSeq){

    }

    /**
     * The callback triggered when the state of stream publishing changes.
     *
     * After publishing the stream successfully, the notification of the publish stream state change can be obtained through the callback interface.
     * You can roughly judge the user's uplink network status based on whether the state parameter is in [PUBLISH_REQUESTING].
     * ExtendedData is extended information with state updates. If you use ZEGO's CDN content distribution network, after the stream is successfully published, the keys of the content of this parameter are flv_url_list, rtmp_url_list, hls_url_list. These correspond to the publishing stream URLs of the flv, rtmp, and hls protocols.
     *
     * @param streamID Stream ID
     * @param state Status of publishing stream
     * @param errorCode The error code corresponding to the status change of the publish stream. Please refer to the common error code documentation [https://doc-en.zego.im/en/308.html] for details.
     * @param extendedData Extended information with state updates.
     */
    public void onPublisherStateUpdate(String streamID, ZegoPublisherState state, int errorCode, JSONObject extendedData){

    }

    /**
     * The callback triggered every 3 seconds to report the current stream publishing quality.
     *
     * After the successful publish, the callback will be received every 3 seconds. Through the callback, the collection frame rate, bit rate, RTT, packet loss rate and other quality data of the published audio and video stream can be obtained, and the health of the publish stream can be monitored in real time.
     * You can monitor the health of the published audio and video streams in real time according to the quality parameters of the callback api, in order to show the uplink network status in real time on the device UI interface.
     * If you does not know how to use the parameters of this callback api, you can only pay attention to the level field of the quality parameter, which is a comprehensive value describing the uplink network calculated by SDK based on the quality parameters.
     *
     * @param streamID Stream ID
     * @param quality Published stream quality, including audio and video frame rate, bit rate, resolution, RTT, etc.
     */
    public void onPublisherQualityUpdate(String streamID, ZegoPublishStreamQuality quality){

    }

    /**
     * The callback triggered when the first audio frame is captured.
     *
     * After the startPublishingStream interface is called successfully, the SDK will receive this callback notification when it collects the first frame of audio data.
     * In the case of no startPublishingStream audio and video stream or preview, the first startPublishingStream audio and video stream or first preview, that is, when the engine of the audio and video module inside SDK starts, it will collect audio data of the local device and receive this callback.
     * Developers can use this callback to determine whether SDK has actually collected audio data. If the callback is not received, the audio capture device is occupied or abnormal.
     */
    public void onPublisherCapturedAudioFirstFrame(){

    }

    /**
     * The callback triggered when the state of relayed streaming to CDN changes.
     *
     * After the ZEGO real-time audio and video cloud relays the audio and video streams to the CDN, this callback will be received if the CDN relay status changes, such as a stop or a retry.
     * Developers can use this callback to determine whether the audio and video streams of the relay CDN are normal. If they are abnormal, further locate the cause of the abnormal audio and video streams of the relay CDN and make corresponding disaster recovery strategies.
     * If you do not understand the cause of the abnormality, you can contact ZEGO technicians to analyze the specific cause of the abnormality.
     *
     * @param streamID Stream ID
     * @param infoList List of information that the current CDN is relaying
     */
    public void onPublisherRelayCDNStateUpdate(String streamID, ArrayList<ZegoStreamRelayCDNInfo> infoList){

    }

    /**
     * The callback triggered when the state of stream playing changes.
     *
     * After publishing the stream successfully, the notification of the publish stream state change can be obtained through the callback interface.
     * You can roughly judge the user's downlink network status based on whether the state parameter is in [PLAY_REQUESTING].
     *
     * @param streamID stream ID
     * @param state Current play state
     * @param errorCode The error code corresponding to the status change of the playing stream. Please refer to the common error code documentation [https://doc-en.zego.im/en/308.html] for details.
     * @param extendedData Extended Information with state updates. As the standby, only an empty json table is currently returned
     */
    public void onPlayerStateUpdate(String streamID, ZegoPlayerState state, int errorCode, JSONObject extendedData){

    }

    /**
     * The callback triggered every 3 seconds to report the current stream playing quality.
     *
     * After calling the startPlayingStream successfully, this callback will be triggered every 3 seconds. The collection frame rate, bit rate, RTT, packet loss rate and other quality data  can be obtained, such the health of the publish stream can be monitored in real time.
     * You can monitor the health of the played audio and video streams in real time according to the quality parameters of the callback api, in order to show the downlink network status on the device UI interface in real time.
     * If you does not know how to use the various parameters of the callback api, you can only focus on the level field of the quality parameter, which is a comprehensive value describing the downlink network calculated by SDK based on the quality parameters.
     *
     * @param streamID Stream ID
     * @param quality Playing stream quality, including audio and video frame rate, bit rate, resolution, RTT, etc.
     */
    public void onPlayerQualityUpdate(String streamID, ZegoPlayStreamQuality quality){

    }

    /**
     * The callback triggered when a media event occurs during streaming playing.
     *
     * This callback is triggered when an event such as audio and video jamming and recovery occurs in the playing stream.
     * You can use this callback to make statistics on stutters or to make friendly displays in the UI interface of the app.
     *
     * @param streamID Stream ID
     * @param event Play media event callback
     */
    public void onPlayerMediaEvent(String streamID, ZegoPlayerMediaEvent event){

    }

    /**
     * The callback triggered when the first audio frame is received.
     *
     * After the startPlayingStream interface is called successfully, the SDK will receive this callback notification when it collects the first frame of audio data.
     *
     * @param streamID Stream ID
     */
    public void onPlayerRecvAudioFirstFrame(String streamID){

    }

    /**
     * The callback triggered when the state of relayed streaming of the mixed stream to CDN changes.
     *
     * In the general case of the ZEGO audio and video cloud mixing stream task, the output stream is published to the CDN using the rtmp protocol, and changes in the state during the publish will be notified from this callback api.
     *
     * @param taskID Mix stream task ID
     * @param infoList List of information that the current CDN is being mixed
     */
    public void onMixerRelayCDNStateUpdate(String taskID, ArrayList<ZegoStreamRelayCDNInfo> infoList){

    }

    /**
     * The callback triggered when the sound level of any input stream changes in the stream mixing process.
     *
     * You can use this callback to show the effect of the anchors sound level when the audience plays the mixed stream. So audience can notice which anchor is speaking.
     *
     * @param soundLevels Sound level hash map, key is the soundLevelID of every single stream in this mixer stream, value is the sound level value of that single stream, value ranging from 0.0 to 100.0
     */
    public void onMixerSoundLevelUpdate(HashMap<Integer, Float> soundLevels){

    }

    /**
     * The callback triggered every 100ms to report the sound level of the locally captured audio.
     *
     * Callback notification period is 100 ms'. To trigger this callback API, the [startSoundLevelMonitor] interface must be called to start the sound level monitor and you must be in a state where it is publishing the audio and video stream or be in [startPreview] state.
     *
     * @param soundLevel Locally captured sound level value, ranging from 0.0 to 100.0
     */
    public void onCapturedSoundLevelUpdate(float soundLevel){

    }

    /**
     * The callback triggered every 100ms to report the sound level of the remote stream.
     *
     * Callback notification period is 100 ms'. To trigger this callback API, the [startSoundLevelMonitor] interface must be called to start the sound level monitor and you must be in a state where it is playing the audio and video stream.
     *
     * @param soundLevels Remote sound level hash map, key is the streamID, value is the sound level value of the corresponding streamID, value ranging from 0.0 to 100.0
     */
    public void onRemoteSoundLevelUpdate(HashMap<String, Float> soundLevels){

    }

    /**
     * The callback triggered every 100ms to report the audio spectrum of the locally captured audio.
     *
     * Callback notification period is 100 ms'. To trigger this callback API, the [startAudioSpectrumMonitor] interface must be called to start the audio spectrum monitor and you must be in a state where it is publishing the audio and video stream or be in [startPreview] state.
     *
     * @param audioSpectrum Locally captured audio spectrum value list. Spectrum value range is [0-2^30]
     */
    public void onCapturedAudioSpectrumUpdate(float[] audioSpectrum){

    }

    /**
     * The callback triggered every 100ms to report the audio spectrum of the remote stream.
     *
     * Callback notification period is 100 ms'. To trigger this callback API, the [startAudioSpectrumMonitor] interface must be called to start the audio spectrum monitor and you must be in a state where it is playing the audio and video stream.
     *
     * @param audioSpectrums Remote audio spectrum hash map, key is the streamID, value is the audio spectrum list of the corresponding streamID. Spectrum value range is [0-2^30]
     */
    public void onRemoteAudioSpectrumUpdate(HashMap<String, float[]> audioSpectrums){

    }

    /**
     * The callback triggered when a device exception occurs.
     *
     * This callback is triggered when an exception occurs when reading or writing the audio and video device.
     *
     * @param errorCode The error code corresponding to the status change of the playing stream. Please refer to the common error code documentation [https://doc-en.zego.im/en/308.html] for details
     * @param deviceName device name
     */
    public void onDeviceError(int errorCode, String deviceName){

    }

    /**
     * The callback triggered when the state of the remote microphone changes.
     *
     * When the state of the remote microphone device is changed, such as switching a microphone, etc., by listening to the callback, it is possible to obtain an event related to the remote microphone, which can be used to prompt the user that the audio may be abnormal.
     * Developers of 1v1 education scenarios or education small class scenarios and similar scenarios can use this callback notification to determine whether the microphone device of the remote publishing stream device is working normally, and preliminary understand the cause of the device problem according to the corresponding state.
     *
     * @param streamID Stream ID
     * @param state Remote microphone status
     */
    public void onRemoteMicStateUpdate(String streamID, ZegoRemoteDeviceState state){

    }

    /**
     * The callback triggered when Broadcast Messages are received.
     *
     * @param roomID Room ID
     * @param messageList list of received messages.
     */
    public void onIMRecvBroadcastMessage(String roomID, ArrayList<ZegoBroadcastMessageInfo> messageList){

    }

    /**
     * The callback triggered when Barrage Messages are received.
     *
     * @param roomID Room ID
     * @param messageList list of received messages.
     */
    public void onIMRecvBarrageMessage(String roomID, ArrayList<ZegoBarrageMessageInfo> messageList){

    }

    /**
     * The callback triggered when a Custom Command is received.
     *
     * @param roomID Room ID
     * @param fromUser Sender of the command
     * @param command Command content received
     */
    public void onIMRecvCustomCommand(String roomID, ZegoUser fromUser, String command){

    }

}

