package im.zego.zegoexpress;

import im.zego.zegoexpress.constants.*;
import im.zego.zegoexpress.callback.*;
import im.zego.zegoexpress.entity.*;
import java.nio.*;
import java.util.*;
import org.json.*;
import im.zego.zegoexpress.*;
import android.app.Application;
import android.graphics.*;


public class ZegoExpressEngine {
    
    ZegoExpressEngine() {}
    
    /**
     * Creates a singleton instance of ZegoExpressEngine.
     *
     * The engine needs to be created and initialized before calling other APIs. The SDK only supports the creation of one instance of ZegoExpressEngine. Multiple calls to this interface return the same object.
     *
     * @param appID Application ID issued by ZEGO for developers, please apply from the ZEGO Admin Console https://console-express.zego.im/. The value ranges from 0 to 4294967295.
     * @param appSign Application signature for each AppID, please apply from the ZEGO Admin Console. Application signature is a 64 character string. Each character has a range of '0' ~ '9', 'a' ~ 'z'.
     * @param isTestEnv Choose to use a test environment or a formal commercial environment, the formal environment needs to submit work order configuration in the ZEGO management console. The test environment is for test development, with a limit of 30 rooms and 230 users. Official environment App is officially launched. ZEGO will provide corresponding server resources according to the configuration records submitted by the developer in the management console. The test environment and the official environment are two sets of environments and cannot be interconnected.
     * @param scenario The application scenario. Developers can choose one of ZegoScenario based on the scenario of the app they are developing, and the engine will preset a more general setting for specific scenarios based on the set scenario. After setting specific scenarios, developers can still call specific api to set specific parameters if they have customized parameter settings.
     * @param application Android Application Context
     * @param eventHandler Event notification callback. [null] means not receiving any callback notifications.It can also be managed later via [setEventHandler]
     * @return Engine singleton instance
     */
    public static ZegoExpressEngine createEngine(long appID, String appSign, boolean isTestEnv, ZegoScenario scenario, Application application, IZegoEventHandler eventHandler){
        return ZegoExpressEngineJni.createEngine(appID, appSign, isTestEnv, scenario, application, eventHandler);
    }

    /**
     * Destroys the singleton instance of ZegoExpressEngine asynchronously.
     *
     * Used to release resources used by ZegoExpressEngine.
     *
     * @param callback Notification callback for destroy engine completion. Developers can listen to this callback to ensure that device hardware resources are released. This callback is only used to notify the completion of the release of internal resources of the engine. Developers cannot release resources related to the engine within this callback. If the developer only uses SDK to implement audio and video functions, this parameter can be passed [null].
     */
    public static void destroyEngine(IZegoDestroyCompletionCallback callback){
        ZegoExpressEngineJni.destroyEngine(callback);
    }

    /**
     * Returns the singleton instance of ZegoExpressEngine.
     *
     * If the engine has not been created or has been destroyed, returns [null].
     *
     * @return Engine singleton instance
     */
    public static ZegoExpressEngine getEngine(){
        return ZegoExpressEngineJni.getEngine();
    }

    /**
     * Sets up the advanced engine configurations. This API will only take effect when called before creating the engine.
     *
     * Developers need to call this interface to set advanced function configuration when they need advanced functions of the engine.
     *
     * @param config Advanced engine configuration
     */
    public static void setEngineConfig(ZegoEngineConfig config){
        ZegoExpressEngineJni.setEngineConfig(config);
    }

    /**
     * Gets the SDK's version number.
     *
     * When the SDK is running, the developer finds that it does not match the expected situation and submits the problem and related logs to the ZEGO technical staff for locating. The ZEGO technical staff may need the information of the engine version to assist in locating the problem.
     * Developers can also collect this information as the version information of the engine used by the app, so that the SDK corresponding to each version of the app on the line.
     *
     * @return SDK version
     */
    public static String getVersion(){
        return ZegoExpressEngineJni.getVersion();
    }

    /**
     * Sets up the event notification callbacks that need to be handled. If the eventHandler is set to [null], all the callbacks set previously will be cleared.
     *
     * Invoke this function will **OVERRIDE** the handler set in [createEngine] or the handler set by the last call to this function.
     *
     * @param eventHandler Event notification callback. Developers should override callback related methods to focus on specific notifications based on their own business scenarios. The main callback methods of SDK are in [IZegoEventHandler].
     */
    public void setEventHandler(IZegoEventHandler eventHandler){
        ZegoExpressEngineJni.setEventHandler(eventHandler);
    }

    /**
     * Uploads logs to the ZEGO server.
     *
     * By default, SDK creates and prints log files in the app's default directory. Each log file defaults to a maximum of 5MB. Three log files are written over and over in a circular fashion. When calling this interface, SDK will auto package and upload the log files to the ZEGO server.
     * Developers can provide a business “feedback” channel in the app. When users feedback problems, they can call this interface to upload the local log information of SDK to help locate user problems.
     * The API is valid for the entire life cycle of the SDK.
     */
    public void uploadLog(){
        ZegoExpressEngineJni.uploadLog();
    }

    /**
     * Turns on/off verbose debugging and sets up the log language.
     *
     * The debug switch is set to on and the language is English by default.
     *
     * @param enable Detailed debugging information switch
     * @param language Debugging information language
     */
    public void setDebugVerbose(boolean enable, ZegoLanguage language){
        ZegoExpressEngineJni.setDebugVerbose(enable, language);
    }

    /**
     * Logs in to a room. You must log in to a room before publishing or playing streams.
     *
     * Different users who log in to the same room can get room related notifications in the same room (eg [onRoomUserUpdate], [onRoomStreamUpdate], etc.), and users in one room cannot receive room signaling notifications in another room.
     * Messages sent in one room (eg apis [setStreamExtraInfo], [sendBroadcastMessage], [sendBarrageMessage], [sendCustomCommand], etc.) cannot be received callback ((eg [onRoomStreamExtraInfoUpdate], [onIMRecvBroadcastMessage], [onIMRecvBarrageMessage], [onIMRecvCustomCommand], etc) in other rooms. Currently, SDK does not provide the ability to send messages across rooms. Developers can integrate the SDK of third-party IM to achieve.
     * SDK supports startPlayingStream audio and video streams from different rooms under the same appID, that is, startPlayingStream audio and video streams across rooms. Since ZegoExpressEngine's room related callback notifications are based on the same room, when developers want to startPlayingStream streams across rooms, developers need to maintain related messages and signaling notifications by themselves.
     * If the network is temporarily interrupted due to network quality reasons, the SDK will automatically reconnect internally. You can get the current connection status of the local room by listening to the [onRoomStateUpdate] callback method, and other users in the same room will receive [onRoomUserUpdate] callback notification.
     * It is strongly recommended that userID corresponds to the user ID of the business APP, that is, a userID and a real user are fixed and unique, and should not be passed to the SDK in a random userID. Because the unique and fixed userID allows ZEGO technicians to quickly locate online problems.
     *
     * @param roomID Room ID, a string of up to 128 bytes in length. Only support numbers, English characters and '~', '!', '@', '#', '$', '%', '^', '&', '*', '(', ')', '_', '+', '=', '-', '`', ';', '’', ',', '.', '<', '>', '/', '\'
     * @param user User object instance, configure userID, userName. Note that the userID needs to be globally unique with the same appID, otherwise the user who logs in later will kick out the user who logged in first.
     */
    public void loginRoom(String roomID, ZegoUser user){
        ZegoExpressEngineJni.loginRoom(roomID, user, null);
    }

    /**
     * Logs in to a room with advanced room configurations. You must log in to a room before publishing or playing streams.
     *
     * To prevent the app from being impersonated by a malicious user, you can add authentication before logging in to the room, that is, the [token] parameter in the ZegoRoomConfig object passed in by the [config] parameter.
     * Different users who log in to the same room can get room related notifications in the same room (eg [onRoomUserUpdate], [onRoomStreamUpdate], etc.), and users in one room cannot receive room signaling notifications in another room.
     * Messages sent in one room (eg apis [setStreamExtraInfo], [sendBroadcastMessage], [sendBarrageMessage], [sendCustomCommand], etc.) cannot be received callback ((eg [onRoomStreamExtraInfoUpdate], [onIMRecvBroadcastMessage], [onIMRecvBarrageMessage], [onIMRecvCustomCommand], etc) in other rooms. Currently, SDK does not provide the ability to send messages across rooms. Developers can integrate the SDK of third-party IM to achieve.
     * SDK supports startPlayingStream audio and video streams from different rooms under the same appID, that is, startPlayingStream audio and video streams across rooms. Since ZegoExpressEngine's room related callback notifications are based on the same room, when developers want to startPlayingStream streams across rooms, developers need to maintain related messages and signaling notifications by themselves.
     * If the network is temporarily interrupted due to network quality reasons, the SDK will automatically reconnect internally. You can get the current connection status of the local room by listening to the [onRoomStateUpdate] callback method, and other users in the same room will receive [onRoomUserUpdate] callback notification.
     * It is strongly recommended that userID corresponds to the user ID of the business APP, that is, a userID and a real user are fixed and unique, and should not be passed to the SDK in a random userID. Because the unique and fixed userID allows ZEGO technicians to quickly locate online problems.
     *
     * @param roomID Room ID, a string of up to 128 bytes in length. Only support numbers, English characters and '~', '!', '@', '#', '$', '%', '^', '&', '*', '(', ')', '_', '+', '=', '-', '`', ';', '’', ',', '.', '<', '>', '/', '\'
     * @param user User object instance, configure userID, userName. Note that the userID needs to be globally unique with the same appID, otherwise the user who logs in later will kick out the user who logged in first.
     * @param config Advanced room configuration
     */
    public void loginRoom(String roomID, ZegoUser user, ZegoRoomConfig config){
        ZegoExpressEngineJni.loginRoom(roomID, user, config);
    }

    /**
     * Logs out of a room.
     *
     * Exiting the room will stop all publishing and playing streams for user, and then SDK will auto stop local preview UI. After calling this interface, you will receive [onRoomStateUpdate] callback notification successfully exits the room, while other users in the same room will receive the [onRoomUserUpdate] callback notification(On the premise of enabling isUserStatusNotify configuration).
     *
     * @param roomID Room ID, a string of up to 128 bytes in length. Only support numbers, English characters and '~', '!', '@', '#', '$', '%', '^', '&', '*', '(', ')', '_', '+', '=', '-', '`', ';', '’', ',', '.', '<', '>', '/', '\'
     */
    public void logoutRoom(String roomID){
        ZegoExpressEngineJni.logoutRoom(roomID);
    }

    /**
     * @deprecated This API is invalid in this SDK package and should not be used. If you need to use this API, please contact ZEGO technical support.
     */
    @Deprecated
    public void sendReliableMessage(String roomID, String type, String content, int latestSeq, IZegoRoomSendReliableMessageCallback callback){
        ZegoExpressEngineJni.sendReliableMessage(roomID, type, content, latestSeq, callback);
    }

    /**
     * @deprecated This API is invalid in this SDK package and should not be used. If you need to use this API, please contact ZEGO technical support.
     */
    @Deprecated
    public void getReliableMessage(String roomID, String type, IZegoRoomGetReliableMessageCallback callback){
        ZegoExpressEngineJni.getReliableMessage(roomID, type, callback);
    }

    /**
     * Starts publishing a stream.
     *
     * This interface allows users to publish their local audio and video streams to the ZEGO real-time audio and video cloud. Other users in the same room can use the streamID to play the audio and video streams for intercommunication.
     * Before you start to publish the stream, you need to join the room first by calling [loginRoom]. Other users in the same room can get the streamID by monitoring the [onRoomStreamUpdate] event callback after the local user publishing stream successfully.
     * In the case of poor network quality, user publish may be interrupted, and the SDK will attempt to reconnect. You can learn about the current state and error information of the stream published by monitoring the [onPublisherStateUpdate] event.
     *
     * @param streamID Stream ID, a string of up to 256 characters, needs to be globally unique within the entire AppID. If in the same AppID, different users publish each stream and the stream ID is the same, which will cause the user to publish the stream failure. You cannot include URL keywords, otherwise publishing stream and playing stream will fails. Only support numbers, English characters and '~', '!', '@', '#', '$', '%', '^', '&', '*', '(', ')', '_', '+', '=', '-', '`', ';', '’', ',', '.', '<', '>', '/', '\'.
     */
    public void startPublishingStream(String streamID){
        ZegoExpressEngineJni.startPublishingStream(streamID, ZegoPublishChannel.MAIN);
    }

    /**
     * Starts publishing a stream (for the specified channel). You can call this API to publish a second stream.
     *
     * This interface allows users to publish their local audio and video streams to the ZEGO real-time audio and video cloud. Other users in the same room can use the streamID to play the audio and video streams for intercommunication.
     * Before you start to publish the stream, you need to join the room first by calling [loginRoom]. Other users in the same room can get the streamID by monitoring the [onRoomStreamUpdate] event callback after the local user publishing stream successfully.
     * In the case of poor network quality, user publish may be interrupted, and the SDK will attempt to reconnect. You can learn about the current state and error information of the stream published by monitoring the [onPublisherStateUpdate] event.
     *
     * @param streamID Stream ID, a string of up to 256 characters, needs to be globally unique within the entire AppID. If in the same AppID, different users publish each stream and the stream ID is the same, which will cause the user to publish the stream failure. You cannot include URL keywords, otherwise publishing stream and playing stream will fails. Only support numbers, English characters and '~', '!', '@', '#', '$', '%', '^', '&', '*', '(', ')', '_', '+', '=', '-', '`', ';', '’', ',', '.', '<', '>', '/', '\'.
     * @param channel Publish stream channel
     */
    public void startPublishingStream(String streamID, ZegoPublishChannel channel){
        ZegoExpressEngineJni.startPublishingStream(streamID, channel);
    }

    /**
     * Stops publishing a stream.
     *
     * This interface allows the user to stop sending local audio and video streams and end the call.
     * If the user has initiated publish flow, this interface must be called to stop the publish of the current stream before publishing the new stream (new streamID), otherwise the new stream publish will return a failure.
     * After stopping streaming, the developer should stop the local preview based on whether the business situation requires it.
     */
    public void stopPublishingStream(){
        ZegoExpressEngineJni.stopPublishingStream(ZegoPublishChannel.MAIN);
    }

    /**
     * Stops publishing a stream (for the specified channel).
     *
     * This interface allows the user to stop sending local audio and video streams and end the call.
     * If the user has initiated publish flow, this interface must be called to stop the publish of the current stream before publishing the new stream (new streamID), otherwise the new stream publish will return a failure.
     * After stopping streaming, the developer should stop the local preview based on whether the business situation requires it.
     * Use this API to stop publishing stream of aux channel.
     *
     * @param channel Publish stream channel
     */
    public void stopPublishingStream(ZegoPublishChannel channel){
        ZegoExpressEngineJni.stopPublishingStream(channel);
    }

    /**
     * Sets the extra information of the stream being published.
     *
     * User this interface to set the extra info of the stream, the result will be notified via the [IZegoPublisherSetStreamExtraInfoCallback].
     * The stream extra information is an extra information identifier of the stream ID. Unlike the stream ID, which cannot be modified during the publishing process, the stream extra information can be modified midway through the stream corresponding to the stream ID.
     * Developers can synchronize variable content related to stream IDs based on stream additional information.
     *
     * @param extraInfo Stream extra information, a string of up to 1024 characters.
     * @param callback Set stream extra information execution result notification
     */
    public void setStreamExtraInfo(String extraInfo, IZegoPublisherSetStreamExtraInfoCallback callback){
        ZegoExpressEngineJni.setStreamExtraInfo(extraInfo, ZegoPublishChannel.MAIN, callback);
    }

    /**
     * Sets the extra information of the stream being published (for the specified channel).
     *
     * User this interface to set the extra info of the stream, the result will be notified via the IZegoPublisherSetStreamExtraInfoCallback.
     *
     * @param extraInfo Stream extra information, a string of up to 1024 characters.
     * @param channel Publish stream channel
     * @param callback Set stream extra information execution result notification
     */
    public void setStreamExtraInfo(String extraInfo, ZegoPublishChannel channel, IZegoPublisherSetStreamExtraInfoCallback callback){
        ZegoExpressEngineJni.setStreamExtraInfo(extraInfo, channel, callback);
    }

    /**
     * Starts the local audio preview. Call this API after creating the engine and before publishing/playing a stream, and then the engine will start to capture audio.
     *
     * Since this api can be called before loginRoom and startPublishingStream, developers can check whether the audio equipment works ok by calling this api and by the soundlevel function of Engine.
     * When this api is called, the audio and video engine module inside SDK will start, and it will start to try to collect audio. Developers can pass [null] to the canvas parameter to cooperate with ZegoExpressEngine's sound wave function in order to achieve the purpose of detecting whether the audio equipment works normally before logging into the room.
     */
    public void startPreview(){
        ZegoExpressEngineJni.startPreview(null, ZegoPublishChannel.MAIN);
    }

    /**
     * Stops the local video preview.
     *
     * This api can be called to stop previewing when there is no need to see the preview locally.
     */
    public void stopPreview(){
        ZegoExpressEngineJni.stopPreview(ZegoPublishChannel.MAIN);
    }

    /**
     * Sets up the audio configurations.
     *
     * You can set the combined value of the audio codec, bit rate, and audio channel through this interface. If this interface is not called, the default is standard quality mode. Should be used before publishing.
     * If the preset value cannot meet the developer's scenario, the developer can set the parameters according to the business requirements.
     *
     * @param config Audio config
     */
    public void setAudioConfig(ZegoAudioConfig config){
        ZegoExpressEngineJni.setAudioConfig(config);
    }

    /**
     * Gets the current audio configurations.
     *
     * You can get the current audio codec, bit rate, and audio channel through this interface.
     *
     * @return Audio config
     */
    public ZegoAudioConfig getAudioConfig(){
        return ZegoExpressEngineJni.getAudioConfig();
    }

    /**
     * Stops or resumes sending the audio part of a stream.
     *
     * This interface can be called when publishing the stream to publish only the video stream without publishing the audio. The SDK still collects and processes the audio, but does not send the audio data to the network. It can be set before publishing.
     * If you stop sending audio streams, the remote user that play stream of local user publishing stream can receive `Mute` status change notification by monitoring [onRemoteMicStateUpdate] callbacks,
     *
     * @param mute Whether to stop sending audio streams, true means that only the video stream is sent without sending the audio stream, and false means that the audio and video streams are sent simultaneously. The default is false.
     */
    public void mutePublishStreamAudio(boolean mute){
        ZegoExpressEngineJni.mutePublishStreamAudio(mute, ZegoPublishChannel.MAIN);
    }

    /**
     * Stops or resumes sending the audio part of a stream (for the specified channel).
     *
     * This interface can be called when publishing the stream to publish only the video stream without publishing the audio. The SDK still collects and processes the audio, but does not send the audio data to the network. It can be set before publishing.
     * If you stop sending audio streams, the remote user that play stream of local user publishing stream can receive `Mute` status change notification by monitoring [onRemoteMicStateUpdate] callbacks,
     *
     * @param mute Whether to stop sending audio streams, true means that only the video stream is sent without sending the audio stream, and false means that the audio and video streams are sent simultaneously. The default is false.
     * @param channel Publish stream channel
     */
    public void mutePublishStreamAudio(boolean mute, ZegoPublishChannel channel){
        ZegoExpressEngineJni.mutePublishStreamAudio(mute, channel);
    }

    /**
     * Enables or disables traffic control.
     *
     * Traffic control enables SDK to dynamically adjust the bitrate of audio and video streaming according to its own and peer current network environment status.
     * Automatically adapt to the current network environment and fluctuations, so as to ensure the smooth publishing of stream.
     *
     * @param enable Whether to enable traffic control. The default is ture.
     * @param property Adjustable property of traffic control, bitmask format. Should be one or the combinations of [ZegoTrafficControlProperty] enumeration. [AdaptiveFPS] as default.
     */
    public void enableTrafficControl(boolean enable, int property){
        ZegoExpressEngineJni.enableTrafficControl(enable, property);
    }

    /**
     * Sets the audio recording volume for stream publishing.
     *
     * This interface is used to set the audio collection volume. The local user can control the volume of the audio stream sent to the far end. It can be set before publishing.
     *
     * @param volume Volume percentage. The range is 0 to 100. Default value is 100.
     */
    public void setCaptureVolume(int volume){
        ZegoExpressEngineJni.setCaptureVolume(volume);
    }

    /**
     * Adds a target CDN URL to which the stream will be relayed from ZEGO's cloud streaming server.
     *
     * You can call this api to publish the audio and video streams that have been published to the ZEGO real-time audio and video cloud to a custom CDN content distribution network that has high latency but supports high concurrent playing stream.
     * Because this called api is essentially a dynamic relay of the audio and video streams published to the ZEGO audio and video cloud to different CDNs, this api needs to be called after the audio and video stream is published to ZEGO real-time cloud successfully.
     * Since ZEGO's audio and video cloud service itself can be configured to support CDN(content distribution networks), this api is mainly used by developers who have CDN content distribution services themselves.
     * You can use ZEGO's CDN audio and video streaming content distribution service at the same time by calling this interface and then use the developer who owns the CDN content distribution service.
     * This interface supports dynamic relay to the CDN content distribution network, so developers can use this api as a disaster recovery solution for CDN content distribution services.
     * When the [enablePublishDirectToCDN] api is set to true to publish the stream straight to the CDN, then calling this interface will have no effect.
     *
     * @param streamID Stream ID
     * @param targetURL CDN relay address, supported address format is rtmp.
     * @param callback The execution result notification of the relay CDN operation, and proceed to the next step according to the execution result.
     */
    public void addPublishCdnUrl(String streamID, String targetURL, IZegoPublisherUpdateCdnUrlCallback callback){
        ZegoExpressEngineJni.addPublishCdnUrl(streamID, targetURL, callback);
    }

    /**
     * Deletes the specified CDN URL, which is used for relaying streams from ZEGO's cloud streaming server to CDN.
     *
     * This api is called when a CDN relayed address has been added and needs to stop propagating the stream to the CDN.
     * This api does not stop publishing audio and video stream to the ZEGO audio and video cloud.
     *
     * @param streamID Stream ID
     * @param targetURL CDN relay address, supported address format rtmp.
     * @param callback Remove CDN relay result notifications
     */
    public void removePublishCdnUrl(String streamID, String targetURL, IZegoPublisherUpdateCdnUrlCallback callback){
        ZegoExpressEngineJni.removePublishCdnUrl(streamID, targetURL, callback);
    }

    /**
     * Whether to publish streams directly from the client to CDN without passing through ZEGO's cloud streaming server.
     *
     * This api needs to be set before start publishing stream.
     * After calling this api to publish the audio and video stream directly to the CDN, calling [addPublishCdnUrl] and [removePublishCdnUrl] to dynamically repost to the CDN no longer takes effect, because these two API relay or stop the audio and video stream from the ZEGO real-time audio and video cloud If it is published to CDN, if the direct audio and video stream is directly published to the CDN, the audio and video stream cannot be dynamically relay to the CDN through the ZEGO real-time audio and video cloud.
     *
     * @param enable Whether to enable direct publish CDN, true: enable direct publish CDN, false: disable direct publish CDN
     * @param config CDN configuration, if null, use Zego's background default configuration
     */
    public void enablePublishDirectToCDN(boolean enable, ZegoCDNConfig config){
        ZegoExpressEngineJni.enablePublishDirectToCDN(enable, config, ZegoPublishChannel.MAIN);
    }

    /**
     * Whether to publish streams directly from the client to CDN without passing through ZEGO's cloud streaming server (for the specified channel).
     *
     * This api needs to be set before start publishing stream.
     * After calling this api to publish the audio and video stream directly to the CDN, calling [addPublishCdnUrl] and [removePublishCdnUrl] to dynamically repost to the CDN no longer takes effect, because these two API relay or stop the audio and video stream from the ZEGO real-time audio and video cloud If it is published to CDN, if the direct audio and video stream is directly published to the CDN, the audio and video stream cannot be dynamically relay to the CDN through the ZEGO real-time audio and video cloud.
     *
     * @param enable Whether to enable direct publish CDN, true: enable direct publish CDN, false: disable direct publish CDN
     * @param config CDN configuration, if null, use Zego's background default configuration
     * @param channel Publish stream channel
     */
    public void enablePublishDirectToCDN(boolean enable, ZegoCDNConfig config, ZegoPublishChannel channel){
        ZegoExpressEngineJni.enablePublishDirectToCDN(enable, config, channel);
    }

    /**
     * Enables or disables hardware encoding.
     *
     * Whether to use the hardware encoding function when publishing the stream, the GPU is used to encode the stream and to reduce the CPU usage. The setting can take effect before the stream published. If it is set after the stream published, the stream should be stopped first before it takes effect.
     * Because hard-coded support is not particularly good for a few models, SDK uses software encoding by default. If the developer finds that the device is hot when publishing a high-resolution audio and video stream during testing of some models, you can consider calling this interface to enable hard coding.
     *
     * @param enable Whether to enable hardware encoding, true: enable hardware encoding, false: disable hardware encoding
     */
    public void enableHardwareEncoder(boolean enable){
        ZegoExpressEngineJni.enableHardwareEncoder(enable);
    }

    /**
     * Starts playing a stream from ZEGO's streaming cloud (for audio streaming).
     *
     * This interface allows users to play audio and video streams from the ZEGO real-time audio and video cloud.
     * Before starting to play the stream, you need to join the room first, you can get the new streamID in the room by listening to the [onRoomStreamUpdate] event callback.
     * In the case of poor network quality, user play may be interrupted, the SDK will try to reconnect, and the current play status and error information can be obtained by listening to the [onPlayerStateUpdate] event.
     * Playing the stream ID that does not exist, the SDK continues to try to play after executing this interface. After the stream ID is successfully published, the audio and video stream can be actually played.
     * The developer can update the player canvas by calling this interface again(the streamID must be the same).
     *
     * @param streamID Stream ID, a string of up to 256 characters. You cannot include URL keywords, otherwise publishing stream and playing stream will fails. Only support numbers, English characters and '~', '!', '@', '#', '$', '%', '^', '&', '*', '(', ')', '_', '+', '=', '-', '`', ';', '’', ',', '.', '<', '>', '/', '\'.
     */
    public void startPlayingStream(String streamID){
        ZegoExpressEngineJni.startPlayingStream(streamID, null, null);
    }

    /**
     * Starts playing a stream from ZEGO's streaming cloud or from third-party CDN (for audio streaming).
     *
     * This interface allows users to play audio and video streams both from the ZEGO real-time audio and video cloud and from third-party cdn.
     * Before starting to play the stream, you need to join the room first, you can get the new streamID in the room by listening to the [onRoomStreamUpdate] event callback.
     * In the case of poor network quality, user play may be interrupted, the SDK will try to reconnect, and the current play status and error information can be obtained by listening to the [onPlayerStateUpdate] event.
     * Playing the stream ID that does not exist, the SDK continues to try to play after executing this interface. After the stream ID is successfully published, the audio and video stream can be actually played.
     * The developer can update the player canvas by calling this interface again(the streamID must be the same).
     *
     * @param streamID Stream ID, a string of up to 256 characters. You cannot include URL keywords, otherwise publishing stream and playing stream will fails. Only support numbers, English characters and '~', '!', '@', '#', '$', '%', '^', '&', '*', '(', ')', '_', '+', '=', '-', '`', ';', '’', ',', '.', '<', '>', '/', '\'.
     * @param config Advanced player configuration
     */
    public void startPlayingStream(String streamID, ZegoPlayerConfig config){
        ZegoExpressEngineJni.startPlayingStream(streamID, null, config);
    }

    /**
     * Stops playing a stream.
     *
     * This interface allows the user to stop playing the stream. When stopped, the attributes set for this stream previously, such as [setPlayVolume], [mutePlayStreamAudio], [mutePlayStreamVideo], etc., will be invalid and need to be reset when playing the the stream next time.
     *
     * @param streamID Stream ID
     */
    public void stopPlayingStream(String streamID){
        ZegoExpressEngineJni.stopPlayingStream(streamID);
    }

    /**
     * Sets the stream playback volume.
     *
     * This interface is used to set the playback volume of the stream. Need to be called after calling startPlayingStream.
     * You need to reset after [stopPlayingStream] and [startPlayingStream].
     *
     * @param streamID Stream ID
     * @param volume Volume percentage. The value ranges from 0 to 100, and the default value is 100.
     */
    public void setPlayVolume(String streamID, int volume){
        ZegoExpressEngineJni.setPlayVolume(streamID, volume);
    }

    /**
     * Stops or resumes playing the audio part of a stream.
     *
     * This api can be used to stop playing/retrieving the audio data of the stream. Need to be called after calling startPlayingStream.
     *
     * @param streamID Stream ID
     * @param mute mute flag, true: mute play stream video, false: resume play stream video
     */
    public void mutePlayStreamAudio(String streamID, boolean mute){
        ZegoExpressEngineJni.mutePlayStreamAudio(streamID, mute);
    }

    /**
     * Enables or disables hardware decoding.
     *
     * Turn on hardware decoding and use hardware to improve decoding efficiency. Need to be called before calling startPlayingStream.
     * Because hard-decoded support is not particularly good for a few models, SDK uses software decoding by default. If the developer finds that the device is hot when playing a high-resolution audio and video stream during testing of some models, you can consider calling this interface to enable hard decoding.
     *
     * @param enable Whether to turn on hardware decoding switch, true: enable hardware decoding, false: disable hardware decoding. The default is false
     */
    public void enableHardwareDecoder(boolean enable){
        ZegoExpressEngineJni.enableHardwareDecoder(enable);
    }

    /**
     * Starts a stream mixing task.
     *
     * Due to the performance considerations of the client device, ZegoExpressEngine's mix stream is to start the mixing stream task on the server side of the ZEGO audio and video cloud for mixing stream.
     * After calling this api, SDK initiates a mixing stream request to the ZEGO audio and video cloud. The ZEGO audio and video cloud will find the current publishing stream and perform video layer blending according to the parameters of the mixing stream task requested by ZegoExpressEngine.
     * When you need to update the mixing stream task, that is, the input stream list needs to be updated when the input stream increases or decreases, you can update the field of the [ZegoMixerTask] object inputList and call this api again to pass the same [ZegoMixerTask] object to update the mixing stream task.
     * If an exception occurs when requesting to start the mixing stream task, for example, the most common mix input stream does not exist, it will be given from the callback error code. For specific error codes, please refer to the common error code documentation [https://doc-en.zego.im/en/308.html].
     * If an input stream does not exist in the middle, the mixing stream task will automatically retry playing the input stream for 90 seconds, and will not retry after 90 seconds.
     *
     * @param task Mix stream task object
     * @param callback Start mix stream task result callback notification
     */
    public void startMixerTask(ZegoMixerTask task, IZegoMixerStartCallback callback){
        ZegoExpressEngineJni.startMixerTask(task, callback);
    }

    /**
     * Stops a stream mixing task.
     *
     * Similar to [startMixerTask], after calling this api, SDK initiates a request to end the mixing stream task to the ZEGO audio and video cloud server.
     * If you starts the next mixing stream task without stopping the previous mixing stream task, the previous mixing stream task will not stop automatically. The previous mixing stream task will not be stopped automatically until 90 seconds after the input stream of the previous mixing stream task does not exist.
     * When using the mixing stream function of the ZEGO audio and video cloud service, you should pay attention to the start of the next mixing stream task, and should stop the previous mixing stream task, so as not to cause the anchor has started the next streaming task and mixing with other anchors, and the audience is still playing the output stream of the previous mixing stream task.
     *
     * @param task Mix stream task object
     * @param callback Stop mix stream task result callback notification
     */
    public void stopMixerTask(ZegoMixerTask task, IZegoMixerStopCallback callback){
        ZegoExpressEngineJni.stopMixerTask(task, callback);
    }

    /**
     * Mutes or unmutes the microphone.
     *
     * This api is used to control whether the collected audio data is used. When the microphone is muted (disabled), the data is collected and discarded, and the microphone is still occupied.
     * The microphone is still occupied because closing or opening the microphone on the hardware is a relatively heavy operation, and real users may have frequent operations. For trade-off reasons, this api simply discards the collected data.
     * If you really want SDK to give up occupy the microphone, you can call the [enableAudioCaptureDevice] interface.
     * Developers who want to control whether to use microphone on the UI should use this interface to avoid unnecessary performance overhead by using the [enableAudioCaptureDevice].
     *
     * @param mute Whether to mute (disable) the microphone, true: mute (disable) microphone, false: enable microphone. The default is false.
     */
    public void muteMicrophone(boolean mute){
        ZegoExpressEngineJni.muteMicrophone(mute);
    }

    /**
     * Checks whether the microphone is muted.
     *
     * Can be used with [muteMicrophone], determine whether the microphone is muted.
     *
     * @return Whether the microphone is muted; true: the microphone is muted; false: the microphone is enable (not muted)
     */
    public boolean isMicrophoneMuted(){
        return ZegoExpressEngineJni.isMicrophoneMuted();
    }

    /**
     * Mutes or unmutes the audio output speaker.
     *
     * After mute speaker, all the SDK sounds will not play, including playing stream, mediaplayer, etc. But the SDK will still occupy the output device.
     *
     * @param mute Whether to mute (disable) speaker audio output, true: mute (disable) speaker audio output, false: enable speaker audio output. The default value is false
     */
    public void muteSpeaker(boolean mute){
        ZegoExpressEngineJni.muteSpeaker(mute);
    }

    /**
     * Checks whether the audio output speaker is muted.
     *
     * Can be used with [muteSpeaker], determine whether the speaker audio output is muted.
     *
     * @return Whether the speaker is muted; true: the speaker is muted; false: the speaker is enable (not muted)
     */
    public boolean isSpeakerMuted(){
        return ZegoExpressEngineJni.isSpeakerMuted();
    }

    /**
     * Enables or disables the audio capture device.
     *
     * This api is used to control whether to release the audio collection device. When the audio collection device is turned off, the SDK will no longer occupy the audio device. Of course, if the stream is being published at this time, there is no audio data.
     * Occupying the audio capture device and giving up Occupying the audio device is a relatively heavy operation, and the [muteMicrophone] interface is generally recommended.
     *
     * @param enable Whether to enable the audio capture device, true: disable audio capture device, false: enable audio capture device
     */
    public void enableAudioCaptureDevice(boolean enable){
        ZegoExpressEngineJni.enableAudioCaptureDevice(enable);
    }

    /**
     * Whether to use the built-in speaker to play audio.
     *
     * When you choose not to use the built-in speaker to play sound, that is, set to false, the SDK will select the currently highest priority audio output device to play the sound according to the system schedule
     *
     * @param enable Whether to use the built-in speaker to play sound, true: use the built-in speaker to play sound, false: use the highest priority audio output device scheduled by the current system to play sound
     */
    public void setBuiltInSpeakerOn(boolean enable){
        ZegoExpressEngineJni.setBuiltInSpeakerOn(enable);
    }

    /**
     * Starts sound level monitoring.
     *
     * After starting monitoring, you can receive local audio sound level via [onCapturedSoundLevelUpdate] callback, and receive remote audio sound level via [onRemoteSoundLevelUpdate] callback.
     * Before entering the room, you can call [startPreview] with this api and combine it with [onCapturedSoundLevelUpdate] callback to determine whether the audio device is working properly.
     * [onCapturedSoundLevelUpdate] and [onRemoteSoundLevelUpdate] callback notification period is 100 ms.
     */
    public void startSoundLevelMonitor(){
        ZegoExpressEngineJni.startSoundLevelMonitor();
    }

    /**
     * Stops sound level monitoring.
     *
     * After the monitoring is stopped, the callback of the local/remote audio sound level will be stopped.
     */
    public void stopSoundLevelMonitor(){
        ZegoExpressEngineJni.stopSoundLevelMonitor();
    }

    /**
     * Starts audio spectrum monitoring.
     *
     * After starting monitoring, you can receive local audio spectrum via [onCapturedAudioSpectrumUpdate] callback, and receive remote audio spectrum via [onRemoteAudioSpectrumUpdate] callback.
     * [onCapturedAudioSpectrumUpdate] and [onRemoteAudioSpectrumUpdate] callback notification period is 100 ms.
     */
    public void startAudioSpectrumMonitor(){
        ZegoExpressEngineJni.startAudioSpectrumMonitor();
    }

    /**
     * Stops audio spectrum monitoring.
     *
     * After the monitoring is stopped, the callback of the local/remote audio spectrum will be stopped.
     */
    public void stopAudioSpectrumMonitor(){
        ZegoExpressEngineJni.stopAudioSpectrumMonitor();
    }

    /**
     * Enables or disables headphone monitoring.
     *
     * enable/disable headphone monitor, this setting takes effect when the headset is connected.
     *
     * @param enable Whether to use headphone monitor, true: enable, false: disable
     */
    public void enableHeadphoneMonitor(boolean enable){
        ZegoExpressEngineJni.enableHeadphoneMonitor(enable);
    }

    /**
     * Sets the headphone monitor volume.
     *
     * set headphone monitor volume, this setting takes effect when the headset is connected.
     *
     * @param volume headphone monitor volume, range from 0 to 100, 80 as default
     */
    public void setHeadphoneMonitorVolume(int volume){
        ZegoExpressEngineJni.setHeadphoneMonitorVolume(volume);
    }

    /**
     * Enables or disables Acoustic Echo Cancellation (AEC).
     *
     * Turning on echo cancellation, the SDK filters the collected audio data to reduce the echo component in the audio. It needs to be set before starting the publish, and the setting is invalid after the start of the publish.
     *
     * @param enable Whether to enable echo cancellation, true: enable echo cancellation, false: disable echo cancellation
     */
    public void enableAEC(boolean enable){
        ZegoExpressEngineJni.enableAEC(enable);
    }

    /**
     * Whether to turn on Acoustic Echo Cancellation (AEC) when using the headphone.
     *
     * Needs to be invoked before startPublishingStream or startPreview to be valid.
     *
     * @param enable Whether to enable, true: enable, false: disable
     */
    public void enableHeadphoneAEC(boolean enable){
        ZegoExpressEngineJni.enableHeadphoneAEC(enable);
    }

    /**
     * Sets the Acoustic Echo Cancellation (AEC) mode.
     *
     * Switch different echo cancellation modes to control the extent to which echo data is eliminated. Need to be set before starting the publish.
     *
     * @param mode Echo cancellation mode
     */
    public void setAECMode(ZegoAECMode mode){
        ZegoExpressEngineJni.setAECMode(mode);
    }

    /**
     * Enables or disables Automatic Gain Control (AGC).
     *
     * When the auto gain is turned on, the sound will be amplified, but it will affect the sound quality to some extent. Need to be set before starting the publish.
     *
     * @param enable Whether to enable automatic gain control, true: enable AGC, false: disable AGC
     */
    public void enableAGC(boolean enable){
        ZegoExpressEngineJni.enableAGC(enable);
    }

    /**
     * Enables or disables Automatic Noise Suppression (ANS).
     *
     * Turning on the noise suppression switch can reduce the noise in the audio data and make the human voice clearer. Need to be set before starting the publish.
     *
     * @param enable Whether to enable noise suppression, true: enable AGC, false: disable AGC
     */
    public void enableANS(boolean enable){
        ZegoExpressEngineJni.enableANS(enable);
    }

    /**
     * Sets the Automatic Noise Suppression (ANS) mode.
     *
     * Default is medium mode
     *
     * @param mode Audio Noise Suppression mode
     */
    public void setANSMode(ZegoANSMode mode){
        ZegoExpressEngineJni.setANSMode(mode);
    }

    /**
     * Enables or disables audio mixing.
     *
     * Enable audio mixing, work with setAudioMixingHandler providing the audio data for mixing
     *
     * @param enable Whether to enable audio mixting, true: enable, false: disable
     */
    public void enableAudioMixing(boolean enable){
        ZegoExpressEngineJni.enableAudioMixing(enable);
    }

    /**
     * Sets up the audio mixing event handler.
     *
     * @param handler 混音回调
     */
    public void setAudioMixingHandler(IZegoAudioMixingHandler handler){
        ZegoExpressEngineJni.setAudioMixingHandler(handler);
    }

    /**
     * Stops or resumes playing the mixed audio locally.
     *
     * When stop play audio mixing locally, the audio will not be heard on the mix side (push side) while still be heard on the remote side (pull side).
     *
     * @param mute Whether to mute local audio mixting, true: mute, false: unmute
     */
    public void muteLocalAudioMixing(boolean mute){
        ZegoExpressEngineJni.muteLocalAudioMixing(mute);
    }

    /**
     * Sets the audio mixing output volume for both local playback and the stream to be published.
     *
     * This API will modify the volume of the local playback and the mixed data that is mixed into the push stream at the same time
     *
     * @param volume audio mixing volume, range from 0 to 100, 50 as default
     */
    public void setAudioMixingVolume(int volume){
        ZegoExpressEngineJni.setAudioMixingVolume(volume);
    }

    /**
     * Sets the audio mixing output volume for either local playback or the stream to published.
     *
     * This API can individually set the mixing volume of local playback or the mixing volume of the push stream
     *
     * @param volume audio mixing volume, range from 0 to 100, 50 as default
     * @param type volume local playback / volume in stream published
     */
    public void setAudioMixingVolume(int volume, ZegoVolumeType type){
        ZegoExpressEngineJni.setAudioMixingVolume(volume, type);
    }

    /**
     * Set the sound equalizer (EQ)
     *
     * @param bandIndex The value range is [0, 9], corresponding to 10 frequency bands, and the center frequencies are [31, 62, 125, 250, 500, 1K, 2K, 4K, 8K, 16K] Hz.
     * @param bandGain The value range is [-15, 15]. Default value is 0, if all gain values in all frequency bands are 0, EQ function will be disabled.
     */
    public void setAudioEqualizerGain(int bandIndex, float bandGain){
        ZegoExpressEngineJni.setAudioEqualizerGain(bandIndex, bandGain);
    }

    /**
     * Sets up the voice changer parameters.
     *
     * sound change effect only works on capture sounds.
     *
     * @param param Voice changer parameters
     */
    public void setVoiceChangerParam(ZegoVoiceChangerParam param){
        ZegoExpressEngineJni.setVoiceChangerParam(param);
    }

    /**
     * Sets up the reverberation parameters.
     *
     * Different values set dynamically after a successful publishing will take effect, When any of the parameters is set to 0.0, the reverb is turned off.
     *
     * @param param Reverb parameter
     */
    public void setReverbParam(ZegoReverbParam param){
        ZegoExpressEngineJni.setReverbParam(param);
    }

    /**
     * Enables the virtual stereo feature.
     *
     * Note: You need to set up a dual channel setAudioConfig for the virtual stereo to take effect!
     *
     * @param enable true to turn on the virtual stereo, false to turn off the virtual stereo
     * @param angle angle of the sound source in the virtual stereo, ranging from 0 to 180, with 90 being the front, and 0 and 180 being respectively Corresponds to rightmost and leftmost, usually defaults to 90.
     */
    public void enableVirtualStereo(boolean enable, int angle){
        ZegoExpressEngineJni.enableVirtualStereo(enable, angle);
    }

    /**
     * Sends a Broadcast Message, which will be delivered to the first 500 users in the room.
     *
     * The total sending frequency limit of [sendBroadcastMessage] and [sendCustomCommand] is 600 times per minute by default.
     * Users of up to the first 500 advanced rooms in the same room can receive it, which is generally used when the number of live broadcast rooms is less than 500.
     *
     * @param roomID Room ID, a string of up to 128 bytes in length. Only support numbers, English characters and '~', '!', '@', '#', '$', '%', '^', '&', '*', '(', ')', '_', '+', '=', '-', '`', ';', '’', ',', '.', '<', '>', '/', '\'
     * @param message Message content, no longer than 256 bytes
     * @param callback Send broadcast message result callback
     */
    public void sendBroadcastMessage(String roomID, String message, IZegoIMSendBroadcastMessageCallback callback){
        ZegoExpressEngineJni.sendBroadcastMessage(roomID, message, callback);
    }

    /**
     * Sends a Barrage Message (bullet screen) to all users in the same room, without guaranteeing the delivery.
     *
     * There is no limit on the number of transmissions, but the server will actively drop messages if it is sent too frequently.
     * It can be received by users with more than 500 people in the same room, but it is not reliable, that is, when there are many users in the room or messages are sent frequently between users, the users who receive the messages may not be able to receive them. Generally used for sending live barrage.
     *
     * @param roomID Room ID, a string of up to 128 bytes in length. Only support numbers, English characters and '~', '!', '@', '#', '$', '%', '^', '&', '*', '(', ')', '_', '+', '=', '-', '`', ';', '’', ',', '.', '<', '>', '/', '\'
     * @param message Message content, no longer than 256 bytes
     * @param callback Send barrage message result callback
     */
    public void sendBarrageMessage(String roomID, String message, IZegoIMSendBarrageMessageCallback callback){
        ZegoExpressEngineJni.sendBarrageMessage(roomID, message, callback);
    }

    /**
     * Sends a Custom Command to the specified users in the same room.
     *
     * The total sending frequency limit of [sendBroadcastMessage] and [sendCustomCommand] is 600 times per minute by default.
     * The type of point-to-point signaling in the same room is generally used for remote control signaling or message sending between users.
     *
     * @param roomID Room ID, a string of up to 128 bytes in length. Only support numbers, English characters and '~', '!', '@', '#', '$', '%', '^', '&', '*', '(', ')', '_', '+', '=', '-', '`', ';', '’', ',', '.', '<', '>', '/', '\'
     * @param command Custom command content, no longer than 256 bytes
     * @param toUserList The users who will receive the command
     * @param callback Send command result callback
     */
    public void sendCustomCommand(String roomID, String command, ArrayList<ZegoUser> toUserList, IZegoIMSendCustomCommandCallback callback){
        ZegoExpressEngineJni.sendCustomCommand(roomID, command, toUserList, callback);
    }

    /**
     * Creates a media player instance.
     *
     * Currently, a maximum of 4 instances can be created, after which it will return null. The more instances of a media player, the greater the performance overhead on the device.
     *
     * @return Media player instance, null will be returned when the maximum number is exceeded.
     */
    public ZegoMediaPlayer createMediaPlayer(){
        return ZegoMediaPlayerJni.createMediaPlayer();
    }

    /**
     * Destroys a media player instance.
     *
     * @param mediaPlayer The media player instance object to be destroyed
     */
    public void destroyMediaPlayer(ZegoMediaPlayer mediaPlayer){
        ZegoMediaPlayerJni.destroyMediaPlayer(mediaPlayer);
    }

    /**
     * Starts to record locally captured audio or video and directly save the data to a file.
     *
     * The recording process should not logoutRoom, if the logoutRoom SDK internally ends the current recording voluntarily.
     * Currently only one task can be recorded simultaneously.
     * The call to this api is effective only if the call to [startPreview] or [startPublishingStream] is successful.
     *
     * @param config Record config
     * @param channel Publishing stream channel
     */
    public void startRecordingCapturedData(ZegoDataRecordConfig config, ZegoPublishChannel channel){
        ZegoExpressEngineJni.startRecordingCapturedData(config, channel);
    }

    /**
     * Stops recording locally captured audio or video.
     *
     * @param channel Publishing stream channel
     */
    public void stopRecordingCapturedData(ZegoPublishChannel channel){
        ZegoExpressEngineJni.stopRecordingCapturedData(channel);
    }

    /**
     * Sets up the event callback handler for data recording.
     *
     * Implemente the functions of ZegoDataRecordEventHandler to get notified when recode state and process changed
     *
     * @param eventHandler Data record event handler
     */
    public void setDataRecordEventHandler(IZegoDataRecordEventHandler eventHandler){
        ZegoExpressEngineJni.setDataRecordEventHandler(eventHandler);
    }

    /**
     * @deprecated This API is invalid in this SDK package and should not be used. If you need to use this API, please contact ZEGO technical support.
     */
    @Deprecated
    public void enableAudioDataCallback(boolean enable, int callbackBitMask, ZegoAudioFrameParam param){
        ZegoExpressEngineJni.enableAudioDataCallback(enable, callbackBitMask, param);
    }

    /**
     * Sets up the event callback handler for receiving audio data.
     *
     * You can call this api to receive audio data from the SDK when you need to get the audio data of the remote user or get the data collected by the local microphone for other purposes (such as audio-only recording, audio-only third-party monitoring, and audio-only real-time analysis).
     * The set callback needs to be after the call to [enableAudioDataCallback] and in a push or pull stream state for the callback to work.
     * The format of the audio data callback by SDK is pcm. The SDK still controls the collection and playback of the sound device. This function is to copy a piece of data collected or played inside the SDK to the developer for use.
     *
     * @param handler Audio data handler for receive audio data
     */
    public void setAudioDataHandler(IZegoAudioDataHandler handler){
        ZegoExpressEngineJni.setAudioDataHandler(handler);
    }

    /**
     * Enables the custom audio I/O function.
     *
     * Needs to be invoked before [startPublishingStream] or [startPreview] to be valid.
     *
     * @param enable Whether to enable custom audio IO, default is false
     * @param config Custom audio IO config
     */
    public void enableCustomAudioIO(boolean enable, ZegoCustomAudioConfig config){
        ZegoExpressEngineJni.enableCustomAudioIO(enable, config, ZegoPublishChannel.MAIN);
    }

    /**
     * Enables the custom audio I/O function (for the specified channel).
     *
     * Needs to be invoked before [startPublishingStream] or [startPreview] to be valid.
     *
     * @param enable Whether to enable custom audio IO, default is false
     * @param config Custom audio IO config
     * @param channel Specify the publish channel to enable custom audio IO
     */
    public void enableCustomAudioIO(boolean enable, ZegoCustomAudioConfig config, ZegoPublishChannel channel){
        ZegoExpressEngineJni.enableCustomAudioIO(enable, config, channel);
    }

    /**
     * Sends AAC audio data produced by custom audio capture to the SDK.
     *
     * @param data AAC buffer data
     * @param dataLength The total length of the buffer data
     * @param configLength The length of AAC specific config (Note: The AAC encoded data length is 'encodedLength = dataLength - configLength')
     * @param referenceTimeMillisecond The timestamp of this AAC audio frame in millisecond
     * @param param The param of this AAC audio frame
     */
    public void sendCustomAudioCaptureAACData(ByteBuffer data, int dataLength, int configLength, long referenceTimeMillisecond, ZegoAudioFrameParam param){
        ZegoExpressEngineJni.sendCustomAudioCaptureAACData(data, dataLength, configLength, referenceTimeMillisecond, param, ZegoPublishChannel.MAIN);
    }

    /**
     * Sends AAC audio data produced by custom audio capture to the SDK (for the specified channel).
     *
     * @param data AAC buffer data
     * @param dataLength The total length of the buffer data
     * @param configLength The length of AAC specific config (Note: The AAC encoded data length is 'encodedLength = dataLength - configLength')
     * @param referenceTimeMillisecond The timestamp of this AAC audio frame in millisecond
     * @param param The param of this AAC audio frame
     * @param channel Publish channel for capturing audio frames
     */
    public void sendCustomAudioCaptureAACData(ByteBuffer data, int dataLength, int configLength, long referenceTimeMillisecond, ZegoAudioFrameParam param, ZegoPublishChannel channel){
        ZegoExpressEngineJni.sendCustomAudioCaptureAACData(data, dataLength, configLength, referenceTimeMillisecond, param, channel);
    }

    /**
     * Sends PCM audio data produced by custom audio capture to the SDK.
     *
     * @param data PCM buffer data
     * @param dataLength The total length of the buffer data
     * @param param The param of this PCM audio frame
     */
    public void sendCustomAudioCapturePCMData(ByteBuffer data, int dataLength, ZegoAudioFrameParam param){
        ZegoExpressEngineJni.sendCustomAudioCapturePCMData(data, dataLength, param, ZegoPublishChannel.MAIN);
    }

    /**
     * Sends PCM audio data produced by custom audio capture to the SDK (for the specified channel).
     *
     * @param data PCM buffer data
     * @param dataLength The total length of the buffer data
     * @param param The param of this PCM audio frame
     * @param channel Publish channel for capturing audio frames
     */
    public void sendCustomAudioCapturePCMData(ByteBuffer data, int dataLength, ZegoAudioFrameParam param, ZegoPublishChannel channel){
        ZegoExpressEngineJni.sendCustomAudioCapturePCMData(data, dataLength, param, channel);
    }

    /**
     * Fetches PCM audio data of the remote stream for custom audio rendering.
     *
     * It is recommended to use the system framework to periodically invoke this API to drive audio data rendering
     *
     * @param data A block of memory for storing audio PCM data that requires user to manage the memory block's lifecycle, the SDK will copy the audio frame rendering data to this memory block
     * @param dataLength The length of the audio data to be fetch this time (dataLength = duration * sampleRate * channels * 2(16 bit depth i.e. 2 Btye))
     * @param param Specify the parameters of the fetched audio frame
     */
    public void fetchCustomAudioRenderPCMData(ByteBuffer data, int dataLength, ZegoAudioFrameParam param){
        ZegoExpressEngineJni.fetchCustomAudioRenderPCMData(data, dataLength, param);
    }

    /**
     * Destroy engine singleton instance
     *
     * Used to release resources used by ZegoExpressEngine.
     *
     * @deprecated This interface is deprecated, please use the overloaded interface with the same name
     */
    @Deprecated
    public static void destroyEngine(){
        ZegoExpressEngineJni.destroyEngine(null);
    }

    /**
     * Turn on/off audio output to the device
     *
     * This interface is used to control whether the SDK needs to throw audio data to the device.
     *
     * @deprecated This interface is deprecated, please use [muteSpeaker] instead.
     * @param mute Whether to disable audio output to the device, true: disable audio output, false: enable audio output
     */
    @Deprecated
    public void muteAudioOutput(boolean mute){
        ZegoExpressEngineJni.muteAudioOutput(mute);
    }

}

