package hk.ids.gws.android.afragment.ui.fragmentmanage;

import android.annotation.SuppressLint;
import android.app.Fragment;
import android.app.FragmentManager;
import android.app.FragmentTransaction;
import android.os.Bundle;
import android.support.annotation.AnimatorRes;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.SparseArray;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.UUID;

import hk.ids.gws.android.afragment.R;
import hk.ids.gws.android.afragment.helper.AnotherFragment;
import hk.ids.gws.android.afragment.ui.activity.IRootActivity;
import hk.ids.gws.android.afragment.ui.fragment.IRootFragment;
import hk.ids.gws.android.afragment.utils.LogUtil;

@SuppressWarnings({"unused", "WeakerAccess", "deprecation", "unchecked"})
public class FragmentStore {
    protected final String INTERNALTAG = "AnotherFragment:" + getClass().getSimpleName();


    private final static String BUNDLE_TAG_FRAGMENT_TAG = "afragment:fragment.tag";
    private final static String BUNDLE_TAG_FRAGMENT_VISIBILITY = "afragment:fragment.visibility";

    private final static String BUNDLE_TAG_TAGMAP = "afragment:tagmap";
    private final static String BUNDLE_TAG_VISIBILITYMAP = "afragment:visibilitymap";
    private final static String BUNDLE_TAG_STACKS = "afragment:stacks";


    private final Object mLock = new Object();

    private IRootActivity mActivity;

    private HashMap<String, String> mTagMap = new HashMap<>();
    private HashMap<String, Boolean> mVisibilityMap = new HashMap<>();
    private SparseArray<ContainerInfo> mStacks = new SparseArray<>();

    private HashMap<String, Object> mFragmentInstances = new HashMap<>();



    public <T extends Fragment & IRootFragment> FragmentStore(@NonNull IRootActivity activity, @Nullable Bundle savedInstanceState) {
        mActivity = activity;

        // Restore state if any
        if (savedInstanceState != null) {
            Serializable tagMapSerializable = savedInstanceState.getSerializable(BUNDLE_TAG_TAGMAP);
            Serializable visibilityMapSerializable = savedInstanceState.getSerializable(BUNDLE_TAG_VISIBILITYMAP);
            SparseArray<ContainerInfo> sparseArray = savedInstanceState.getSparseParcelableArray(BUNDLE_TAG_STACKS);

            if (tagMapSerializable != null) {
                mTagMap = (HashMap<String, String>) tagMapSerializable;

                LogUtil.i(INTERNALTAG, "FragmentStore create from saved state, mTagMap restored");
            }
            if (visibilityMapSerializable != null) {
                mVisibilityMap = (HashMap<String, Boolean>) visibilityMapSerializable;

                LogUtil.i(INTERNALTAG, "FragmentStore create from saved state, mVisibilityMap restored");
            }
            if (sparseArray != null) {
                mStacks = sparseArray;

                LogUtil.i(INTERNALTAG, "FragmentStore create from saved state, mStacks restored");
            }

            // Restore instances
            List<String> tags = new ArrayList<>();
            for (int i = 0; i < mStacks.size(); i++) {
                ContainerInfo containerInfo = mStacks.get(mStacks.keyAt(i));
                tags.addAll(containerInfo.fragmentTags);
            }
            for (String tag : tags) {
                Fragment fragment = getFragmentManager().findFragmentByTag(tag);
                if (fragment != null && fragment instanceof IRootFragment) {
                    mFragmentInstances.put(tag, fragment);

                    if (!getVisibility(fragment)) {
                        hide((T) fragment);
                    }

                    LogUtil.i(INTERNALTAG, String.format("FragmentStore restored an instance. Tag: %s, Fragment: %s", tag, fragment.getClass().getSimpleName()));
                }
            }
        }
    }

    public Bundle getInstanceState() {
        Bundle bundle = new Bundle();
        bundle.putSerializable(BUNDLE_TAG_TAGMAP, mTagMap);
        bundle.putSerializable(BUNDLE_TAG_VISIBILITYMAP, mVisibilityMap);
        bundle.putSparseParcelableArray(BUNDLE_TAG_STACKS, mStacks);

        return bundle;
    }


    // Private methods

    private FragmentManager getFragmentManager() {
        return mActivity.getFragmentManager();
    }


    @Nullable
    private String getTopFragmentTag(int containerId) {
        return getTopNFragmentTag(containerId, 0);
    }

    @Nullable
    private String getTopNFragmentTag(int containerId, int n) {
        ContainerInfo containerInfo = mStacks.get(containerId);
        if (containerInfo == null || containerInfo.fragmentTags.size() <= n) return null;

        return containerInfo.fragmentTags.get(containerInfo.fragmentTags.size() - n - 1);
    }


    @Nullable
    private <T extends Fragment & IRootFragment> T getFragmentByTag(String internalTag) {
        return (T) mFragmentInstances.get(internalTag);
    }


    @NonNull
    private <T extends Fragment & IRootFragment> String assignTag(T fragment) {
        String uuid = getTag(fragment);

        // Assign only if needed
        if (uuid == null) {
            uuid = UUID.randomUUID().toString();

            Bundle bundle = fragment.getArguments();

            if (bundle == null) bundle = new Bundle();
            bundle.putString(BUNDLE_TAG_FRAGMENT_TAG, uuid);

            fragment.setArguments(bundle);

            LogUtil.i(INTERNALTAG, String.format(Locale.getDefault(),
                    "Assigned new tag to fragment. Fragment: %s, Tag: %s", fragment.getClass().getSimpleName(), uuid));
        }

        return uuid;
    }

    @Nullable
    private String getTag(Fragment fragment) {
        Bundle bundle = fragment.getArguments();
        if (bundle != null) {
            return bundle.getString(BUNDLE_TAG_FRAGMENT_TAG);
        }

        return null;
    }


    private boolean getVisibility(Fragment fragment) {
        String tag = getTag(fragment);
        return tag == null || (mVisibilityMap.get(tag) != null ? mVisibilityMap.get(tag) : true);
    }

    private void setVisibility(Fragment fragment, boolean visibility) {
        String tag = getTag(fragment);
        if (tag == null) return;

        mVisibilityMap.put(tag, visibility);
    }


    @SuppressLint("CommitTransaction")
    private <T extends Fragment & IRootFragment> List<FragmentTransaction> getFlushTransactions(int containerId, int overrideAnimation) {
        ContainerInfo containerInfo = mStacks.get(containerId);
        if (containerInfo != null && containerInfo.fragmentTags.size() > 0) {
            List<T> pending = new ArrayList<>();

            for (int i = 0; i < containerInfo.fragmentTags.size(); i++) {
                T fragment = getFragmentByTag(containerInfo.fragmentTags.get(i));
                if (fragment != null) pending.add(fragment);
            }

            FragmentTransaction aniTransaction = null;
            FragmentTransaction nonaniTransaction = null;

            for (int i = 0; i < pending.size(); i++) {
                T fragment = pending.get(i);

                if (i == pending.size() - 1) {
                    if (aniTransaction == null) {
                        aniTransaction = getFragmentManager().beginTransaction();

                        if ((fragment.getEnterAnimation() != -1 && fragment.getExitAnimation() != -1) || overrideAnimation != -1) {
                            aniTransaction.setCustomAnimations(fragment.getEnterAnimation(), (overrideAnimation == -1) ? fragment.getExitAnimation() : overrideAnimation);
                        }
                    }

                    aniTransaction.remove(fragment);
                } else {
                    if (nonaniTransaction == null) {
                        nonaniTransaction = getFragmentManager().beginTransaction();
                        nonaniTransaction.setCustomAnimations(R.animator.animator_fragment_no_ani, R.animator.animator_fragment_no_ani);
                    }

                    nonaniTransaction.remove(fragment);
                }
            }

            List<FragmentTransaction> fragmentTransactions = new ArrayList<>();
            if (nonaniTransaction != null) fragmentTransactions.add(nonaniTransaction);
            if (aniTransaction != null) fragmentTransactions.add(aniTransaction);

            return fragmentTransactions;
        }

        return null;
    }


    private <T extends Fragment & IRootFragment> void notifyTopmost(int containerId) {
        if (!AnotherFragment.getDefault().isNotifyTopmost()) return;

        ContainerInfo containerInfo = mStacks.get(containerId);
        if (containerInfo != null && containerInfo.fragmentTags.size() >= 0) {
            boolean foundVisible = false;
            for (int i = 0; i < containerInfo.fragmentTags.size(); i++) {
                T fragment = getFragmentByTag(containerInfo.fragmentTags.get(i));
                if (fragment == null) {
                    LogUtil.e(INTERNALTAG, String.format(Locale.getDefault(),
                            "Unexpected behaviour: getFragmentByTag return null. Index: %d, Container: %d", i, containerId));
                    continue;
                }

                if (!foundVisible && !fragment.isHidden()) {
                    foundVisible = true;

                    fragment.onTopmostChanged(true);
                } else {
                    fragment.onTopmostChanged(false);
                }
            }
        }
    }


    private void completeAppendTransaction(int containerId, IRootFragment fragment, String tag) {
        synchronized (mLock) {
            ContainerInfo containerInfo = mStacks.get(containerId);
            if (containerInfo == null) containerInfo = new ContainerInfo();

            containerInfo.fragmentTags.add(tag);

            mStacks.put(containerId, containerInfo);

            mFragmentInstances.put(tag, fragment);
        }
    }

    private void completeReplaceTransaction(int containerId, IRootFragment fragment, String tag) {
        synchronized (mLock) {
            ContainerInfo containerInfo = mStacks.get(containerId);
            if (containerInfo != null) {
                for (String current : containerInfo.fragmentTags) {
                    mFragmentInstances.remove(current);
                }
            }

            containerInfo = new ContainerInfo();
            containerInfo.fragmentTags.add(tag);

            mStacks.put(containerId, containerInfo);

            mFragmentInstances.put(tag, fragment);
        }
    }

    private void completeRemoveTransaction(int containerId, String tag) {
        synchronized (mLock) {
            ContainerInfo containerInfo = mStacks.get(containerId);
            if (containerInfo == null) containerInfo = new ContainerInfo();

            containerInfo.fragmentTags.remove(tag);

            mStacks.put(containerId, containerInfo);

            mFragmentInstances.remove(tag);
        }
    }

    private void completeFlushTransaction(int containerId) {
        synchronized (mLock) {
            ContainerInfo containerInfo = mStacks.get(containerId);
            if (containerInfo != null) {
                for (String current : containerInfo.fragmentTags) {
                    mFragmentInstances.remove(current);
                }
            }

            containerInfo = new ContainerInfo();
            mStacks.put(containerId, containerInfo);
        }
    }


    // Public methods

    /**
     * Get fragment count of given container
     * @param containerId Target container id
     */
    public int getCount(int containerId) {
        ContainerInfo containerInfo = mStacks.get(containerId);
        return containerInfo == null ? 0 : containerInfo.fragmentTags.size();
    }

    /**
     * Check if the given container is empty
     * @param containerId Target container id
     */
    public boolean isEmpty(int containerId) {
        ContainerInfo containerInfo = mStacks.get(containerId);
        return (containerInfo == null || containerInfo.fragmentTags.size() == 0);
    }

    /**
     * Check if given fragment is exist in given container
     * @param containerId Target container id
     * @param fragment Fragment instance to check
     */
    public <T extends Fragment & IRootFragment> boolean isExist(int containerId, T fragment) {
        String tag = getTag(fragment);
        if (tag != null) {
            ContainerInfo containerInfo = mStacks.get(containerId);
            if (containerInfo != null && containerInfo.fragmentTags.contains(tag)) {
                return true;
            }
        }

        return false;
    }


    /**
     * Get the top most fragment in given container
     * @param containerId Target container id
     */
    @Nullable
    public <T extends Fragment & IRootFragment> T getTopFragment(int containerId) {
        return getTopNFragment(containerId, 0);
    }

    /**
     * Get the fragment on specific index in given container
     * @param containerId Target container id
     * @param n z index of container
     */
    @Nullable
    public <T extends Fragment & IRootFragment> T getTopNFragment(int containerId, int n) {
        String internalTag = getTopNFragmentTag(containerId, n);
        if (internalTag == null) return null;

        return (T) mFragmentInstances.get(internalTag);
    }


    /**
     * Get the fragment by tag
     * @param tag Tag given on add or replace transaction
     */
    @Nullable
    public <T extends Fragment & IRootFragment> T findFragmentByTag(String tag) {
        String internalTag = mTagMap.get(tag);
        if (internalTag == null) return null;

        return (T) mFragmentInstances.get(internalTag);
    }


    /**
     * Flush the specific container by given id
     * @param containerId Target container id
     */
    public void flush(int containerId) {
        flush(containerId, -1);
    }

    /**
     * Flush the specific container by given id
     * @param containerId Target container id
     * @param overrideAnimation Specific the exit animation, it will only append to
     *                          topmost fragment, behide will remove without animation
     */
    public void flush(int containerId, int overrideAnimation) {
        List<FragmentTransaction> fragmentTransactions = getFlushTransactions(containerId, overrideAnimation);
        if (fragmentTransactions != null && fragmentTransactions.size() > 0) {
            try {
                for (FragmentTransaction fragmentTransaction : fragmentTransactions) {
                    fragmentTransaction.commit();
                }

                completeFlushTransaction(containerId);

                // Notify activity
                mActivity.onContainerChange(containerId);
            } catch (IllegalStateException e) {
                if (AnotherFragment.getDefault().isThrowStateLost()) {
                    throw e;
                } else {
                    LogUtil.e(INTERNALTAG, String.format(Locale.getDefault(),
                            "IllegalStateException detected, it may due to state lost. Exception: %s", e.toString()));
                }
            }
        }
    }


    /**
     * Remove the top most fragment in container
     * @param containerId Target container id
     */
    public boolean back(int containerId) {
        return back(containerId, false, -1);
    }

    /**
     * Remove the top most fragment in container
     * @param containerId Target container id
     * @param allowEmpty Allow back to empty, last fragment will not be remove if false
     */
    public boolean back(int containerId, boolean allowEmpty) {
        return back(containerId, allowEmpty, -1);
    }

    /**
     * Remove the top most fragment in container
     * @param containerId Target container id
     * @param allowEmpty Allow back to empty, last fragment will not be remove if false
     * @param overrideAnimation Specific the exit animation
     */
    public <T extends Fragment & IRootFragment> boolean back(int containerId, boolean allowEmpty, @AnimatorRes int overrideAnimation) {
        // Check if able to back
        int count = getCount(containerId);
        if (count > (allowEmpty ? 0 : 1)) {
            // Check if tag available
            String tag = getTopFragmentTag(containerId);
            if (tag != null) {
                // Check if fragment exist and can be back
                T fragment = getFragmentByTag(tag);
                if (fragment != null && fragment.canBack()) {
                    if (fragment.onBackPressed()) return true;

                    FragmentTransaction fragmentTransaction = getFragmentManager().beginTransaction();

                    if ((fragment.getEnterAnimation() != -1 && fragment.getExitAnimation() != -1) || overrideAnimation != -1) {
                        fragmentTransaction.setCustomAnimations(fragment.getEnterAnimation(), (overrideAnimation == -1) ? fragment.getExitAnimation() : overrideAnimation);
                    }

                    fragmentTransaction.remove(fragment);

                    try {
                        fragmentTransaction.commit();

                        completeRemoveTransaction(containerId, tag);
                        notifyTopmost(containerId);

                        // Deprecated since 0.9.2
                        if (count == 1) {
                            mActivity.onContainerEmpty(containerId);
                        }

                        // Notify activity
                        mActivity.onContainerChange(containerId);

                        return true;
                    } catch (IllegalStateException e) {
                        if (AnotherFragment.getDefault().isThrowStateLost()) {
                            throw e;
                        } else {
                            LogUtil.e(INTERNALTAG, String.format(Locale.getDefault(),
                                    "IllegalStateException detected, it may due to state lost. Exception: %s", e.toString()));
                        }
                    }
                }
            } else {
                LogUtil.e(INTERNALTAG, String.format(Locale.getDefault(),
                        "Unexpected behaviour: getTopFragmentTag return null. Container: %d", containerId));
            }
        } else {
            LogUtil.i(INTERNALTAG, String.format(Locale.getDefault(), "Unable to back any more. Container: %d", containerId));
        }

        return false;
    }


    /**
     * Add a fragment to specific container
     * @param containerId Target container id
     * @param fragment Fragment instance
     */
    public <T extends Fragment & IRootFragment> void add(int containerId, T fragment) {
        add(containerId, fragment, null);
    }

    /**
     * Add a fragment to specific container
     * @param containerId Target container id
     * @param fragment Fragment instance
     * @param tag User defined tag for find the fragment instance later
     */
    public <T extends Fragment & IRootFragment> void add(int containerId, T fragment, String tag) {
        add(containerId, fragment, false, tag);
    }

    /**
     * Add a fragment to specific container
     * @param containerId Target container id
     * @param fragment Fragment instance
     * @param hide Hide the fragment
     */
    public <T extends Fragment & IRootFragment> void add(int containerId, T fragment, boolean hide) {
        add(containerId, fragment, hide, null);
    }

    /**
     * Add a fragment to specific container
     * @param containerId Target container id
     * @param fragment Fragment instance
     * @param hide Hide the fragment
     * @param tag User defined tag for find the fragment instance later
     */
    public <T extends Fragment & IRootFragment> void add(int containerId, T fragment, boolean hide, String tag) {
        // Assign tag
        String internalTag = assignTag(fragment);

        // Map tag if any
        if (tag != null) {
            mTagMap.put(tag, internalTag);
        }

        // Transaction
        FragmentTransaction fragmentTransaction = getFragmentManager().beginTransaction();


        if (fragment.getEnterAnimation() != -1 && fragment.getExitAnimation() != -1) {
            fragmentTransaction.setCustomAnimations(fragment.getEnterAnimation(), fragment.getExitAnimation());
        }

        fragmentTransaction.add(containerId, fragment, internalTag);

        if (hide) {
            fragmentTransaction.hide(fragment);
        }

        try {
            fragmentTransaction.commit();

            completeAppendTransaction(containerId, fragment, internalTag);

            // Default hide will not affect topmost
            if (hide) {
                notifyTopmost(containerId);
            }

            // Notify activity
            mActivity.onContainerChange(containerId);
        } catch (IllegalStateException e) {
            if (AnotherFragment.getDefault().isThrowStateLost()) {
                throw e;
            } else {
                LogUtil.e(INTERNALTAG, String.format(Locale.getDefault(),
                        "IllegalStateException detected, it may due to state lost. Exception: %s", e.toString()));
            }
        }
    }


    /**
     * Flush before add a fragment to specific container
     * @param containerId Target container id
     * @param fragment Fragment instance
     */
    public <T extends Fragment & IRootFragment> void replace(int containerId, T fragment) {
        replace(containerId, fragment, null);
    }

    /**
     * Flush before add a fragment to specific container
     * @param containerId Target container id
     * @param fragment Fragment instance
     * @param tag User defined tag for find the fragment instance later
     */
    public <T extends Fragment & IRootFragment> void replace(int containerId, T fragment, String tag) {
        // Assign tag
        String internalTag = assignTag(fragment);

        // Map tag if any
        if (tag != null) {
            mTagMap.put(tag, internalTag);
        }

        // Since we don't use native back stack, we need to remove existing fragment manually
        List<FragmentTransaction> fragmentTransactions = getFlushTransactions(containerId, -1);

        // Transaction
        FragmentTransaction fragmentTransaction = getFragmentManager().beginTransaction();

        if (fragment.getEnterAnimation() != -1 && fragment.getExitAnimation() != -1) {
            fragmentTransaction.setCustomAnimations(fragment.getEnterAnimation(), fragment.getExitAnimation());
        }

        fragmentTransaction.add(containerId, fragment, internalTag);

        try {
            fragmentTransaction.commit();

            if (fragmentTransactions != null) {
                for (FragmentTransaction removeTransaction : fragmentTransactions) {
                    removeTransaction.commit();
                }
            }

            completeReplaceTransaction(containerId, fragment, internalTag);
            notifyTopmost(containerId);

            // Notify activity
            mActivity.onContainerChange(containerId);
        } catch (IllegalStateException e) {
            if (AnotherFragment.getDefault().isThrowStateLost()) {
                throw e;
            } else {
                LogUtil.e(INTERNALTAG, String.format(Locale.getDefault(),
                        "IllegalStateException detected, it may due to state lost. Exception: %s", e.toString()));
            }
        }
    }


    /**
     * Show a specific fragment
     * Please note that this tranaction will not record in backstack
     * @param fragment Fragment instance
     */
    public <T extends Fragment & IRootFragment> void show(T fragment) {
        String tag = getTag(fragment);
        if (tag == null) return;

        // Find container
        Integer containerId = null;
        for (int i = 0; i < mStacks.size(); i++) {
            int key = mStacks.keyAt(i);
            ContainerInfo containerInfo = mStacks.get(key);
            if (containerInfo.fragmentTags.contains(tag)) {
                containerId = key;
                break;
            }
        }
        if (containerId == null) return;

        // Do transaction
        FragmentTransaction fragmentTransaction = getFragmentManager().beginTransaction();
        fragmentTransaction.show(fragment);

        try {
            fragmentTransaction.commit();

            setVisibility(fragment, true);
            notifyTopmost(containerId);
        } catch (IllegalStateException e) {
            if (AnotherFragment.getDefault().isThrowStateLost()) {
                throw e;
            } else {
                LogUtil.e(INTERNALTAG, String.format(Locale.getDefault(),
                        "IllegalStateException detected, it may due to state lost. Exception: %s", e.toString()));
            }
        }
    }


    /**
     * Hide a specific fragment
     * Please note that this tranaction will not record in backstack
     * @param fragment Fragment instance
     */
    public <T extends Fragment & IRootFragment> void hide(T fragment) {
        String tag = getTag(fragment);
        if (tag == null) return;

        // Find container
        Integer containerId = null;
        for (int i = 0; i < mStacks.size(); i++) {
            int key = mStacks.keyAt(i);
            ContainerInfo containerInfo = mStacks.get(key);
            if (containerInfo.fragmentTags.contains(tag)) {
                containerId = key;
                break;
            }
        }
        if (containerId == null) return;

        // Do transaction
        FragmentTransaction fragmentTransaction = getFragmentManager().beginTransaction();
        fragmentTransaction.hide(fragment);

        try {
            fragmentTransaction.commit();

            setVisibility(fragment, false);
            notifyTopmost(containerId);
        } catch (IllegalStateException e) {
            if (AnotherFragment.getDefault().isThrowStateLost()) {
                throw e;
            } else {
                LogUtil.e(INTERNALTAG, String.format(Locale.getDefault(),
                        "IllegalStateException detected, it may due to state lost. Exception: %s", e.toString()));
            }
        }
    }


    /**
     * Remove the specific fragment in container
     * @param containerId Target container id
     * @param fragment Target fragment instance
     * @return true if removed, false if specific fragment not added
     * or not added by FragmentStore or not in the specific container
     */
    public <T extends Fragment & IRootFragment> boolean remove(int containerId, T fragment) {
        return remove(containerId, fragment, -1);
    }

    /**
     * Remove the specific fragment in container
     * @param containerId Target container id
     * @param fragment Target fragment instance
     * @param overrideAnimation Specific the exit animation
     * @return true if removed, false if specific fragment not added
     * or not added by FragmentStore or not in the specific container
     */
    public <T extends Fragment & IRootFragment> boolean remove(int containerId, T fragment, int overrideAnimation) {
        // Check if tag available
        String internalTag = getTag(fragment);
        if (internalTag != null) {
            // Check if in right container
            ContainerInfo containerInfo = mStacks.get(containerId);
            if (containerInfo != null && containerInfo.fragmentTags.contains(internalTag)) {
                FragmentTransaction fragmentTransaction = getFragmentManager().beginTransaction();

                if ((fragment.getEnterAnimation() != -1 && fragment.getExitAnimation() != -1) || overrideAnimation != -1) {
                    fragmentTransaction.setCustomAnimations(fragment.getEnterAnimation(), (overrideAnimation == -1) ? fragment.getExitAnimation() : overrideAnimation);
                }

                fragmentTransaction.remove(fragment);

                try {
                    fragmentTransaction.commit();

                    completeRemoveTransaction(containerId, internalTag);
                    notifyTopmost(containerId);

                    // Notify activity
                    mActivity.onContainerChange(containerId);

                    return true;
                } catch (IllegalStateException e) {
                    if (AnotherFragment.getDefault().isThrowStateLost()) {
                        throw e;
                    } else {
                        LogUtil.e(INTERNALTAG, String.format(Locale.getDefault(),
                                "IllegalStateException detected, it may due to state lost. Exception: %s", e.toString()));
                    }
                }
            } else {
                LogUtil.e(INTERNALTAG, String.format(Locale.getDefault(),
                        "Remove fragment container not match. Fragment: %s, Container: %d", fragment.getClass().getSimpleName(), containerId));
            }
        } else {
            LogUtil.e(INTERNALTAG, String.format(Locale.getDefault(),
                    "Remove fragment getTag return null. Fragment: %s", fragment.getClass().getSimpleName()));
        }

        return false;
    }
}