package hk.ids.gws.android.afragment.ui.fragmentmanage;

import android.app.Fragment;
import android.app.FragmentManager;
import android.app.FragmentTransaction;
import android.os.Bundle;
import android.support.annotation.AnimatorRes;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.SparseArray;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.UUID;

import hk.ids.gws.android.afragment.helper.AnotherFragment;
import hk.ids.gws.android.afragment.ui.activity.RootActivity;
import hk.ids.gws.android.afragment.ui.fragment.RootFragment;
import hk.ids.gws.android.afragment.utils.LogUtil;

@SuppressWarnings({"unused", "WeakerAccess", "deprecation", "unchecked"})
public class FragmentStore {
    protected final String INTERNALTAG = "AnotherFragment:" + getClass().getSimpleName();


    private final static String BUNDLE_TAG_FRAGMENT_TAG = "afragment:fragment.tag";

    private final static String BUNDLE_TAG_TAGMAP = "afragment:tagmap";
    private final static String BUNDLE_TAG_STACKS = "afragment:stacks";


    private RootActivity mActivity;

    private HashMap<String, String> mTagMap = new HashMap<>();
    private SparseArray<ContainerInfo> mStacks = new SparseArray<>();


    public FragmentStore(@NonNull RootActivity activity, @Nullable Bundle savedInstanceState) {
        mActivity = activity;

        // Restore state if any
        if (savedInstanceState != null) {
            Serializable serializable = savedInstanceState.getSerializable(BUNDLE_TAG_TAGMAP);
            SparseArray<ContainerInfo> sparseArray = savedInstanceState.getSparseParcelableArray(BUNDLE_TAG_STACKS);

            if (serializable != null) {
                mTagMap = (HashMap<String, String>) serializable;

                LogUtil.i(INTERNALTAG, "FragmentStore create from saved state, mTagMap restored");
            }
            if (sparseArray != null) {
                mStacks = sparseArray;

                LogUtil.i(INTERNALTAG, "FragmentStore create from saved state, mStacks restored");
            }
        }
    }

    public Bundle getInstanceState() {
        Bundle bundle = new Bundle();
        bundle.putSerializable(BUNDLE_TAG_TAGMAP, mTagMap);
        bundle.putSparseParcelableArray(BUNDLE_TAG_STACKS, mStacks);

        return bundle;
    }


    // Private methods

    private FragmentManager getFragmentManager() {
        return mActivity.getFragmentManager();
    }

    @Nullable
    private String getTopFragmentTag(int containerId) {
        return getTopNFragmentTag(containerId, 0);
    }

    @Nullable
    private String getTopNFragmentTag(int containerId, int n) {
        ContainerInfo containerInfo = mStacks.get(containerId);
        if (containerInfo == null || containerInfo.fragmentTags.size() <= n) return null;

        return containerInfo.fragmentTags.get(containerInfo.fragmentTags.size() - n - 1);
    }

    @Nullable
    private RootFragment getFragmentByTag(String internalTag) {
        Fragment fragment = getFragmentManager().findFragmentByTag(internalTag);
        if (fragment == null || !(fragment instanceof RootFragment)) {
            return null;
        } else {
            return ((RootFragment) fragment);
        }
    }

    @NonNull
    private String assignTag(RootFragment fragment) {
        String uuid = getTag(fragment);

        // Assign only if needed
        if (uuid == null) {
            uuid = UUID.randomUUID().toString();

            Bundle bundle = fragment.getArguments();

            if (bundle == null) bundle = new Bundle();
            bundle.putString(BUNDLE_TAG_FRAGMENT_TAG, uuid);

            fragment.setArguments(bundle);

            LogUtil.i(INTERNALTAG, String.format(Locale.getDefault(),
                    "Assigned new tag to fragment. Fragment: %s, Tag: %s", fragment.getClass().getSimpleName(), uuid));
        }

        return uuid;
    }

    @Nullable
    private String getTag(RootFragment fragment) {
        Bundle bundle = fragment.getArguments();
        if (bundle != null) {
            return bundle.getString(BUNDLE_TAG_FRAGMENT_TAG);
        }

        return null;
    }

    private void notifyTopmost(int containerId) {
        if (!AnotherFragment.getDefault().isNotifyTopmost()) return;

        ContainerInfo containerInfo = mStacks.get(containerId);
        if (containerInfo != null && containerInfo.fragmentTags.size() >= 0) {
            for (int i = 0; i < containerInfo.fragmentTags.size(); i++) {
                RootFragment fragment = getFragmentByTag(containerInfo.fragmentTags.get(i));
                if (fragment == null) {
                    LogUtil.e(INTERNALTAG, String.format(Locale.getDefault(),
                                    "Unexpected behaviour: getFragmentByTag return null. Index: %d, Container: %d", i, containerId));
                    continue;
                }

                if (i == containerInfo.fragmentTags.size() - 1) {
                    fragment.onTopmostChanged(true);
                } else {
                    fragment.onTopmostChanged(false);
                }
            }
        }
    }

    private void completeAppendTransaction(int containerId, String tag) {
        ContainerInfo containerInfo = mStacks.get(containerId);
        if (containerInfo == null) containerInfo = new ContainerInfo();

        containerInfo.fragmentTags.add(tag);

        mStacks.put(containerId, containerInfo);
    }

    private void completeRemoveTransaction(int containerId, String tag) {
        ContainerInfo containerInfo = mStacks.get(containerId);
        if (containerInfo == null) containerInfo = new ContainerInfo();

        containerInfo.fragmentTags.remove(tag);

        mStacks.put(containerId, containerInfo);
    }


    // Public methods

    /**
     * Get fragment count of given container
     * @param containerId Target container id
     */
    public int getCount(int containerId) {
        ContainerInfo containerInfo = mStacks.get(containerId);
        return containerInfo == null ? 0 : containerInfo.fragmentTags.size();
    }

    /**
     * Check if the given container is empty
     * @param containerId Target container id
     */
    public boolean isEmpty(int containerId) {
        ContainerInfo containerInfo = mStacks.get(containerId);
        return (containerInfo == null || containerInfo.fragmentTags.size() == 0);
    }

    /**
     * Check if given fragment is exist in given container
     * @param containerId Target container id
     * @param fragment Fragment instance to check
     */
    public boolean isExist(int containerId, RootFragment fragment) {
        String tag = getTag(fragment);
        if (tag != null) {
            ContainerInfo containerInfo = mStacks.get(containerId);
            if (containerInfo != null && containerInfo.fragmentTags.contains(tag)) {
                return true;
            }
        }

        return false;
    }


    /**
     * Get the top most fragment in given container
     * @param containerId Target container id
     */
    @Nullable
    public RootFragment getTopFragment(int containerId) {
        return getTopNFragment(containerId, 0);
    }

    /**
     * Get the fragment on specific index in given container
     * @param containerId Target container id
     * @param n z index of container
     */
    @Nullable
    public RootFragment getTopNFragment(int containerId, int n) {
        String tag = getTopNFragmentTag(containerId, n);
        if (tag == null) return null;

        Fragment fragment = getFragmentManager().findFragmentByTag(tag);
        if (fragment == null || !(fragment instanceof RootFragment)) {
            return null;
        } else {
            return ((RootFragment) fragment);
        }
    }

    /**
     * Get the fragment by tag
     * @param tag Tag given on add or replace transaction
     */
    @Nullable
    public RootFragment findFragmentByTag(String tag) {
        String storeTag = mTagMap.get(tag);
        if (storeTag == null) return null;

        Fragment fragment = getFragmentManager().findFragmentByTag(storeTag);
        if (fragment == null || !(fragment instanceof RootFragment)) {
            return null;
        } else {
            return ((RootFragment) fragment);
        }
    }


    /**
     * Flush the specific container by given id
     * @param containerId Target container id
     */
    public void flush(int containerId) {
        flush(containerId, -1);
    }

    /**
     * Flush the specific container by given id
     * @param containerId Target container id
     * @param overrideAnimation Specific the exit animation, it will only append to
     *                          topmost fragment, behide will remove without animation
     */
    public void flush(int containerId, int overrideAnimation) {
        ContainerInfo containerInfo = mStacks.get(containerId);
        if (containerInfo != null && containerInfo.fragmentTags.size() > 0) {
            List<RootFragment> pending = new ArrayList<>();

            for (int i = 0; i < containerInfo.fragmentTags.size(); i++) {
                RootFragment fragment = getFragmentByTag(containerInfo.fragmentTags.get(i));
                if (fragment != null) pending.add(fragment);
            }

            for (int i = 0; i < pending.size(); i++) {
                if (i == pending.size() - 1) {
                    remove(containerId, pending.get(i), overrideAnimation);
                } else {
                    remove(containerId, pending.get(i), 0);
                }
            }
        }
    }


    /**
     * Remove the top most fragment in container
     * @param containerId Target container id
     */
    public boolean back(int containerId) {
        return back(containerId, false, -1);
    }

    /**
     * Remove the top most fragment in container
     * @param containerId Target container id
     * @param allowEmpty Allow back to empty, last fragment will not be remove if false
     */
    public boolean back(int containerId, boolean allowEmpty) {
        return back(containerId, allowEmpty, -1);
    }

    /**
     * Remove the top most fragment in container
     * @param containerId Target container id
     * @param allowEmpty Allow back to empty, last fragment will not be remove if false
     * @param overrideAnimation Specific the exit animation
     */
    public boolean back(int containerId, boolean allowEmpty, @AnimatorRes int overrideAnimation) {
        // Check if able to back
        int count = getCount(containerId);
        if (count > (allowEmpty ? 0 : 1)) {
            // Check if tag available
            String tag = getTopFragmentTag(containerId);
            if (tag != null) {
                // Check if fragment exist and can be back
                RootFragment fragment = getFragmentByTag(tag);
                if (fragment != null && !fragment.onBackPressed()) {
                    FragmentTransaction fragmentTransaction = getFragmentManager().beginTransaction();

                    if ((fragment.getEnterAnimation() != -1 && fragment.getExitAnimation() != -1) || overrideAnimation != -1) {
                        fragmentTransaction.setCustomAnimations(fragment.getEnterAnimation(), (overrideAnimation == -1) ? fragment.getExitAnimation() : overrideAnimation);
                    }

                    fragmentTransaction.remove(fragment);

                    try {
                        fragmentTransaction.commit();

                        getFragmentManager().executePendingTransactions();

                        completeRemoveTransaction(containerId, tag);
                        notifyTopmost(containerId);

                        // Notify empty if needed
                        if (count == 1) {
                            mActivity.onContainerEmpty(containerId);
                        }

                        return true;
                    } catch (IllegalStateException e) {
                        if (AnotherFragment.getDefault().isThrowStateLost()) {
                            throw e;
                        } else {
                            LogUtil.e(INTERNALTAG, String.format(Locale.getDefault(),
                                    "IllegalStateException detected, it may due to state lost. Exception: %s", e.toString()));
                        }
                    }
                }
            } else {
                LogUtil.e(INTERNALTAG, String.format(Locale.getDefault(),
                        "Unexpected behaviour: getTopFragmentTag return null. Container: %d", containerId));
            }
        } else {
            LogUtil.i(INTERNALTAG, String.format(Locale.getDefault(), "Unable to back any more. Container: %d", containerId));
        }

        return false;
    }


    /**
     * Add a fragment to specific container
     * @param containerId Target container id
     * @param fragment Fragment instance
     */
    public void add(int containerId, RootFragment fragment) {
        add(containerId, fragment, null);
    }

    /**
     * Add a fragment to specific container
     * @param containerId Target container id
     * @param fragment Fragment instance
     * @param tag User defined tag for find the fragment instance later
     */
    public void add(int containerId, RootFragment fragment, String tag) {
        // Assign tag
        String internalTag = assignTag(fragment);

        // Map tag if any
        if (tag != null) {
            mTagMap.put(tag, internalTag);
        }

        // Transaction
        FragmentTransaction fragmentTransaction = getFragmentManager().beginTransaction();

        if (fragment.getEnterAnimation() != -1 && fragment.getExitAnimation() != -1) {
            fragmentTransaction.setCustomAnimations(fragment.getEnterAnimation(), fragment.getExitAnimation());
        }

        fragmentTransaction.add(containerId, fragment, internalTag);

        try {
            fragmentTransaction.commit();

            getFragmentManager().executePendingTransactions();

            completeAppendTransaction(containerId, internalTag);
            notifyTopmost(containerId);
        } catch (IllegalStateException e) {
            if (AnotherFragment.getDefault().isThrowStateLost()) {
                throw e;
            } else {
                LogUtil.e(INTERNALTAG, String.format(Locale.getDefault(),
                        "IllegalStateException detected, it may due to state lost. Exception: %s", e.toString()));
            }
        }
    }


    /**
     * Flush before add a fragment to specific container
     * @param containerId Target container id
     * @param fragment Fragment instance
     */
    public void replace(int containerId, RootFragment fragment) {
        replace(containerId, fragment, null);
    }

    /**
     * Flush before add a fragment to specific container
     * @param containerId Target container id
     * @param fragment Fragment instance
     * @param tag User defined tag for find the fragment instance later
     */
    public void replace(int containerId, RootFragment fragment, String tag) {
        // Assign tag
        String internalTag = assignTag(fragment);

        // Map tag if any
        if (tag != null) {
            mTagMap.put(tag, internalTag);
        }

        // Transaction
        FragmentTransaction fragmentTransaction = getFragmentManager().beginTransaction();

        if (fragment.getEnterAnimation() != -1 && fragment.getExitAnimation() != -1) {
            fragmentTransaction.setCustomAnimations(fragment.getEnterAnimation(), fragment.getExitAnimation());
        }

        fragmentTransaction.replace(containerId, fragment, internalTag);

        try {
            fragmentTransaction.commit();

            getFragmentManager().executePendingTransactions();

            completeAppendTransaction(containerId, internalTag);
            notifyTopmost(containerId);
        } catch (IllegalStateException e) {
            if (AnotherFragment.getDefault().isThrowStateLost()) {
                throw e;
            } else {
                LogUtil.e(INTERNALTAG, String.format(Locale.getDefault(),
                        "IllegalStateException detected, it may due to state lost. Exception: %s", e.toString()));
            }
        }
    }


    /**
     * Show a specific fragment
     * Please note that this tranaction will not record in backstack
     * @param fragment Fragment instance
     */
    public void show(RootFragment fragment) {
        FragmentTransaction fragmentTransaction = getFragmentManager().beginTransaction();
        fragmentTransaction.show(fragment);

        try {
            fragmentTransaction.commit();

            getFragmentManager().executePendingTransactions();
        } catch (IllegalStateException e) {
            if (AnotherFragment.getDefault().isThrowStateLost()) {
                throw e;
            } else {
                LogUtil.e(INTERNALTAG, String.format(Locale.getDefault(),
                        "IllegalStateException detected, it may due to state lost. Exception: %s", e.toString()));
            }
        }
    }


    /**
     * Hide a specific fragment
     * Please note that this tranaction will not record in backstack
     * @param fragment Fragment instance
     */
    public void hide(RootFragment fragment) {
        FragmentTransaction fragmentTransaction = getFragmentManager().beginTransaction();
        fragmentTransaction.hide(fragment);

        try {
            fragmentTransaction.commit();

            getFragmentManager().executePendingTransactions();
        } catch (IllegalStateException e) {
            if (AnotherFragment.getDefault().isThrowStateLost()) {
                throw e;
            } else {
                LogUtil.e(INTERNALTAG, String.format(Locale.getDefault(),
                        "IllegalStateException detected, it may due to state lost. Exception: %s", e.toString()));
            }
        }
    }


    /**
     * Remove the specific fragment in container
     * @param containerId Target container id
     * @param fragment Target fragment instance
     * @return true if removed, false if specific fragment not added
     * or not added by FragmentStore or not in the specific container
     */
    public boolean remove(int containerId, RootFragment fragment) {
        return remove(containerId, fragment, -1);
    }

    /**
     * Remove the specific fragment in container
     * @param containerId Target container id
     * @param fragment Target fragment instance
     * @param overrideAnimation Specific the exit animation
     * @return true if removed, false if specific fragment not added
     * or not added by FragmentStore or not in the specific container
     */
    public boolean remove(int containerId, RootFragment fragment, int overrideAnimation) {
        // Check if tag available
        String internalTag = getTag(fragment);
        if (internalTag != null) {
            // Check if in right container
            ContainerInfo containerInfo = mStacks.get(containerId);
            if (containerInfo != null && containerInfo.fragmentTags.contains(internalTag)) {
                FragmentTransaction fragmentTransaction = getFragmentManager().beginTransaction();

                if ((fragment.getEnterAnimation() != -1 && fragment.getExitAnimation() != -1) || overrideAnimation != -1) {
                    fragmentTransaction.setCustomAnimations(fragment.getEnterAnimation(), (overrideAnimation == -1) ? fragment.getExitAnimation() : overrideAnimation);
                }

                fragmentTransaction.remove(fragment);

                try {
                    fragmentTransaction.commit();

                    getFragmentManager().executePendingTransactions();

                    completeRemoveTransaction(containerId, internalTag);
                    notifyTopmost(containerId);

                    return true;
                } catch (IllegalStateException e) {
                    if (AnotherFragment.getDefault().isThrowStateLost()) {
                        throw e;
                    } else {
                        LogUtil.e(INTERNALTAG, String.format(Locale.getDefault(),
                                "IllegalStateException detected, it may due to state lost. Exception: %s", e.toString()));
                    }
                }
            } else {
                LogUtil.e(INTERNALTAG, String.format(Locale.getDefault(),
                        "Remove fragment container not match. Fragment: %s, Container: %d", fragment.getClass().getSimpleName(), containerId));
            }
        } else {
            LogUtil.e(INTERNALTAG, String.format(Locale.getDefault(),
                    "Remove fragment getTag return null. Fragment: %s", fragment.getClass().getSimpleName()));
        }

        return false;
    }
}
