package de.cidaas.jwt;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.UUID;

public class JWTHelper {

	public static Map<String, Object> encodedPayload(Map<String, Object> _claims, Options options) throws Exception {
		Map<String, Object> claims = new HashMap<String, Object>(_claims);
		enforceStringOrURI(claims, "iss");
		enforceStringOrURI(claims, "sub");
		enforceStringOrURICollection(claims, "aud");
		enforceIntDate(claims, "exp");
		enforceIntDate(claims, "nbf");
		enforceIntDate(claims, "iat");
		enforceString(claims, "jti");
		processPayloadOptions(claims, options);

		return claims;
	}

	public static void processPayloadOptions(Map<String, Object> claims, Options options) {
		long now = System.currentTimeMillis() / 1000l;
		if (options.getNotValidBeforeLeeway() != null) {
			claims.put("nbf", now - options.getNotValidBeforeLeeway());
		}

		claims.put("iat", now);
		claims.put("jti", UUID.randomUUID().toString());
	}

	public static void enforceIntDate(Map<String, Object> claims, String claimName) {
		Object value = handleNullValue(claims, claimName);
		if (value == null)
			return;
		if (!(value instanceof Number)) {
			throw new RuntimeException(
					String.format("Claim '%s' is invalid: must be an instance of Number", claimName));
		}
		long longValue = ((Number) value).longValue();
		if (longValue < 0)
			throw new RuntimeException(String.format("Claim '%s' is invalid: must be non-negative", claimName));
		claims.put(claimName, longValue);
	}

	public static void enforceStringOrURICollection(Map<String, Object> claims, String claimName) {
		Object values = handleNullValue(claims, claimName);
		if (values == null)
			return;
		if (values instanceof Collection) {
			@SuppressWarnings({ "unchecked" })
			Iterator<Object> iterator = ((Collection<Object>) values).iterator();
			while (iterator.hasNext()) {
				Object value = iterator.next();
				String error = checkStringOrURI(value);
				if (error != null)
					throw new RuntimeException(String.format("Claim 'aud' element is invalid: %s", error));
			}
		} else {
			enforceStringOrURI(claims, "aud");
		}
	}

	public static void enforceStringOrURI(Map<String, Object> claims, String claimName) {
		Object value = handleNullValue(claims, claimName);
		if (value == null)
			return;
		String error = checkStringOrURI(value);
		if (error != null)
			throw new RuntimeException(String.format("Claim '%s' is invalid: %s", claimName, error));
	}

	public static void enforceString(Map<String, Object> claims, String claimName) {
		Object value = handleNullValue(claims, claimName);
		if (value == null)
			return;
		if (!(value instanceof String))
			throw new RuntimeException(String.format("Claim '%s' is invalid: not a string", claimName));
	}

	public static Object handleNullValue(Map<String, Object> claims, String claimName) {
		if (!claims.containsKey(claimName))
			return null;
		Object value = claims.get(claimName);
		if (value == null) {
			claims.remove(claimName);
			return null;
		}
		return value;
	}

	public static String checkStringOrURI(Object value) {
		if (!(value instanceof String))
			return "not a string";
		String stringOrUri = (String) value;
		if (!stringOrUri.contains(":"))
			return null;
		try {
			new URI(stringOrUri);
		} catch (URISyntaxException e) {
			return "not a valid URI";
		}
		return null;
	}

}
