package de.cidaas.jwt;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.security.InvalidKeyException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.SignatureException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import javax.naming.OperationNotSupportedException;

import org.apache.commons.codec.binary.Base64;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import com.fasterxml.jackson.databind.node.ObjectNode;

public class JWTGenerator {

	public String sign(Map<String, Object> claims, Options options) {

		Algorithm algorithm = Algorithm.HS256;

		List<String> segments = new ArrayList<String>();
		try {
			segments.add(encodedHeader(algorithm));
			segments.add(encodedPayload(claims, options));
			segments.add(encodedSignature(join(segments, "."), algorithm, options.getSecret()));
		} catch (Exception e) {
			throw (e instanceof RuntimeException) ? (RuntimeException) e : new RuntimeException(e);
		}

		return join(segments, ".");
	}

	private String encodedHeader(Algorithm algorithm) throws UnsupportedEncodingException {
		if (algorithm == null) { // default the algorithm if not specified
			algorithm = Algorithm.HS256;
		}

		// create the header
		ObjectNode header = JsonNodeFactory.instance.objectNode();
		header.put("typ", "JWT");
		header.put("alg", algorithm.name());

		return base64UrlEncode(header.toString().getBytes("UTF-8"));
	}

	private String encodedPayload(Map<String, Object> _claims, Options options) throws Exception {
		String payload = new ObjectMapper().writeValueAsString(JWTHelper.encodedPayload(_claims, options));
		return base64UrlEncode(payload.getBytes("UTF-8"));
	}

	/**
	 * Sign the header and payload
	 */
	private String encodedSignature(String signingInput, Algorithm algorithm, String secret) throws Exception {
		byte[] signature = sign(algorithm, signingInput, secret.getBytes());
		return base64UrlEncode(signature);
	}

	/**
	 * Safe URL encode a byte array to a String
	 */
	private String base64UrlEncode(byte[] str) {
		return new String(Base64.encodeBase64URLSafe(str));
	}

	/**
	 * Switch the signing algorithm based on input, RSA not supported
	 */
	private static byte[] sign(Algorithm algorithm, String msg, byte[] secret) throws Exception {
		switch (algorithm) {
		case HS256:
		case HS384:
		case HS512:
			return signHmac(algorithm, msg, secret);
		case RS256:
		case RS384:
		case RS512:
		default:
			throw new OperationNotSupportedException("Unsupported signing method");
		}
	}

	/**
	 * Sign an input string using HMAC and return the encrypted bytes
	 */
	private static byte[] signHmac(Algorithm algorithm, String msg, byte[] secret) throws Exception {
		Mac mac = Mac.getInstance(algorithm.getValue());
		mac.init(new SecretKeySpec(secret, algorithm.getValue()));
		return mac.doFinal(msg.getBytes());
	}

	private String join(List<String> input, String on) {
		int size = input.size();
		int count = 1;
		StringBuilder joined = new StringBuilder();
		for (String string : input) {
			joined.append(string);
			if (count < size) {
				joined.append(on);
			}
			count++;
		}

		return joined.toString();
	}

	public Map<String, Object> verify(String jwtToken, Options options) throws NoSuchAlgorithmException,
			InvalidKeyException, IllegalStateException, IOException, SignatureException, JWTVerifyException {
		if (jwtToken == null || "".equals(jwtToken)) {
			throw new IllegalStateException("token not set");
		}

		String[] pieces = jwtToken.split("\\.");

		// check number of segments
		if (pieces.length != 3) {
			throw new IllegalStateException("Wrong number of segments: " + pieces.length);
		}

		// get JWTHeader JSON object. Extract algorithm
		JsonNode jwtHeader = decodeAndParse(pieces[0]);

		String algorithm = getAlgorithm(jwtHeader);

		// get JWTClaims JSON object
		JsonNode jwtPayload = decodeAndParse(pieces[1]);

		// check signature
		if (verifySignature(pieces, algorithm, options.getSecret())) {
			return new ObjectMapper().treeToValue(jwtPayload, Map.class);
		}
		return null;
	}

	public Map<String, Object> parse(String jwtToken) throws NoSuchAlgorithmException,
			InvalidKeyException, IllegalStateException, IOException, SignatureException, JWTVerifyException {
		if (jwtToken == null || "".equals(jwtToken)) {
			throw new IllegalStateException("token not set");
		}

		String[] pieces = jwtToken.split("\\.");

		// check number of segments
		if (pieces.length != 3) {
			throw new IllegalStateException("Wrong number of segments: " + pieces.length);
		}

		// get JWTClaims JSON object
		JsonNode jwtPayload = decodeAndParse(pieces[1]);
		return new ObjectMapper().treeToValue(jwtPayload, Map.class);
	}

	static boolean verifySignature(String[] pieces, String algorithm, String secret)
			throws NoSuchAlgorithmException, InvalidKeyException, SignatureException {
		Mac hmac = Mac.getInstance(algorithm);
		hmac.init(new SecretKeySpec(secret.getBytes(), algorithm));
		byte[] sig = hmac.doFinal(new StringBuilder(pieces[0]).append(".").append(pieces[1]).toString().getBytes());

		if (!MessageDigest.isEqual(sig, Base64.decodeBase64(pieces[2]))) {
			return false;
		}
		return true;
	}

	static String getAlgorithm(JsonNode jwtHeader) {
		Map<String, String> algorithms = new HashMap<String, String>();
		algorithms.put("HS256", "HmacSHA256");
		algorithms.put("HS384", "HmacSHA384");
		algorithms.put("HS512", "HmacSHA512");

		final String algorithmName = jwtHeader.has("alg") ? jwtHeader.get("alg").asText() : null;

		if (jwtHeader.get("alg") == null) {
			throw new IllegalStateException("algorithm not set");
		}

		if (algorithms.get(algorithmName) == null) {
			throw new IllegalStateException("unsupported algorithm");
		}

		return algorithms.get(algorithmName);
	}

	static JsonNode decodeAndParse(String b64String) throws IOException {
		String jsonString = new String(Base64.decodeBase64(b64String), "UTF-8");
		JsonNode jwtHeader = new ObjectMapper().readValue(jsonString, JsonNode.class);
		return jwtHeader;
	}

}
