package de.cidaas.jwt;

import java.security.interfaces.RSAPrivateKey;
import java.security.interfaces.RSAPublicKey;
import java.text.ParseException;
import java.util.HashMap;
import java.util.Map;

import com.nimbusds.jose.EncryptionMethod;
import com.nimbusds.jose.JWEAlgorithm;
import com.nimbusds.jose.JWEHeader;
import com.nimbusds.jose.crypto.RSADecrypter;
import com.nimbusds.jose.crypto.RSAEncrypter;
import com.nimbusds.jose.crypto.bc.BouncyCastleProviderSingleton;
import com.nimbusds.jwt.EncryptedJWT;
import com.nimbusds.jwt.JWTClaimsSet;
import com.nimbusds.jwt.JWTClaimsSet.Builder;

import net.minidev.json.JSONObject;

public class JWEGenerator {

	public String sign(Map<String, Object> claims, Options options) throws Exception {

		RSAPublicKey publicKey = null;

		if (options.getPublicKey() != null && options.getPublicKey().trim().length() > 0) {
			publicKey = RSAKeyHelper.parsePublicKey(options.getPublicKey());
		} else if (options.getPublicKeyPath() != null && options.getPublicKeyPath().trim().length() > 0) {
			publicKey = RSAKeyHelper.getPublicKey(options.getPublicKeyPath());
		}

		if (publicKey == null) {
			return null;
		}

		claims = JWTHelper.encodedPayload(claims, options);

		Builder claimBuilder = new JWTClaimsSet.Builder();
		for (Map.Entry<String, Object> entry : claims.entrySet()) {
			String key = entry.getKey();
			Object value = entry.getValue();
			claimBuilder.claim(key, value);
		}

		JWTClaimsSet jwtClaims = claimBuilder.build();
		
		// Request JWT encrypted with RSA-OAEP and 128-bit AES/GCM
		JWEHeader header = new JWEHeader(JWEAlgorithm.RSA_OAEP, EncryptionMethod.A128GCM);

		// Create the encrypted JWT object
		EncryptedJWT jwt = new EncryptedJWT(header, jwtClaims);

		// Create an encrypter with the specified public RSA key
		RSAEncrypter encrypter = new RSAEncrypter(publicKey);
		encrypter.getJCAContext().setProvider(BouncyCastleProviderSingleton.getInstance());

		// Do the actual encryption
		jwt.encrypt(encrypter);

		// Serialise to JWT compact form
		String jwtString = jwt.serialize();
		System.out.println(jwtString);

		return jwtString;
	}

	public Map<String, Object> verify(String jwtString, Options options) throws Exception {

		RSAPrivateKey privateKey = null;

		if (options.getPrivateKey() != null && options.getPrivateKey().trim().length() > 0) {
			privateKey = RSAKeyHelper.parsePrivateKey(options.getPrivateKey());
		} else if (options.getPrivateKeyPath() != null && options.getPrivateKeyPath().trim().length() > 0) {
			privateKey = RSAKeyHelper.getPrivateKey(options.getPrivateKeyPath());
		}

		if (privateKey == null) {
			return null;
		}

		// Parse back
		EncryptedJWT jwt = EncryptedJWT.parse(jwtString);

		// Create an decrypter with the specified private RSA key
		RSADecrypter decrypter = new RSADecrypter(privateKey);
		decrypter.getJCAContext().setProvider(BouncyCastleProviderSingleton.getInstance());

		// Decrypt
		jwt.decrypt(decrypter);

		if (jwt.getPayload() == null) {
			return null;
		}

		JSONObject json = jwt.getPayload().toJSONObject();

		if (json == null) {
			throw new ParseException("Payload of JWE object is not a valid JSON object", 0);
		}
		
		Map<String, Object> claims = new HashMap<>();

		for (final String name : json.keySet()) {
			claims.put(name, json.get(name));
		}
		return claims;
	}

}
