package de.cidaas.oauth.util;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.net.URI;

import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.HttpClientBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;

public class CidaasConfigResolver {
	private static final Logger LOG = LoggerFactory.getLogger(ConfigurationLoader.class);
	private static final String PROPS = "cidaas_config.properties";
	private ObjectMapper om;
	private static CidaasConfigResolver _URLResolver;

	public OpenIdConfiguration openIdConfiguration;

	public String client_id;
	public String client_secret;

	public static synchronized CidaasConfigResolver getInstance() throws Exception {
		if (_URLResolver == null) {
			_URLResolver = new CidaasConfigResolver();
		}
		return _URLResolver;
	}

	public CidaasConfigResolver() throws Exception {
		om = new ObjectMapper();
		om = om.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
		om = om.configure(DeserializationFeature.FAIL_ON_IGNORED_PROPERTIES, false);

		String cidaas_base_url = System.getenv("cidaas_base_url");
		if (StringUtils.isEmpty(cidaas_base_url)) {
			cidaas_base_url = ConfigurationLoader.getPropertyByKey("base_url", PROPS);
			if (StringUtils.isEmpty(cidaas_base_url)) {
				throw new Exception(
						"Cidaas Base URL is must not be empty. Please add it in the Environment Variable wiht Key : cidaas_base_url or Create the cidaas_config.properties file and the property as base_url");
			}
		}

		openIdConfiguration = resolveOpenIdConnectURL(cidaas_base_url);

		String cidaas_client_id = System.getenv("cidaas_client_id");
		if (StringUtils.isEmpty(cidaas_client_id)) {
			cidaas_client_id = ConfigurationLoader.getPropertyByKey("client_id", PROPS);
			if (StringUtils.isEmpty(cidaas_client_id)) {
				throw new Exception(
						"Cidaas client_id is must not be empty. Please add it in the Environment Variable wiht Key : cidaas_client_id or Create the cidaas_config.properties file and the property as client_id");
			}
		}

		this.client_id = cidaas_client_id;

		String cidaas_client_secret = System.getenv("cidaas_client_secret");
		if (StringUtils.isEmpty(cidaas_client_secret)) {
			cidaas_client_secret = ConfigurationLoader.getPropertyByKey("client_secret", PROPS);
			if (StringUtils.isEmpty(cidaas_client_secret)) {
				throw new Exception(
						"Cidaas client_secret is must not be empty. Please add it in the Environment Variable wiht Key : cidaas_client_secret or Create the cidaas_config.properties file and the property as client_secret");
			}
		}
		
		this.client_secret = cidaas_client_secret;

	}

	private OpenIdConfiguration resolveOpenIdConnectURL(String base_url) throws Exception {
		try {

			String uri = combinePath(base_url, ".well-known/openid-configuration");

			HttpGet resetPost = new HttpGet(new URI(uri));

			HttpResponse response = HttpClientBuilder.create().build().execute(resetPost);
			int responseStatusCode = response.getStatusLine().getStatusCode();
			if (responseStatusCode == HttpStatus.SC_OK) {
				BufferedReader rd = new BufferedReader(new InputStreamReader(response.getEntity().getContent()));
				StringBuffer resultData = new StringBuffer();
				String line = "";
				while ((line = rd.readLine()) != null) {
					resultData.append(line);
				}
				OpenIdConfiguration json = om.readValue(resultData.toString(), OpenIdConfiguration.class);
				return json;
			} else {
				throw new Exception(
						"Error while resolving the openid-configuration service, status code : " + responseStatusCode);
			}
		} catch (Exception ex) {
			throw new Exception("Error while resolving the openid-configuration, Error : " + ex);
		}

	}

	private static String combinePath(String... paths) {

		String urlToReturn = paths[0];
		for (int i = 1; i < paths.length; i++) {
			if (urlToReturn.endsWith("/") && paths[i].startsWith("/")) {
				urlToReturn = urlToReturn.substring(0, urlToReturn.length() - 1);
				urlToReturn = urlToReturn + paths[i];
			} else if (urlToReturn.endsWith("/") && !paths[i].startsWith("/")) {
				urlToReturn = urlToReturn + paths[i];
			} else if (!urlToReturn.endsWith("/") && paths[i].startsWith("/")) {
				urlToReturn = urlToReturn + paths[i];
			} else if (!urlToReturn.endsWith("/") && !paths[i].startsWith("/")) {
				urlToReturn = urlToReturn + "/" + paths[i];
			}
		}

		if (urlToReturn.endsWith("/")) {
			urlToReturn = urlToReturn.substring(0, urlToReturn.length() - 1);
		}

		return urlToReturn;
	}

}
