/*
 * Copyright 2000-2024 Vaadin Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.vaadin.pro.licensechecker.dau;

import java.time.Duration;
import java.time.Instant;
import java.time.LocalTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.function.LongConsumer;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vaadin.pro.licensechecker.Constants;
import com.vaadin.pro.licensechecker.LicenseException;
import com.vaadin.pro.licensechecker.LicenseValidationException;

/**
 * Memory storage for Daily Active User tracking.
 * <p>
 * </p>
 * This component is responsible for collecting DAU data and periodically
 * synchronize with Vaadin License Server.
 * <p>
 * </p>
 * When tracking a new user, DauStorage applies the enforcement rule that the
 * License Server provide after synchronization, potentially raising an error if
 * the subscription limits have been exceeded.
 * <p>
 * </p>
 * DAU data is synchronized with fixed interval of 24 hours.
 */
final class DauStorage {

    private static final Logger LOGGER = LoggerFactory
            .getLogger(DauStorage.class);
    private static final String LICENSE_SERVER_URL = System.getProperty(Constants.LICENSE_SERVER_URL_PARAMETER,
            Constants.LICENSE_SERVER_URL);

    private ScheduledFuture<?> publishingFuture;

    private static class Holder {
        private static DauStorage INSTANCE = new DauStorage(
                new LicenseServerPublisher(LICENSE_SERVER_URL),
                new PublishingSettings(Duration.ofHours(24),
                        Duration.ofMinutes(5)));
    }

    private final ScheduledExecutorService executorService = Executors
            .newSingleThreadScheduledExecutor(dauThreadFactory());
    private final Map<String, TrackedUser> storage = new HashMap<>();
    // the blockedUser map registers all the user identities that have been
    // blocked for a tracking hash because of DAU enforcement.
    // (key = trackingHash, value = set of blocked user identities)
    // The aim of this registry, is to be able to immediately throw enforcement
    // exception when an identified tracked user has already been blocked on a
    // different device, without counting it as a new DAU user.
    // For example, a Desktop browser with trackingHash T1 logged as user U1
    // gets blocked; if the same user tries to access from another device
    // (trackingHash T2) with the same credentials (or deletes the tracking
    // cookie on the same device), it can be immediately blocked without
    // decreasing the remaining user quota.
    private final Map<String, Set<String>> blockedUsers = new HashMap<>();
    private final PublishingTask publishingTask = new PublishingTask();

    private final Publisher publisher;
    private final PublishingSettings publishingSettings;

    private EnforcementRule enforcementRule = new EnforcementRule(false,
            Integer.MAX_VALUE);
    private Instant enforcementRuleUpdatedAt;
    private String applicationName;
    private volatile int newUsersCounter = 0;

    // visible for test
    DauStorage(Publisher publisher, PublishingSettings publishingSettings) {
        this.publisher = publisher;
        this.publishingSettings = publishingSettings;
    }

    /**
     * Gets the {@link DauStorage} instance.
     *
     * @return the {@link DauStorage} instance.
     */
    static DauStorage getInstance() {
        return Holder.INSTANCE;
    }

    /**
     * Starts Daily Active User tracking for the calling application.
     * <p>
     * </p>
     * On start, License Server is queried to verify subscription key validity
     * and enforcement status. If checks are passed, a background Job is started
     * to synchronize DAU data and get updated enforcement information at
     * regular intervals (24 hours).
     * <p>
     * </p>
     * This method is usually called once at application startup. Subsequent
     * calls to the method have no effects. Calling this method after
     * {@link #stop()} is not allowed.
     *
     * @param applicationName
     *            name of the application associated to the subscription key.
     * @throws com.vaadin.pro.licensechecker.LicenseException
     *             if subscription key is invalid or expired
     * @throws IllegalStateException
     *             if invoked after a call to {@link #stop()}
     */
    void start(String applicationName) {
        synchronized (storage) {
            if (executorService.isShutdown()
                    || executorService.isTerminated()) {
                throw new IllegalStateException(
                        "DAU storage cannot be restarted after stop has been invoked");
            }
            if (publishingFuture != null) {
                LOGGER.debug("Attempt to restart DAU storage after stop");
                return;
            }
            long delay = publishingSettings.publishingInterval.toMillis();
            publishingFuture = executorService.scheduleWithFixedDelay(
                    () -> this.publishUpdates(PublishingPhase.PERIOD), delay,
                    delay, TimeUnit.MILLISECONDS);
        }
        this.applicationName = applicationName;
        LOGGER.debug("Started DAU publishing Job with interval {}",
                publishingSettings.publishingInterval);

        Runtime.getRuntime().addShutdownHook(new Thread(this::stop));
        LOGGER.debug("DAU synchronization started for application {}.",
                applicationName);
        publishUpdates(PublishingPhase.START);
    }

    // Visible for test
    int getNewUsersCounter() {
        return newUsersCounter;
    }

    // Visible for test
    Map<String, TrackedUser> getUsersInStorage() {
        return storage;
    }

    // Visible for test
    Map<String, Set<String>> getBlockedUsers() {
        return blockedUsers;
    }

    /**
     * Stops Daily Active User tracking for the calling application.
     * <p>
     * </p>
     * On stop, a last synchronization with License Server is attempted, to
     * flush remaining DAU data.
     * <p>
     * </p>
     * This method is usually called once by the application at shutdown, but it
     * is also invoked when JVM stops. Subsequent calls to the method have no
     * effects. Tracking cannot be restarted after this method is called.
     * Calling {@code stop} without {@link #start(String)} does not produce any
     * effect.
     */
    void stop() {
        synchronized (storage) {
            if (publishingFuture != null) {
                LOGGER.debug("Stopping DAU synchronization for application {}.",
                        applicationName);
                publishingFuture.cancel(false);
                publishingFuture = null;
                publishUpdates(PublishingPhase.STOP);
                LOGGER.debug("DAU synchronization stopped for application {}.",
                        applicationName);
            }
            executorService.shutdown();
        }
    }

    /**
     * Potentially adds new tracking entry to the DAU data set, if the given
     * tracking corresponds to a new user.
     * <p>
     * </p>
     * If the {@code trackingHash} is not already known, a new entry is added to
     * the DAU dataset and the new users counter is increased.
     * <p>
     * </p>
     * When a new entry is added, the enforcement rule is evaluated and a
     * {@link EnforcementException} is thrown if enforcement is required and the
     * number of new users exceeds the limit.
     *
     * @param trackingHash
     *            the user to track
     * @throws EnforcementException
     *             if enforcement should be applied by the client.
     */
    void track(String trackingHash) throws EnforcementException {
        track(trackingHash, null);
    }

    /**
     * Adds or updates the tracking entry for the given tracking hash.
     * <p>
     * </p>
     * If the {@code trackingHash} is not already known, a new entry is added to
     * the DAU dataset and the new users counter is increased. If an entry
     * already exists, the user counter is increased only if the provided
     * {@code identityHash} was not already linked to the tracked entry.
     * <p>
     * </p>
     * When a new entry is added, the enforcement rule is evaluated and a
     * {@link EnforcementException} is thrown if enforcement is required and the
     * number of new users exceeds the limit.
     *
     * @param trackingHash
     *            the user tracking identifier
     * @throws EnforcementException
     *             if enforcement should be applied by the client.
     */
    void track(String trackingHash, String identityHash)
            throws EnforcementException {
        Objects.requireNonNull(trackingHash, "trackingHash cannot be null");
        synchronized (storage) {
            boolean isNewTrackingHash = !storage.containsKey(trackingHash);
            if (blockedUsers.getOrDefault(trackingHash, Collections.emptySet())
                    .contains(identityHash)) {
                LOGGER.debug(
                        "User {} with identity {} has already been blocked because of enforcement rule",
                        trackingHash, identityHash);
                applyEnforcement(trackingHash, identityHash);
            }

            // If a user identity is given, check if the user is already tracked
            // with different tracking hashes (e.g. login from different
            // devices)
            boolean isTrackedIdentity;
            if (identityHash != null) {
                Set<String> sameIdentityTrackingHashes = storage.entrySet()
                        .stream()
                        .filter(entry -> !entry.getKey().equals(trackingHash))
                        .filter(entry -> entry.getValue()
                                .getUserIdentityHashes().contains(identityHash))
                        .map(Map.Entry::getKey).collect(Collectors.toSet());
                isTrackedIdentity = !sameIdentityTrackingHashes.isEmpty();

                if (isNewTrackingHash && isTrackedIdentity) {
                    LOGGER.debug(
                            "User {} with identity {} is already linked with other tracking hashes {}.",
                            trackingHash, identityHash,
                            sameIdentityTrackingHashes);
                }

                if (sameIdentityTrackingHashes.stream()
                        .anyMatch(otherTrackingHash -> blockedUsers
                                .getOrDefault(otherTrackingHash,
                                        Collections.emptySet())
                                .contains(identityHash))) {
                    LOGGER.debug(
                            "User {} with identity {} has already been blocked because of enforcement rule on another device",
                            trackingHash, identityHash);
                    applyEnforcement(trackingHash, identityHash);
                }
            } else {
                isTrackedIdentity = false;
            }

            TrackedUser trackedUser = storage.computeIfAbsent(trackingHash,
                    TrackedUser::new);
            boolean isAnonymous = !trackedUser.hasLinkedIdentities();
            boolean newIdentity = trackedUser.linkUserIdentity(identityHash);

            // The user count must be increased in two cases:
            // - it is the first time that the trackingHash has been seen and
            // its user identity has not been tracked on different devices
            // - a new identity has been associated to a tracking hash that was
            // previously linked to a different not null identity; it happens
            // for example if a user accesses the application with different
            // credentials from the same device.
            if ((isNewTrackingHash && !isTrackedIdentity)
                    || (newIdentity && !isAnonymous)) {
                newUsersCounter++;
                LOGGER.debug("Tracking user {}", trackedUser);
                if (shouldEnforce()) {
                    LOGGER.warn(
                            "{}, currently tracked users: {}. Enforcement applied for user {} identified by {}",
                            enforcementRule, newUsersCounter, trackingHash,
                            identityHash);
                    blockedUsers
                            .computeIfAbsent(trackingHash, k -> new HashSet<>())
                            .add(identityHash);
                    publishingTask.checkEnforcementUpdates();
                    applyEnforcement(trackingHash, identityHash);
                }
            }
        }

    }

    /**
     * Tells whether new user, i.e. not yet tracked and not yet counted,
     * should be blocked immediately.
     *
     * @return {@literal true} if the current request/user should be blocked,
     *     {@literal false} otherwise.
     */
    boolean shouldEnforceThisUser() {
        synchronized (storage) {
            return shouldEnforce(newUsersCounter + 1);
        }
    }

    private boolean shouldEnforce() {
        synchronized (storage) {
            return shouldEnforce(newUsersCounter);
        }
    }

    private boolean shouldEnforce(int users) {
        return enforcementRule.shouldEnforce(users);
    }

    private static void applyEnforcement(String trackingHash,
            String identityHash) {
        throw new EnforcementException(
                "Daily active user limit exceeded. Enforcement applied to user "
                        + trackingHash
                        + ((identityHash != null)
                                ? " with identity " + identityHash
                                : ""));
    }

    /**
     * Publishes DAU updates to the Vaadin License Server.
     * <p>
     * </p>
     * Sends current DAU data to the License Server and clears the in-memory
     * data set.
     */
    void publishUpdates(PublishingPhase phase) {
        LOGGER.debug("Publishing {} DAU updates", phase);
        if (phase != PublishingPhase.START) {
            synchronized (storage) {
                publishingTask.addAll(storage.values());
                newUsersCounter = 0;
                if (phase == PublishingPhase.STOP) {
                    clearStorage();
                } else {
                    flushInactiveUsers();
                }
            }
        }
        publishingTask.cancelRetry();
        try {
            updateEnforcementRule(publishingTask.apply(phase), Instant.now());
        } catch (Exception ex) {
            // On start fail immediately if the license is invalid, so server
            // will crash
            if (phase == PublishingPhase.START
                    && ex instanceof LicenseException) {
                throw (LicenseException) ex;
            }
            publishingTask.retry(phase, 1, ex);
        }
    }

    private void clearStorage() {
        storage.clear();
        blockedUsers.clear();
    }

    private void flushInactiveUsers() {
        ZonedDateTime dateTimeNowUTC = Instant.now().atZone(ZoneId.of("UTC"));
        Instant midnightUTC = dateTimeNowUTC.toLocalDate().atStartOfDay()
                .toInstant(ZoneOffset.UTC);

        HashSet<String> removedUsers = new HashSet<>(storage.size());

        // Retains active users, i.e. who started working with application today
        // (after latest UTC midnight). Removes not active users (who worked
        // yesterday).
        storage.entrySet().removeIf(entry -> {
            Instant creationTime = entry.getValue().getCreationTime();
            boolean removed = creationTime.isBefore(midnightUTC);
            if (removed) {
                removedUsers.add(entry.getKey());
            }
            return removed;
        });

        // Remove same users from the cache for blocked users
        blockedUsers.entrySet().removeIf(entry -> removedUsers.contains(
                entry.getKey()));
    }

    // Updates enforcement rule if it has not already been updated by later
    // publishing execution. Can very rarely happen if a retry task completes
    // after a sync task.
    private void updateEnforcementRule(DAUServerResponse serverResponse,
            Instant publishingTime) {
        EnforcementRule enforcementRule = serverResponse.getEnforcementRule();
        if (enforcementRule != null && (enforcementRuleUpdatedAt == null
                || enforcementRuleUpdatedAt.isBefore(publishingTime))) {
            this.enforcementRule = enforcementRule;
            if (!enforcementRule.isActiveEnforcement()) {
                // Vaadin should not block any user, when server drops the
                // enforcement, e.g. after DAU limit update.
                blockedUsers.clear();
            }
            enforcementRuleUpdatedAt = publishingTime;
            if (serverResponse.getTrackedUsers() != null && !serverResponse
                    .getTrackedUsers().isEmpty()) {
                storage.putAll(serverResponse.getTrackedUsers().stream()
                        .collect(Collectors.toMap(TrackedUser::getTrackingHash,
                                Function.identity())));
            }
            LOGGER.debug("Publishing DAU updates completed: {}",
                    enforcementRule);
        }
    }

    /**
     * Contacts License Server to check if current subscription key is valid.
     *
     * @throws LicenseException
     *             if a license is required but not available or invalid.
     */
    void checkSubscriptionKey() {
        try {
            publisher.publish("vaadin-license-checker", Collections.emptySet(),
                    PublishingPhase.CHECK);
        } catch (LicenseException ex) {
            throw ex;
        } catch (PublishingException ex) {
            Throwable cause = ex.getCause();
            if (cause != null) {
                throw new LicenseValidationException(ex.getMessage(), cause);
            }
            throw new LicenseValidationException(ex.getMessage());
        } catch (Exception ex) {
            throw new LicenseException(ex.getMessage(), ex);
        }
    }

    private class PublishingTask implements Function<PublishingPhase, DAUServerResponse> {
        private final List<TrackedUser> unsentData = new ArrayList<>();
        private CompletableFuture<DAUServerResponse> retryFuture;
        private Long lastDelay;

        private void addAll(Collection<TrackedUser> data) {
            synchronized (unsentData) {
                unsentData.addAll(data);
            }
        }

        // If enforcement is ON and new users are blocked, retry with a
        // fixed delay to get an answer from License Server as soon as
        // possible. Otherwise, apply a backoff multiplier to avoid useless
        // pressure on License Server
        private CompletableFuture<DAUServerResponse> scheduleNextAttempt(
                int attempt, PublishingPhase phase, LongConsumer logger) {
            long retryDelay;
            long defaultDelay = publishingSettings.retryInterval.toMillis();
            if (shouldEnforce()) {
                retryDelay = defaultDelay;
            } else {
                long maxDelay = publishingSettings.publishingInterval.toMillis()
                        / 2;
                retryDelay = lastDelay != null ? lastDelay : defaultDelay;
                retryDelay += (long) ((attempt - 1) * defaultDelay * 0.5);
                retryDelay = Math.min(retryDelay, maxDelay);
                lastDelay = retryDelay;
            }
            logger.accept(retryDelay);
            long finalDelay = retryDelay;
            Executor delayedExecutor = command -> executorService
                    .schedule(command, finalDelay, TimeUnit.MILLISECONDS);
            return CompletableFuture.supplyAsync(() -> publishingTask.apply(
                    phase), delayedExecutor);

        }

        private void retry(PublishingPhase phase, int attempt,
                Throwable error) {
            // Do not retry on stop
            if (phase == PublishingPhase.STOP) {
                LOGGER.warn("Failed to publish {} DAU updates.", phase, error);
                return;
            }

            Instant publishingTime = Instant.now();
            retryFuture = scheduleNextAttempt(attempt, phase, retryDelay -> {
                if (phase == PublishingPhase.ENFORCE) {
                    LOGGER.warn(
                            "Enforcement is ON. Scheduling re-check attempt {} at {}",
                            attempt, LocalTime.now().plus(retryDelay,
                                    ChronoUnit.MILLIS));
                } else {
                    LOGGER.warn(
                            "Attempt {} to publish {} DAU updates failed. Scheduling retry at {}",
                            attempt, phase,
                            LocalTime.now().plus(retryDelay, ChronoUnit.MILLIS),
                            error);
                }
            });
            retryFuture.whenComplete((result, ex) -> {
                if (ex instanceof CancellationException) {
                    // ignore cancelled tasks
                    return;
                }
                if (ex != null) {
                    retry(phase, attempt + 1, ex);
                } else {
                    cancelRetry();
                    updateEnforcementRule(result, publishingTime);
                    if (phase == PublishingPhase.ENFORCE
                        && shouldEnforce()) {
                        // Enforcement still ON, check again
                        retry(phase, attempt + 1, null);
                    }
                }
            });
        }

        private void cancelRetry() {
            if (retryFuture != null && !retryFuture.isDone()) {
                LOGGER.debug("Retry request cancelled");
                retryFuture.cancel(true);
            }
            lastDelay = null;
            retryFuture = null;
        }

        @Override
        public DAUServerResponse apply(PublishingPhase phase) {
            List<TrackedUser> copy;
            synchronized (unsentData) {
                copy = new ArrayList<>(unsentData);
                unsentData.clear();
            }
            try {
                return publisher.publish(applicationName, copy, phase);
            } catch (Exception ex) {
                synchronized (unsentData) {
                    unsentData.addAll(0, copy);
                }
                if (ex instanceof PublishingException
                        || ex instanceof LicenseException) {
                    throw ex;
                }
                throw new PublishingException(
                        "Publishing failed with an unexpected exception", ex);
            } finally {
                copy.clear();
            }
        }

        private void checkEnforcementUpdates() {
            cancelRetry();
            retry(PublishingPhase.ENFORCE, 1, null);
        }
    }

    /**
     * Publishing settings.
     */
    static class PublishingSettings {
        private final Duration publishingInterval;
        private final Duration retryInterval;

        /**
         * Creates publishing configuration.
         *
         * @param publishingInterval
         *            the default publishing scheduled interval
         * @param retryInterval
         *            time to wait before retry to publish data after a failed
         *            attempt.
         */
        PublishingSettings(Duration publishingInterval,
                Duration retryInterval) {
            this.publishingInterval = Objects.requireNonNull(publishingInterval,
                    "publishingInterval must not be null");
            this.retryInterval = Objects.requireNonNull(retryInterval,
                    "retryInterval must not be null");
        }
    }

    private static ThreadFactory dauThreadFactory() {
        ThreadFactory wrapped = Executors.defaultThreadFactory();
        return r -> {
            Thread thread = wrapped.newThread(r);
            thread.setName("DAU-Publisher-" + thread.getId());
            return thread;
        };
    }

}
