/*
 * Copyright 2000-2024 Vaadin Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.vaadin.pro.licensechecker;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;

import org.slf4j.Logger;

import static com.vaadin.pro.licensechecker.LocalProKey.getVaadinFolder;

/**
 * Subscription key loader.
 * <p>
 * </p>
 * A subscription key can be provided in several ways, to fit different needs
 * and environments. The priority order is {@literal Java system property},
 * {@literal environment variable}, {@literal file} in Vaadin home folder.
 *
 * <ol>
 * <li>System property with {@literal vaadin.subscriptionKey} key</li>
 * <li>Environment variable named {@literal VAADIN_SUBSCRIPTION_KEY}</li>
 * <li>File named {@literal subscriptionKey} placed in the
 * {@literal $HOME/.vaadin} folder</li>
 * </ol>
 *
 * For consistency and backward compatibility, the {@literal vaadin.key} system
 * property and {@literal VAADIN_KEY} environment variable ara also supported.
 * In this case the key must start with the {@literal sub-} prefix.
 *
 */
public final class LocalSubscriptionKey {

    private static final String VAADIN_SUBSCRIPTION_KEY_SYSTEM_PROPERTY = "vaadin.subscriptionKey";
    private static final String VAADIN_SUBSCRIPTION_KEY_ENV_VAR = "VAADIN_SUBSCRIPTION_KEY";

    private static SubscriptionKey read(File subscriptionKeyLocation)
            throws IOException {
        if (!subscriptionKeyLocation.exists()) {
            return null;
        }
        try (FileInputStream is = new FileInputStream(
                subscriptionKeyLocation)) {
            return new SubscriptionKey(Util.toString(is));
        }
    }

    /**
     * Gets the current subscription key, if found.
     * <p>
     * </p>
     * A valid subscription key is searched in system properties, environment
     * variables and filesystem. See class level Javadoc for details on how the
     * key can be provided.
     *
     * @return the subscription key, or {@literal null} if the key cannot be
     *         found.
     */
    public static SubscriptionKey get() {
        SubscriptionKey subscriptionKey = getSystemProperty();
        if (subscriptionKey != null) {
            getLogger().debug("Using subscription key from system property");
            return subscriptionKey;
        }
        subscriptionKey = getEnvironmentVariable();
        if (subscriptionKey != null) {
            getLogger()
                    .debug("Using subscription key from environment variable");
            return subscriptionKey;
        }
        File subscriptionKeyLocation = new File(getVaadinFolder(),
                "subscriptionKey");
        try {
            subscriptionKey = read(subscriptionKeyLocation);
            if (subscriptionKey != null) {
                getLogger().debug(
                        "Found subscription key in " + subscriptionKeyLocation);
                return subscriptionKey;
            }
            return getFromLocalOfflineKey();
        } catch (IOException e) {
            getLogger().debug("Unable to read subscription key", e);
            return getFromLocalOfflineKey();
        }
    }

    private static SubscriptionKey getFromLocalOfflineKey() {
        OfflineKey offlineKey = LocalOfflineKey.get();
        if (offlineKey != null) {
            String key = offlineKey.getSubscriptionKey();
            if (key != null) {
                getLogger().debug("Using subscription key from offline key");
                return new SubscriptionKey(key);
            }
        }
        return null;
    }

    /**
     * Gets the current subscription key or throws an exception if not
     * available.
     * <p>
     * </p>
     * A valid subscription key is searched in system properties, environment
     * variables and filesystem. See class level Javadoc for details on how the
     * key can be provided.
     *
     * @return the subscription key.
     * @throws LicenseException
     *             if the subscription key cannot be found.
     */
    public static SubscriptionKey getOrFail() {
        SubscriptionKey subscriptionKey = get();
        if (subscriptionKey == null) {
            throw new LicenseException(
                    "A subscription key is required for Daily Active User tracking, but it was not found. "
                            + "Please provide a valid subscription key by either setting "
                            + VAADIN_SUBSCRIPTION_KEY_ENV_VAR
                            + " environment variable, " + "-D"
                            + VAADIN_SUBSCRIPTION_KEY_SYSTEM_PROPERTY
                            + " system property or a $HOME/.vaadin/subscriptionKey file.");
        }
        return subscriptionKey;
    }

    private static SubscriptionKey getSystemProperty() {
        String value = System
                .getProperty(VAADIN_SUBSCRIPTION_KEY_SYSTEM_PROPERTY);
        if (value == null) {
            value = System.getProperty("vaadin.key");
            if (!isSubscriptionKey(value)) {
                return null;
            }
        }
        return parseSubscriptionKey(value, "system property");
    }

    private static SubscriptionKey getEnvironmentVariable() {
        String value = EnvironmentVariables
                .get(VAADIN_SUBSCRIPTION_KEY_ENV_VAR);
        if (value == null) {
            value = EnvironmentVariables.get("VAADIN_KEY");
            if (!isSubscriptionKey(value)) {
                return null;
            }
        }

        return parseSubscriptionKey(value, "environment variable");
    }

    private static SubscriptionKey parseSubscriptionKey(String value,
            String source) {
        if (!value.startsWith("sub-")) {
            getLogger().warn(
                    "Invalid subscription key format in the {}. "
                            + "The value must be of type [sub-123]. Was {}",
                    source, value);
            return null;
        }

        return new SubscriptionKey(value);
    }

    private static boolean isSubscriptionKey(String value) {
        return value != null && value.startsWith("sub-");
    }

    private static Logger getLogger() {
        return LicenseChecker.getLogger();
    }

}
