package com.vaadin.copilot.plugins.docs;

import java.io.IOException;
import java.net.URI;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.WeakHashMap;

import org.apache.commons.lang3.StringUtils;
import org.jsoup.Jsoup;
import org.jsoup.nodes.Document;
import org.jsoup.nodes.Element;
import org.jsoup.nodes.Node;
import org.jsoup.select.Elements;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DocsDataLoader {
    private static final String TARGET = "target";
    private static final String TARGET_BLANK = "_blank";
    private final Map<DocsRequest, DocsResponse> parsed = new WeakHashMap<>();

    /**
     * Contains entries of element tags that need to be replaced in order to load
     * the related page.
     */
    private final Map<String, String> docTagNameReplacementMap = Map.of("avatar-group", "avatar", "checkbox-group",
            "checkbox", "radio-group", "radio-button", "chart", "charts");

    /**
     * Gets the documentation of requested component or throws exception on failure
     * cases.
     *
     * @param docsRequest
     * @return Documentation
     * @throws DocsException
     */
    public DocsResponse get(DocsRequest docsRequest) throws DocsException {
        if (docsRequest.isCustomComponent()) {
            throw new DocsException(DocsException.CUSTOM_COMPONENT_NOT_SUPPORTED);
        }
        if (!isVaadinComponent(docsRequest.getTag())) {
            throw new DocsException(DocsException.NOT_A_VAADIN_COMPONENT);
        }
        if (parsed.containsKey(docsRequest)) {
            return parsed.get(docsRequest);
        }
        fetchAndCache(docsRequest);
        return parsed.get(docsRequest);
    }

    private void fetchAndCache(DocsRequest docsRequest) throws DocsException {
        String url = getUrl(docsRequest.getTag());
        try {
            Document document = fetch(url);
            parsed.put(docsRequest, parse(document, url));
        } catch (org.jsoup.HttpStatusException ex) {
            if (ex.getStatusCode() == 404) {
                throw new DocsException(DocsException.DOCUMENTATION_PAGE_NOT_FOUND);
            }
            if (ex.getStatusCode() == 503) {
                throw new DocsException(DocsException.DOCUMENTATION_PAGE_SERVICE_NOT_AVAILABLE);
            }
            throw new DocsException(DocsException.UNKNOWN_ERROR);
        } catch (Exception ex) {
            throw new DocsException(DocsException.UNKNOWN_ERROR);
        }
    }

    private Document fetch(String url) throws IOException {
        return Jsoup.connect(url).timeout(10 * 1000).get();
    }

    private DocsResponse parse(Document document, String url) {
        DocsResponse docsResponse = new DocsResponse();
        docsResponse.setDocumentationPageUrl(url);
        Element mainElement = document.select("main").first();
        if (mainElement == null) {
            return docsResponse;
        }
        filterElements(mainElement);
        fixUrls(url, mainElement);
        replaceCodeBlocksShowExampleLinks(url, mainElement);
        Element header = mainElement.selectFirst("header");
        parseHeader(docsResponse, header);
        for (DocsMainSection mainSection : docsResponse.getMainSectionList()) {
            if (mainSection.getUrl() != null) {
                try {
                    Document mainSectionDoc = fetch(mainSection.getUrl());
                    Element main = mainSectionDoc.select("main").first();
                    if (main != null) {
                        filterElements(main);
                        fixUrls(mainSection.getUrl(), main);
                        replaceCodeBlocksShowExampleLinks(url, main);
                        parseSections(mainSection, main);
                    }
                } catch (Exception ex) {
                    getLogger().error(ex.getMessage(), ex);
                }
            } else {
                parseSections(mainSection, mainElement);
            }
        }
        parseLinks(docsResponse, header);
        filterSections(docsResponse);
        return docsResponse;
    }

    private void filterSections(DocsResponse docsResponse) {
        for (DocsMainSection mainSection : docsResponse.getMainSectionList()) {
            mainSection.getSections().removeIf(f -> StringUtils.equals(f.getTitle(), "Related Components"));
        }
    }

    private void parseHeader(DocsResponse docsResponse, Element header) {
        if (header == null) {
            return;
        }

        Element mainSectionHeader = header.selectFirst("ul");
        if (mainSectionHeader == null) {
            docsResponse.getMainSectionList().add(new DocsMainSection("Usage"));
            return;
        }
        Elements mainSectionElements = mainSectionHeader.select("li");
        for (Element mainSectionElement : mainSectionElements) {
            Element element = mainSectionElement.firstElementChild();
            if (element != null) {
                DocsMainSection mainSection = new DocsMainSection();
                if ("span".equals(element.tagName())) {
                    mainSection.setTitle(element.text());
                } else if ("a".equals(element.tagName())) {
                    mainSection.setUrl(element.attr("href"));
                    mainSection.setTitle(element.text());
                }
                docsResponse.getMainSectionList().add(mainSection);
            }
        }
    }

    private void parseLinks(DocsResponse docsResponse, Element header) {
        if (header == null) {
            return;
        }
        Optional<Element> linksElement = findByClassNameContainsRecursively(header, "PageHeader-module--links");

        if (linksElement.isPresent()) {
            Elements links = linksElement.get().select("a");
            for (Element link : links) {
                if (!link.hasAttr(TARGET)) {
                    link.attr(TARGET, TARGET_BLANK);
                }
            }
            // API and Source
            DocsMainSection linksSection = new DocsMainSection("Links");
            DocsSection section = new DocsSection();
            section.setLinks(true);
            section.setHtmlContent(linksElement.get().outerHtml());
            linksSection.getSections().add(section);
            docsResponse.getMainSectionList().add(linksSection);
        }
    }

    private void parseSections(DocsMainSection mainSection, Element mainElement) {
        Elements sectionElements = getSectionElements(mainElement);
        for (Element sectionElement : sectionElements) {
            DocsSection section = new DocsSection();
            Element maybeH2Element = sectionElement.firstElementChild();
            if (maybeH2Element != null && "h2".equals(maybeH2Element.tagName())) {
                section.setTitle(maybeH2Element.text());
            }
            Element first = sectionElement.select("div.sectionbody").first();
            if (first != null) {
                section.setHtmlContent(first.outerHtml());
            }

            mainSection.getSections().add(section);
        }
    }

    private Elements getSectionElements(Element mainElement) {
        if (mainElement == null) {
            return new Elements();
        }
        Element article = mainElement.selectFirst("article");
        if (article == null) {
            return new Elements();
        }
        Element preamble = article.selectFirst("div#preamble");
        Elements sections = article.select("div.sect1");
        Elements elements = new Elements();
        if (preamble != null) {
            elements.add(preamble);
        }
        elements.addAll(sections);
        return elements;
    }

    private void filterElements(Element mainElement) {
        Element discussionIdElement = mainElement.select("code.discussion-id").first();
        if (discussionIdElement != null && discussionIdElement.parent() != null) {
            if (discussionIdElement.parent().parent() != null) {
                discussionIdElement.parent().parent().remove();
            } else {
                discussionIdElement.parent().remove();
            }
        }
        Elements commercialNotes = mainElement.select("div.note.commercial");
        commercialNotes.remove();
    }

    private void fixUrls(String pageUrl, Element mainElement) {
        URI uri = URI.create(pageUrl);
        Elements anchors = mainElement.select("a");
        for (Element anchor : anchors) {
            if (anchor.hasAttr("href")) {
                String href = anchor.attr("href");
                if (href.startsWith("/")) {
                    String scheme = uri.getScheme();
                    String host = uri.getHost();
                    anchor.attr("href", scheme + "://" + host + href);
                } else if (href.startsWith("#")) {
                    anchor.attr("href", pageUrl + href);
                }
            }
            anchor.attr("target", TARGET_BLANK);
        }
    }

    private boolean isVaadinComponent(String elementTagName) {
        return elementTagName.toLowerCase(Locale.ENGLISH).startsWith("vaadin");
    }

    private String getUrl(String elementTagName) {
        String docTagName = elementTagName.toLowerCase(Locale.ENGLISH).replace("vaadin-", "");
        if (docTagNameReplacementMap.containsKey(docTagName)) {
            docTagName = docTagNameReplacementMap.get(docTagName);
        }
        return "https://vaadin.com/docs/latest/components/" + docTagName + "?copilot-request";
    }

    private Optional<Element> findByClassNameContainsRecursively(Element parent, String className) {
        for (Node childNode : parent.childNodes()) {
            if (childNode instanceof Element element && element.className().contains(className)) {
                return Optional.of(element);
            }
        }
        for (Node childNode : parent.childNodes()) {
            if (childNode instanceof Element element) {
                Optional<Element> foundInChild = findByClassNameContainsRecursively(element, className);
                if (foundInChild.isPresent()) {
                    return foundInChild;
                }
            }
        }
        return Optional.empty();
    }

    private void replaceCodeBlocksShowExampleLinks(String pageUrl, Element mainElement) {
        Elements codeExampleDivs = mainElement.select("div.code-example");
        for (Element codeExampleDiv : codeExampleDivs) {
            for (Node childNode : codeExampleDiv.childNodes()) {
                childNode.remove();
            }
            Element anchorElement = new Element("a");
            Element showExample = anchorElement.text("Show Example");
            showExample.addClass("code-show-example");
            showExample.attr(TARGET, TARGET_BLANK);
            codeExampleDiv.appendChild(anchorElement);
            String sectionAnchorHref = findSectionAnchorHref(codeExampleDiv);
            if (sectionAnchorHref != null) {
                showExample.attr("href", sectionAnchorHref);
            } else {
                showExample.attr("href", pageUrl);
            }
        }
    }

    private String findSectionAnchorHref(Element element) {
        for (Element siblingElement : element.siblingElements()) {
            if ((siblingElement.tagName().equals("h2") || siblingElement.tagName().equals("h3"))
                    && siblingElement.selectFirst("a.anchor") != null) {
                return siblingElement.selectFirst("a.anchor").attr("href");
            }
        }
        Element parent = element.parent();
        if (parent != null && !"main".equals(parent.tagName())) {
            return findSectionAnchorHref(parent);
        }
        return null;
    }

    protected Logger getLogger() {
        return LoggerFactory.getLogger(getClass());
    }
}
