package com.vaadin.copilot.javarewriter;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import com.vaadin.copilot.ProjectManager;
import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.internal.ComponentTracker;

import com.github.javaparser.StaticJavaParser;
import com.github.javaparser.ast.CompilationUnit;
import com.github.javaparser.ast.NodeList;
import com.github.javaparser.ast.body.ClassOrInterfaceDeclaration;
import com.github.javaparser.ast.body.MethodDeclaration;
import com.github.javaparser.ast.expr.BinaryExpr;
import com.github.javaparser.ast.expr.BooleanLiteralExpr;
import com.github.javaparser.ast.expr.Expression;
import com.github.javaparser.ast.expr.FieldAccessExpr;
import com.github.javaparser.ast.expr.IntegerLiteralExpr;
import com.github.javaparser.ast.expr.MethodCallExpr;
import com.github.javaparser.ast.expr.MethodReferenceExpr;
import com.github.javaparser.ast.expr.NameExpr;
import com.github.javaparser.ast.expr.ObjectCreationExpr;
import com.github.javaparser.ast.expr.StringLiteralExpr;
import com.github.javaparser.ast.stmt.BlockStmt;
import com.github.javaparser.ast.stmt.ExpressionStmt;
import com.github.javaparser.ast.stmt.IfStmt;
import com.github.javaparser.ast.stmt.ReturnStmt;
import com.github.javaparser.ast.type.ClassOrInterfaceType;
import com.github.javaparser.ast.type.Type;

/**
 * Handler for data provider changes. Any updates on component which are
 * generated from DataProvider is handled in this class.
 */
public class JavaDataProviderHandler {
    private static final String LABEL_KEY = "label";
    private static final String VALUE_KEY = "value";
    private static final String DISABLED_KEY = "disabled";
    private static final String SET_ITEMS_METHOD = "setItems";
    private static final String PARENT = "parent";
    private static final String ENABLED = "enabled";

    /**
     * Returns the file that is changed along with the new result
     *
     * @param file
     *            File that is changed
     * @param result
     *            Result text after modifications
     */
    public record JavaDataProviderHandlerResult(File file, String result) {
    }

    public record FieldOrVariable(Expression reference, Type type, ObjectCreationExpr initializer) {
    }

    private final JavaRewriter javaRewriter;
    private final ComponentTypeAndSourceLocation typeAndSourceLocation;
    private final File sourceFile;

    /**
     * Reads source file which is found from typeAndSourceLocation
     *
     * @param projectManager
     *            Project manager for handling file operations
     * @param typeAndSourceLocation
     *            of the component
     * @throws IOException
     *             when file read operation fails
     */
    public JavaDataProviderHandler(ProjectManager projectManager, ComponentTypeAndSourceLocation typeAndSourceLocation)
            throws IOException {
        this.typeAndSourceLocation = typeAndSourceLocation;
        ComponentTypeAndSourceLocation parent = typeAndSourceLocation.parent();
        if (parent == null) {
            throw new IllegalArgumentException("Could not find the parent of the component");
        }
        sourceFile = projectManager.getSourceFile(parent.createLocation());
        javaRewriter = new JavaRewriter(projectManager.readFile(sourceFile));
    }

    /**
     * Check for attach and create location is generated from DataProvider
     *
     * @param componentTypeAndSourceLocation
     *            component type and source location
     * @return true if data provider operation, false otherwise
     */
    public static boolean isDataProviderItemChange(ComponentTypeAndSourceLocation componentTypeAndSourceLocation) {
        ComponentTracker.Location createLocation = componentTypeAndSourceLocation.createLocation();
        ComponentTracker.Location attachLocation = componentTypeAndSourceLocation.attachLocation();
        if (createLocation != null && attachLocation != null) {
            return createLocation.className().startsWith("com.vaadin.flow.data.provider")
                    && attachLocation.className().startsWith("com.vaadin.flow.data.provider");
        }
        return false;
    }

    /**
     * Replaces the value of items from data provider
     *
     * @param property
     *            {@code} only label is support for now which is sent from client
     * @param value
     *            to replace
     * @return Final result or exception if any occurs
     */
    public JavaDataProviderHandlerResult handleSetComponentProperty(String property, String value) {
        // radio button information
        if (!LABEL_KEY.equals(property)) {
            throw new IllegalArgumentException("Unknown property" + property);
        }
        ComponentInfo parentInfo = javaRewriter.findComponentInfo(typeAndSourceLocation.parent());
        Component component = typeAndSourceLocation.component();
        Object item = getItem(component);

        Optional<MethodCallExpr> setItemsCall = JavaRewriterUtil.findMethodCallStatements(parentInfo).stream()
                .filter(f -> f.getNameAsString().equals(SET_ITEMS_METHOD)).findFirst();
        Expression expressionToReplace;
        if (setItemsCall.isPresent()) {
            expressionToReplace = findItemExpression(setItemsCall.get(), item);
        } else {
            expressionToReplace = findItemExpression(parentInfo.objectCreationExpr(), item);
        }
        if (expressionToReplace == null) {
            throw new IllegalArgumentException("Could not find node to replace");
        }
        // when item is POJO.
        if (expressionToReplace.isObjectCreationExpr()) {
            NodeList<Expression> arguments = expressionToReplace.asObjectCreationExpr().getArguments();

            Expression labelArg = arguments.get(0);
            Expression valueArg = arguments.get(1);
            boolean valueEqualsToLabel = JavaRewriterUtil.equalsByNameAsString(labelArg, valueArg);
            Expression newExpression = JavaRewriterUtil.toExpression(value);
            boolean replace = labelArg.replace(newExpression);
            if (!replace) {
                throw new IllegalArgumentException("Could not replace label expression");
            }
            if (valueEqualsToLabel) {
                // replace the value if it is equal to label.
                replace = valueArg.replace(newExpression);
                if (!replace) {
                    throw new IllegalArgumentException("Could not replace value expression");
                }
            }

        } else {
            // when item is primitive type
            Expression expression = JavaRewriterUtil.toExpression(value);
            boolean replace = expressionToReplace.replace(expression);
            if (!replace) {
                throw new IllegalArgumentException("Could not replace expression");
            }
        }
        return new JavaDataProviderHandlerResult(sourceFile, javaRewriter.getResult());
    }

    /**
     * @param searchExpression
     *            {@code setItems} or {@code constructor}
     * @param item
     *            from the component
     * @return Expression to be replaced, null if not found
     */
    private Expression findItemExpression(Expression searchExpression, Object item) {
        List<Expression> argsToSearch = new ArrayList<>();
        if (searchExpression.isMethodCallExpr()) {
            argsToSearch = new ArrayList<>(searchExpression.asMethodCallExpr().getArguments());
        } else if (searchExpression.isObjectCreationExpr()) {
            argsToSearch = new ArrayList<>(searchExpression.asObjectCreationExpr().getArguments());
            // excluding label
            if (!argsToSearch.isEmpty()) {
                argsToSearch.remove(0);
            }
        }
        Optional<LabelValueDisabled> labelValueRecordOptional = getLabelValueRecord(item);

        if (labelValueRecordOptional.isPresent()) {
            LabelValueDisabled labelValueRecord = labelValueRecordOptional.get();
            List<Expression> foundArgs = new ArrayList<>();
            List<ObjectCreationExpr> objectCreationExprs = argsToSearch.stream()
                    .filter(expression -> expression.isObjectCreationExpr()
                            && !expression.asObjectCreationExpr().getArguments().isEmpty())
                    .map(Expression::asObjectCreationExpr).toList();

            for (ObjectCreationExpr objectCreationExpr : objectCreationExprs) {
                Expression maybeLabelNode = objectCreationExpr.getArguments().get(0);
                Expression maybeValueNode = objectCreationExpr.getArguments().get(1);
                if (maybeLabelNode.isStringLiteralExpr()
                        && JavaRewriterUtil.equalsByNameAsString(maybeLabelNode,
                                new StringLiteralExpr(labelValueRecord.label))
                        && JavaRewriterUtil.equalsByNameAsString(maybeValueNode,
                                new StringLiteralExpr(labelValueRecord.value))) {
                    foundArgs.add(objectCreationExpr);
                }
            }
            if (foundArgs.size() > 1) {
                // thrown for situations like setItems(new
                // LabelAndValue("samelabel", "1"), new
                // LabelAndValue("samelabel", "1"));
                throw new IllegalArgumentException("More than one equal label-value pair is found");
            }
            if (foundArgs.size() == 1) {
                return foundArgs.get(0);
            }
            return null;
        }
        Expression searchingExpression = JavaRewriterUtil.toExpression(item);
        for (Expression arg : argsToSearch) {
            if (JavaRewriterUtil.typesEqual(arg.calculateResolvedType(), item.getClass())
                    && JavaRewriterUtil.equalsByNameAsString(searchingExpression, arg)) {
                return arg;
            }
        }
        return null;
    }

    /**
     * Checks item type is generated by the copilot
     *
     * @param item
     *            from Component
     * @return the record definition if found, empty otherwise.
     */
    private Optional<LabelValueDisabled> getLabelValueRecord(Object item) {
        Field[] declaredFields = item.getClass().getDeclaredFields();
        // exact equal check.
        if (declaredFields.length != 3) {
            return Optional.empty();
        }
        if (hasField(item.getClass(), LABEL_KEY) && hasField(item.getClass(), VALUE_KEY)
                && hasField(item.getClass(), ENABLED)) {
            String label = getFieldValue(item, LABEL_KEY);
            String value = getFieldValue(item, VALUE_KEY);
            boolean enabled = Boolean.TRUE.equals(getFieldValue(item, ENABLED));
            return Optional.of(new LabelValueDisabled(label, value, !enabled));
        }
        return Optional.empty();
    }

    private <T> T getFieldValue(Object object, String fieldName) {
        try {
            Field declaredField = object.getClass().getDeclaredField(fieldName);
            declaredField.setAccessible(true);
            return (T) declaredField.get(object);
        } catch (Exception ex) {
            return null;
        }
    }

    private boolean hasField(Class<?> clazz, String fieldName) {
        try {
            clazz.getDeclaredField(fieldName);
            return true;
        } catch (NoSuchFieldException e) {
            return false;
        }
    }

    /**
     * Gets the item field from the component using reflection API.
     *
     * @param component
     *            Flow component
     * @return field value or exception
     */
    private Object getItem(Component component) {
        try {
            Field item = component.getClass().getDeclaredField("item");
            item.setAccessible(true);
            return item.get(component);
        } catch (Exception ex) {
            throw new IllegalArgumentException("Could not find item of the component", ex);
        }
    }

    /**
     * Creates the data statements when adding a new template e.g.{@code setItems},
     * {@code setItemLabelGenerator} etc... and clears children or props if they are
     * definition of data instead of real components.
     *
     * @param compilationUnit
     *            the compilation unit to look for the record definitions.
     * @param parent
     *            The parent variable that has items
     * @param javaComponent
     *            component that has children and properties.
     * @param insertionPoint
     *            to insert new method calls
     */
    public static void handleDataStatementsAndClearDataProps(CompilationUnit compilationUnit, FieldOrVariable parent,
            JavaComponent javaComponent, InsertionPoint insertionPoint, String dataEntityRecordName) {
        // This list is not empty if the items of the component are generated
        // from the children of the component.
        List<JavaComponent> itemComponents = javaComponent.children().stream()
                .filter(child -> FlowComponentQuirks.childrenGeneratesData(javaComponent, child)).toList();

        if (!itemComponents.isEmpty()) {
            if (javaComponent.tag().equals("Grid")) {
                addEntityDataStatements(compilationUnit, parent, insertionPoint, javaComponent.getItemsFromProperty(),
                        dataEntityRecordName);
            } else if (javaComponent.tag().equalsIgnoreCase("TreeGrid")) {
                addTreeGridEntityDataStatements(compilationUnit, parent, insertionPoint,
                        javaComponent.getItemsFromProperty(), dataEntityRecordName);
            } else {
                addLabelValueDataStatements(compilationUnit, parent, childrenToLabelValueEnabledList(javaComponent),
                        insertionPoint);
            }
            javaComponent.children().removeAll(itemComponents);
        }
        boolean hasSetItemsProps = FlowComponentQuirks.hasSetItemsProps(javaComponent);
        if (hasSetItemsProps) {
            if (javaComponent.tag().equals("Grid")) {
                addEntityDataStatements(compilationUnit, parent, insertionPoint, javaComponent.getItemsFromProperty(),
                        dataEntityRecordName);
            } else if (javaComponent.tag().equals("TreeGrid")) {
                addTreeGridEntityDataStatements(compilationUnit, parent, insertionPoint,
                        javaComponent.getItemsFromProperty(), dataEntityRecordName);
            } else {
                addLabelValueDataStatements(compilationUnit, parent, propsToLabelValueDisabledList(javaComponent),
                        insertionPoint);
            }
        }
    }

    private static void addLabelValueDataStatements(CompilationUnit compilationUnit, FieldOrVariable parent,
            List<LabelValueDisabled> labelValueDisabledList, InsertionPoint insertionPoint) {
        LabelValuePairRecordRewriter javaClassOrRecordRewriter = new LabelValuePairRecordRewriter(compilationUnit);
        String recordName = javaClassOrRecordRewriter.addOrGetLabelValueEnabledRecordName();

        ClassOrInterfaceType recordType = new ClassOrInterfaceType();
        recordType.setName(recordName);
        updateVariableOrFieldDeclarationType(parent, recordType);

        boolean anyDisabled = false;
        MethodCallExpr setItemCallExpr = new MethodCallExpr(parent.reference(), SET_ITEMS_METHOD);

        for (LabelValueDisabled labelValueDisabled : labelValueDisabledList) {
            anyDisabled = labelValueDisabled.disabled || anyDisabled;
            NodeList<Expression> args = new NodeList<>();
            ObjectCreationExpr newItemExpr = new ObjectCreationExpr(null, recordType, args);
            args.add(new StringLiteralExpr(labelValueDisabled.label));
            args.add(new StringLiteralExpr(labelValueDisabled.value));
            if (Boolean.TRUE.equals(labelValueDisabled.disabled)) {
                args.add(new BooleanLiteralExpr(false));
            }
            setItemCallExpr.addArgument(newItemExpr);
        }
        insertionPoint.add(new ExpressionStmt(setItemCallExpr));
        addSetItemLabelGeneratorStmt(parent, insertionPoint, recordType);
        if (anyDisabled) {
            addSetItemEnabledProviderMethodExpr(parent, insertionPoint, recordType);
        }
    }

    /**
     * Adds data items to a new component using setItems method
     *
     * @param compilationUnit
     *            The compilation unit
     * @param parent
     *            The parent variable which is the component
     * @param insertionPoint
     *            The insertion point to add the new method calls
     * @param items
     *            The items to be added to the component
     */
    private static void addEntityDataStatements(CompilationUnit compilationUnit, FieldOrVariable parent,
            InsertionPoint insertionPoint, List<Map<String, Object>> items, String dataEntityRecordName) {
        if (items.isEmpty()) {
            return;
        }
        DataEntityRecordRewriter dataEntityRecordRewriter = new DataEntityRecordRewriter(compilationUnit);
        String recordName = dataEntityRecordRewriter.addOrGetDataEntityRecordName(items.get(0), dataEntityRecordName);

        ClassOrInterfaceType recordType = new ClassOrInterfaceType();
        recordType.setName(recordName);
        updateVariableOrFieldDeclarationType(parent, recordType);

        MethodCallExpr setItemCallExpr = new MethodCallExpr(parent.reference(), SET_ITEMS_METHOD);

        for (Map<String, Object> item : items) {
            NodeList<Expression> args = new NodeList<>();
            ObjectCreationExpr newItemExpr = new ObjectCreationExpr(null, recordType, args);
            for (Map.Entry<String, Object> itemValues : item.entrySet()) {
                if (itemValues.getValue() instanceof Integer) {
                    args.add(new IntegerLiteralExpr(itemValues.getValue().toString()));
                } else {
                    args.add(new StringLiteralExpr(itemValues.getValue().toString()));
                }
            }
            setItemCallExpr.addArgument(newItemExpr);
        }
        insertionPoint.add(new ExpressionStmt(setItemCallExpr));
    }

    /**
     * Adds data items to a new TreeGrid using the method
     * HasHierarchyDataProvider::setItems(Collection<T> rootItems,
     * ValueProvider<T,Collection<T>> childItemProvider). It creates the
     * childItemProvider method as a fake data service provider
     *
     * @param compilationUnit
     *            The compilation unit
     * @param parent
     *            The parent variable which is the TreeGrid
     * @param insertionPoint
     *            The insertion point to add the new method calls
     * @param items
     *            The items to be added to the TreeGrid
     * @param dataEntityRecordName
     *            The name of the record to be used for the data entity
     */
    private static void addTreeGridEntityDataStatements(CompilationUnit compilationUnit, FieldOrVariable parent,
            InsertionPoint insertionPoint, List<Map<String, Object>> items, String dataEntityRecordName) {
        DataEntityRecordRewriter dataEntityRecordRewriter = new DataEntityRecordRewriter(compilationUnit);
        String recordName = dataEntityRecordRewriter.addOrGetDataEntityRecordName(items.get(0), dataEntityRecordName);

        ClassOrInterfaceType recordType = new ClassOrInterfaceType();
        recordType.setName(recordName);
        updateVariableOrFieldDeclarationType(parent, recordType);

        MethodCallExpr setItemCallExpr = new MethodCallExpr(parent.reference(), SET_ITEMS_METHOD);

        NameExpr staticArrayClass = new NameExpr("Arrays");
        MethodCallExpr arraysAsList = new MethodCallExpr(staticArrayClass, "asList");
        MethodCallExpr arraysAsListChildren = new MethodCallExpr(staticArrayClass, "asList");

        for (Map<String, Object> item : items) {
            NodeList<Expression> args = new NodeList<>();
            NodeList<Expression> getChildrenArgs = new NodeList<>();
            ObjectCreationExpr newItemExpr = new ObjectCreationExpr(null, recordType, args);
            ObjectCreationExpr newChildItemExpr = new ObjectCreationExpr(null, recordType, getChildrenArgs);
            for (Map.Entry<String, Object> itemValues : item.entrySet()) {
                if (itemValues.getKey().equalsIgnoreCase("children")) {
                    continue;
                }
                if (itemValues.getValue() instanceof Integer) {
                    args.add(new IntegerLiteralExpr(itemValues.getValue().toString()));
                    if (itemValues.getKey().equals("id")) {
                        // Create the expression "3 + Integer.valueOf(parent.id
                        // * 10)"
                        FieldAccessExpr parentId = new FieldAccessExpr(new NameExpr(PARENT), "id");
                        BinaryExpr multiplyExpr = new BinaryExpr(parentId, new IntegerLiteralExpr("10"),
                                BinaryExpr.Operator.MULTIPLY);
                        MethodCallExpr valueOfCall = new MethodCallExpr(new NameExpr("Integer"), "valueOf");
                        valueOfCall.addArgument(multiplyExpr);
                        BinaryExpr finalExpr = new BinaryExpr(new IntegerLiteralExpr(itemValues.getValue().toString()),
                                valueOfCall, BinaryExpr.Operator.PLUS);
                        getChildrenArgs.add(finalExpr);
                    } else {
                        getChildrenArgs.add(new IntegerLiteralExpr(itemValues.getValue().toString()));
                    }
                } else {
                    args.add(new StringLiteralExpr(itemValues.getValue().toString()));
                    getChildrenArgs.add(new StringLiteralExpr(itemValues.getValue().toString()));
                }
            }
            arraysAsList.addArgument(newItemExpr);
            arraysAsListChildren.addArgument(newChildItemExpr);
        }
        setItemCallExpr.addArgument(arraysAsList).addArgument("this::get" + dataEntityRecordName + "Children");
        insertionPoint.add(new ExpressionStmt(setItemCallExpr));

        // the method declaration is directly added to the class declaration
        // where the parent is declared
        createGetChildrenMethod(parent, arraysAsListChildren, items.size(), dataEntityRecordName);

        JavaRewriterUtil.addImport(compilationUnit, "java.util.ArrayList");
        JavaRewriterUtil.addImport(compilationUnit, "java.util.Arrays");
        JavaRewriterUtil.addImport(compilationUnit, "java.util.List");
    }

    /**
     * Creates the getChildren method for the TreeGrid data provider with this
     * structure:
     *
     * <pre>
     * public List<DataEntity> getChildren(@NotNull DataEntity parent) {
     *     if (parent.id < 4) {
     *         return Arrays.asList(
     *                 new DataEntity("John", "Doe", "john.doe@gmail.com", "Gardener", "!params.parentItem",
     *                         1 + Integer.valueOf(parent.id * 10)),
     *                 new DataEntity("Jane", "Doe", "jane.doe@gmail.com", "Engineer", "!params.parentItem",
     *                         2 + Integer.valueOf(parent.id * 10)),
     *                 new DataEntity("Bob", "Smith", "bob.smith@gmail.com", "Doctor", "!params.parentItem",
     *                         3 + Integer.valueOf(parent.id * 10)));
     *     } else {
     *         return new ArrayList<>();
     *     }
     * }
     * </pre>
     *
     * @param arraysAsListChildren
     *            The method call expression for Arrays.asList
     * @param itemsSize
     *            The size of the items
     * @param dataEntityRecordName
     *            The name of the record to be used for the data entity
     * @return The method declaration for the getChildren method
     */
    private static MethodDeclaration createGetChildrenMethod(FieldOrVariable parent,
            MethodCallExpr arraysAsListChildren, int itemsSize, String dataEntityRecordName) {

        // Build the declaration
        MethodDeclaration methodDeclaration = JavaRewriterUtil
                .findAncestor(parent.initializer(), ClassOrInterfaceDeclaration.class)
                .addMethod("get" + dataEntityRecordName + "Children");
        methodDeclaration.setType(new ClassOrInterfaceType("List<" + dataEntityRecordName + ">"));
        methodDeclaration.addParameter(dataEntityRecordName, PARENT);

        // Build the body
        Expression parentId = new FieldAccessExpr(new NameExpr(PARENT), "id");
        Expression condition = new BinaryExpr(parentId, new IntegerLiteralExpr(itemsSize + 1),
                BinaryExpr.Operator.LESS);
        ReturnStmt returnChildrenStmt = new ReturnStmt(arraysAsListChildren);
        ObjectCreationExpr newArrayList = new ObjectCreationExpr(null,
                StaticJavaParser.parseClassOrInterfaceType("ArrayList"), new NodeList<>());
        ReturnStmt returnNewArrayListStmt = new ReturnStmt(newArrayList);
        IfStmt ifStmt = new IfStmt(condition, new BlockStmt(NodeList.nodeList(returnChildrenStmt)),
                new BlockStmt(NodeList.nodeList(returnNewArrayListStmt)));
        methodDeclaration.setBody(new BlockStmt(NodeList.nodeList(ifStmt)));

        return methodDeclaration;
    }

    private static void addSetItemLabelGeneratorStmt(FieldOrVariable fieldOrVariable, InsertionPoint insertionPoint,
            ClassOrInterfaceType recordType) {
        MethodCallExpr methodCallExpr = new MethodCallExpr(fieldOrVariable.reference(), "setItemLabelGenerator");
        // checking setItemLabelGenerator exist in the given class might be a
        // safer option.
        MethodReferenceExpr methodReferenceExpr = new MethodReferenceExpr();
        methodReferenceExpr.setScope(new NameExpr(recordType.getName()));
        methodReferenceExpr.setIdentifier(LABEL_KEY);
        methodCallExpr.addArgument(methodReferenceExpr);
        insertionPoint.add(new ExpressionStmt(methodCallExpr));
    }

    private static void addSetItemEnabledProviderMethodExpr(FieldOrVariable fieldOrVariable,
            InsertionPoint insertionPoint, ClassOrInterfaceType recordType) {
        MethodCallExpr methodCallExpr = new MethodCallExpr(fieldOrVariable.reference(), "setItemEnabledProvider");
        // checking setItemEnabledProvider exist in the given class might be a
        // safer option.
        MethodReferenceExpr methodReferenceExpr = new MethodReferenceExpr();
        methodReferenceExpr.setScope(new NameExpr(recordType.getName()));
        methodReferenceExpr.setIdentifier(ENABLED);
        methodCallExpr.addArgument(methodReferenceExpr);
        insertionPoint.add(new ExpressionStmt(methodCallExpr));
    }

    private static void updateVariableOrFieldDeclarationType(FieldOrVariable parent, ClassOrInterfaceType type) {
        if (parent.type() instanceof ClassOrInterfaceType classOrInterfaceType) {
            classOrInterfaceType.setTypeArguments(type);
            if (parent.initializer() != null) {
                // This hack is to render the expression with empty type
                // arguments in the instantiation
                // as Java Parser interprets an empty NodeList as not
                // parametrized
                // The token DELETE_THIS is used to replace the empty NodeList
                // with an empty type arguments when rendering the result
                ClassOrInterfaceType toDeleteType = new ClassOrInterfaceType();
                toDeleteType.setName("DELETE_THIS");
                parent.initializer().getType().setTypeArguments(toDeleteType);
            }
        } else {
            throw new IllegalArgumentException("Unknown variable type " + parent.type());
        }
    }

    private static List<LabelValueDisabled> childrenToLabelValueEnabledList(JavaComponent component) {
        List<JavaComponent> children = component.children();
        List<LabelValueDisabled> list = new ArrayList<>();
        for (JavaComponent child : children) {
            String label = null;
            String value;
            if (child.props().containsKey(LABEL_KEY)) {
                label = (String) child.props().get(LABEL_KEY);
            } else if (child.props().containsKey("text")) {
                label = (String) child.props().get("text");
            }
            if (child.props().containsKey(VALUE_KEY)) {
                value = (String) child.props().get(VALUE_KEY);
            } else {
                value = label;
            }
            if (label == null) {
                throw new IllegalArgumentException("Unknown property. Label cannot be null");
            }
            list.add(new LabelValueDisabled(label, value,
                    child.props().containsKey(DISABLED_KEY) && child.props().get(DISABLED_KEY).equals(Boolean.TRUE)));
        }
        return list;
    }

    private static List<LabelValueDisabled> propsToLabelValueDisabledList(JavaComponent component) {
        List<LabelValueDisabled> list = new ArrayList<>();
        if (component.props().containsKey("items")) {
            List<Map<String, Object>> items = (List<Map<String, Object>>) component.props().get("items");
            for (Map<String, Object> item : items) {
                String label = (String) item.get(LABEL_KEY);
                String value = (String) item.get(VALUE_KEY);
                Boolean disabled = (Boolean) item.get(DISABLED_KEY);
                list.add(new LabelValueDisabled(label, value, Boolean.TRUE.equals(disabled)));
            }
        }
        return list;
    }

    private record LabelValueDisabled(String label, String value, boolean disabled) {
    }
}
