/*
 * Copyright 2000-2024 Vaadin Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.vaadin.copilot.ide;

import java.io.File;

import com.vaadin.copilot.ComponentSourceFinder;
import com.vaadin.copilot.ProjectManager;
import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.internal.ComponentTracker.Location;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/** Integration with IDEs for development mode. */
public final class IdeIntegration {

    private final ProjectManager projectManager;
    private final ComponentSourceFinder sourceFinder;

    /** Creates a new integration with the given configuration. */
    public IdeIntegration(ProjectManager projectManager) {
        this.projectManager = projectManager;
        sourceFinder = new ComponentSourceFinder(projectManager);
    }

    /**
     * Opens, in the current IDE, the location (file + line number) where the given
     * component was created.
     *
     * @param component
     *            the component to show
     */
    public void showComponentCreateInIde(Component component) {
        internalShowInIde(component, sourceFinder.findTypeAndSourceLocation(component, false).createLocation());
    }

    /**
     * Opens, in the current IDE, the location (file + line number) where the given
     * component was attached.
     *
     * @param component
     *            the component to show
     */
    public void showComponentAttachInIde(Component component) {
        internalShowInIde(component, sourceFinder.findTypeAndSourceLocation(component, false).attachLocation());
    }

    private void internalShowInIde(Component component, Location location) {
        if (location == null) {
            getLogger().error("Unable to find the location where the component {} was created",
                    component.getClass().getName());
            return;
        }
        File sourceFile = projectManager.getFileForClass(location.className());
        if (sourceFile == null || !sourceFile.exists()) {
            getLogger().error("Unable to find source file for class {}", location.className());
            return;
        }

        IdeUtils.openFile(sourceFile, location.lineNumber());
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(IdeIntegration.class);
    }
}
