package com.vaadin.copilot;

import jakarta.annotation.security.DenyAll;
import jakarta.annotation.security.PermitAll;
import jakarta.annotation.security.RolesAllowed;

import java.lang.reflect.AnnotatedElement;

import com.vaadin.flow.server.auth.AnonymousAllowed;

/**
 * Utility class for figuring out access requirements for Java methods such as
 * routes and browser callables.
 */
public class AccessRequirementUtil {

    /**
     * Get the access requirement for the given annotated class or method based on
     * its annotations.
     *
     * @param annotatedClassOrMethod
     *            the annotated class or method
     * @param fallback
     *            the class or method to check if the given class has no access
     *            control annotations
     * @return the access requirement for using the class or method
     */
    public static AccessRequirement getAccessRequirement(AnnotatedElement annotatedClassOrMethod,
            AnnotatedElement fallback) {
        // Based on AccessAnnotationChecker.hasAccess
        if (annotatedClassOrMethod.isAnnotationPresent(DenyAll.class)) {
            return new AccessRequirement(AccessRequirement.Type.DENY_ALL);
        }
        if (annotatedClassOrMethod.isAnnotationPresent(AnonymousAllowed.class)) {
            return new AccessRequirement(AccessRequirement.Type.ANONYMOUS_ALLOWED);
        }
        RolesAllowed rolesAllowed = annotatedClassOrMethod.getAnnotation(RolesAllowed.class);
        if (rolesAllowed != null) {
            return new AccessRequirement(AccessRequirement.Type.ROLES_ALLOWED, rolesAllowed.value());
        } else if (annotatedClassOrMethod.isAnnotationPresent(PermitAll.class)) {
            return new AccessRequirement(AccessRequirement.Type.PERMIT_ALL);
        }

        if (fallback != null) {
            return getAccessRequirement(fallback, null);
        }
        return new AccessRequirement(AccessRequirement.Type.DENY_ALL);
    }
}
