package com.tenqube.visual_third.api;

import android.content.Context;

import com.google.gson.Gson;
import com.tenqube.visual_third.manager.PrefManager;
import com.tenqube.visual_third.model.api.AnalysisResponse;
import com.tenqube.visual_third.model.api.CurrencyResponse;
import com.tenqube.visual_third.model.api.SearchCompanyRequest;
import com.tenqube.visual_third.model.api.SearchCompanyResponse;
import com.tenqube.visual_third.model.api.SignUpRequest;
import com.tenqube.visual_third.model.api.SignUpResponse;
import com.tenqube.visual_third.model.api.SyncCategoryResponse;
import com.tenqube.visual_third.model.api.TransactionRequest;
import com.tenqube.visual_third.repository.AnalysisRepository;
import com.tenqube.visual_third.repository.VisualRepository;
import com.tenqube.visual_third.util.Utils;

import java.util.Calendar;
import java.util.concurrent.TimeUnit;

import okhttp3.OkHttpClient;
import retrofit2.Call;
import retrofit2.Callback;
import retrofit2.Response;
import retrofit2.Retrofit;
import retrofit2.converter.gson.GsonConverterFactory;
import tenqube.parser.OnNetworkResultListener;
import tenqube.parser.model.ParsingRule;

import static com.tenqube.visual_third.util.Utils.isEmpty;

public class VisualApi {

    public static final String TAG = VisualApi.class.getSimpleName();

    public static VisualApi mInstance;

    private Context context;
    private VisualApiService visualApiService;

    private PrefManager prefManager;
    private VisualRepository repository;
    private AnalysisRepository analysisRepository;

    private static final int API_KEY = 0;
    private static final int AUTHORIZATION = 1;


    public static VisualApi getInstance(Context context){
        synchronized (VisualApi.class) {
            if(mInstance == null) {
                mInstance = new VisualApi(context.getApplicationContext());
            }
        }
        return mInstance;
    }

    private VisualApi(Context context) {
        this.context = context;
        prefManager = PrefManager.getInstance(context);
        repository = VisualRepository.getInstance(context);
        analysisRepository = AnalysisRepository.getInstance(context);
    }

    private VisualApiService getApiService() {
        if (visualApiService == null) {
            visualApiService = provideRetrofit(getUrl()).create(VisualApiService.class);
        }
        return visualApiService;
    }

    private String[] getAuthInfo() {
        return new String[]{prefManager.loadStringValue(PrefManager.API_KEY, ""),
                prefManager.loadStringValue(PrefManager.UID, "")
        };
    }

    private String getUrl() {
        return VisualApiService.BASE_URL + prefManager.loadStringValue(PrefManager.QUALIFIER, "dev") + "/";
    }

    private Retrofit provideRetrofit(String url) {
        return new Retrofit.Builder()
                .baseUrl(url)
                .client(provideOkHttpClient())
                .addConverterFactory(GsonConverterFactory.create(new Gson()))

                .build();
    }

    private OkHttpClient provideOkHttpClient() {
        OkHttpClient.Builder okhttpClientBuilder = new OkHttpClient.Builder();
        okhttpClientBuilder.connectTimeout(5, TimeUnit.SECONDS);
        okhttpClientBuilder.readTimeout(5, TimeUnit.SECONDS);
        okhttpClientBuilder.writeTimeout(5, TimeUnit.SECONDS);
//        okhttpClientBuilder.addInterceptor(new LogInterceptor());

        return okhttpClientBuilder.build();
    }

    public SignUpResponse signUp(SignUpRequest signUpInfo) {

        try {
            Response<SignUpResponse> response = getApiService().signUp(getAuthInfo()[API_KEY], signUpInfo).execute();
            if (response.isSuccessful()) {
                return response.body();
            }
        } catch (Exception e) {

            e.printStackTrace();
        }

        return null;
    }

    public void syncCategory(int version) {
        try {

            String[] authInfo = getAuthInfo();
            Call<SyncCategoryResponse> call = getApiService().syncCategory(authInfo[API_KEY], authInfo[AUTHORIZATION], version);
            call.enqueue(new Callback<SyncCategoryResponse>() {
                @Override
                public void onResponse(Call<SyncCategoryResponse> call, Response<SyncCategoryResponse> response) {

                    if(response.isSuccessful()) {
                        SyncCategoryResponse categoryResponse = response.body();
                        if(categoryResponse != null) {
                            prefManager.saveIntValue(PrefManager.CATEGORY_VERSION, categoryResponse.getVersion());
                            if(!categoryResponse.getCategories().isEmpty())
                                repository.mergeCategory(categoryResponse.getCategories());
                        }
                    }
                }

                @Override
                public void onFailure(Call<SyncCategoryResponse> call, Throwable t) {

                }
            });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public ParsingRule syncParsingRule(int version) {
        try {
            String[] authInfo = getAuthInfo();

            Response<ParsingRule> response = getApiService().syncParsingRule(authInfo[API_KEY], authInfo[AUTHORIZATION], version).execute();
            if(response.isSuccessful()) {
                return response.body();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }



    public SearchCompanyResponse searchCompany(SearchCompanyRequest searchCompanyRequest) {
        try {

            String[] authInfo = getAuthInfo();
            String searchUrl = prefManager.loadStringValue(PrefManager.SEARCH_URL, "");

            Response<SearchCompanyResponse> response = getApiService().searchCompany(authInfo[API_KEY], authInfo[AUTHORIZATION],
                    searchUrl, searchCompanyRequest).execute();

            if(response.isSuccessful()) {
                repository.updateRetrySearch(searchCompanyRequest.getTransactions(), false);

                return response.body();
            } else {
                if(response.code() == 499 || response.code() == 500) {
                    // retry
                    // 사용자가 변경 하지않고, retry flag true 인경우 쿼리해서 다시 보내줘야함
                    // update retry flag, isSynced = false
                    repository.updateRetrySearch(searchCompanyRequest.getTransactions(), true);

                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }

        return null;
    }

    private void saveTransactions(TransactionRequest transactionRequest, Callback<Void> callback) {
        String[] authInfo = getAuthInfo();

        Call<Void> call = getApiService().saveTransactions(authInfo[API_KEY], authInfo[AUTHORIZATION], transactionRequest);
        call.enqueue(callback);
    }

    public void syncTransactions(final OnNetworkResultListener callback) {
        final TransactionRequest transactionRequest = repository.loadNotSyncedTransactions();
        if(transactionRequest != null) {
            saveTransactions(transactionRequest, new Callback<Void>() {
                @Override
                public void onResponse(Call<Void> call, Response<Void> response) {

                    if(response.isSuccessful() || response.code() == 400) {
                        repository.updateSyncedTransactions(transactionRequest);
                    }
                    if(callback != null) callback.onResult(true);
                }

                @Override
                public void onFailure(Call<Void> call, Throwable t) {

                    if(callback != null) callback.onResult(true);
                }
            });
        } else {
            if(callback != null) callback.onResult(true);
        }
    }

    public float getCurrencyRate(String from , String to) {
        try {
            String[] authInfo = getAuthInfo();

            Response<CurrencyResponse> response = getApiService().callCurrencyRate(authInfo[API_KEY], authInfo[AUTHORIZATION],
                    from, to).execute();

            if(response.isSuccessful()) {
                CurrencyResponse currencyResponse = response.body();
                if(currencyResponse != null) {
                    repository.mergeCurrency(from, to, currencyResponse.getRate(), Utils.getStringDateAsYYYYMMddHHmmss(Calendar.getInstance()));
                    return currencyResponse.getRate();
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }

        return 1;
    }

    public void syncAnalysis(int version) {
        try {

            String[] authInfo = getAuthInfo();
            Call<AnalysisResponse> call = getApiService().syncAnalysis(authInfo[API_KEY], authInfo[AUTHORIZATION], version);
            call.enqueue(new Callback<AnalysisResponse>() {
                @Override
                public void onResponse(Call<AnalysisResponse> call, Response<AnalysisResponse> response) {

                    if(response.isSuccessful()) {
                        AnalysisResponse analysisResponse = response.body();
                        if(analysisResponse != null) {
                            prefManager.saveIntValue(PrefManager.ANALYSIS_VERSION, analysisResponse.getVersion());
                            if(!isEmpty(analysisResponse.getContents()))
                                analysisRepository.mergeAnalysis(analysisResponse.getContents());
                        }
                    }
                }

                @Override
                public void onFailure(Call<AnalysisResponse> call, Throwable t) {
                }
            });
        } catch (Exception e) {
            e.printStackTrace();
        }

    }


}
