package com.tenqube.visual_third.repository;

import android.content.Context;
import android.text.TextUtils;

import com.tenqube.visual_third.db.DatabaseHelper;
import com.tenqube.visual_third.db.Db;
import com.tenqube.visual_third.db.dao.CardDao;
import com.tenqube.visual_third.db.dao.CategoryDao;
import com.tenqube.visual_third.db.dao.TransactionDao;
import com.tenqube.visual_third.db.dao.UserCategoryDao;
import com.tenqube.visual_third.entity.Card;
import com.tenqube.visual_third.entity.Category;
import com.tenqube.visual_third.entity.JoinedTransaction;
import com.tenqube.visual_third.entity.Transaction;
import com.tenqube.visual_third.entity.UserCategory;
import com.tenqube.visual_third.model.api.SearchCompanyRequest;
import com.tenqube.visual_third.model.api.SearchCompanyResponse;
import com.tenqube.visual_third.model.api.SyncCategoryResponse;
import com.tenqube.visual_third.model.js.Cards;
import com.tenqube.visual_third.model.js.CategoryInfo;
import com.tenqube.visual_third.model.js.TransactionRequest;
import com.tenqube.visual_third.model.js.Transactions;
import com.tenqube.visual_third.model.js.UpdateTransactionRequest;

import java.util.ArrayList;

import tenqube.parser.constants.Constants;

import static com.tenqube.visual_third.util.Mapper.toApiTransactions;
import static com.tenqube.visual_third.util.Mapper.toJsCards;
import static com.tenqube.visual_third.util.Mapper.toJsCategories;
import static com.tenqube.visual_third.util.Mapper.toJsTransactions;
import static com.tenqube.visual_third.util.Mapper.toJsUserCategories;
import static com.tenqube.visual_third.util.Mapper.toSearchTransactionWithEntity;

public class VisualRepository {

    private static VisualRepository mInstance;

    private final CardDao cardDao;
    private final CategoryDao categoryDao;
    private final TransactionDao transactionDao;
    private final UserCategoryDao userCategoryDao;

    public static VisualRepository getInstance(Context context){
        synchronized (VisualRepository.class) {
            if(mInstance == null){
                mInstance = new VisualRepository(context);
            }
        }
        return mInstance;
    }

    private VisualRepository(Context context) {
        DatabaseHelper dbHelper = DatabaseHelper.getInstance(context);
        Db db = new Db(dbHelper.getWritableDatabase(), dbHelper.getReadableDatabase());

        this.cardDao = new CardDao(db);
        this.categoryDao = new CategoryDao(db);
        this.transactionDao = new TransactionDao(db);
        this.userCategoryDao = new UserCategoryDao(db);
    }


    public void loadAll() {
//        cardDao.loadAll();
//        categoryDao.loadAll();
//        transactionDao.loadAll();
//        userCategoryDao.loadAll();
    }

    public JoinedTransaction loadJoinedTransaction(int tranId) {
        return transactionDao.loadJoinedTransaction(tranId);
    }

    public int loadTranId(String identifier) {
        return transactionDao.loadTranId(identifier);
    }

    public Transaction loadApplyAllTran(String keyword) {
        if(TextUtils.isEmpty(keyword)) return  null;
        return transactionDao.loadApplyAllTran(keyword);
    }

    // load
    public ArrayList<Transactions.Transaction> loadTransactions(TransactionRequest transactionRequest) {
        ArrayList<JoinedTransaction> transactions = transactionDao.loadJoinedTransactions(transactionRequest);
        return toJsTransactions(transactions);
    }

    public com.tenqube.visual_third.model.api.TransactionRequest loadNotSyncedTransactions() {
        ArrayList<JoinedTransaction> transactions = transactionDao.loadNotSyncedTransactions();
        if(!transactions.isEmpty())
            return new com.tenqube.visual_third.model.api.TransactionRequest(toApiTransactions(transactions));
        else
            return null;
    }

    public SearchCompanyRequest loadFailedSearchTransactions() {
        ArrayList<JoinedTransaction> transactions = transactionDao.loadFailedSearchTransactions();
        if(!transactions.isEmpty())
            return new SearchCompanyRequest(toSearchTransactionWithEntity(transactions));
        else
            return null;
    }

    public ArrayList<Cards.Card> loadCards() {
        ArrayList<Card> cards = cardDao.loadCards();
        return toJsCards(cards);
    }

    public ArrayList<CategoryInfo.ServerCategory> loadCategories() {
        ArrayList<Category> categories = categoryDao.loadCategories();
        return toJsCategories(categories);
    }

    public Category loadCategory(String categoryCode) {
        return categoryDao.loadCategory(categoryCode);
    }

    public ArrayList<CategoryInfo.UserCategory> loadUserCategories() {
        ArrayList<UserCategory> categories = userCategoryDao.loadUserCategories();
        return toJsUserCategories(categories);
    }

    public void mergeCategory(ArrayList<SyncCategoryResponse.Category> categories) {

        try {
            for(SyncCategoryResponse.Category category : categories) {

                categoryDao.mergeCategory(category);
                // 새로운 카테고리 가 생김
                // 카테고리 아이디 추가

                // 기존 카테고리가 삭제됨
                // 내역 삭제
                // 사용자 카테고리 삭제

            }
        } catch (Exception e) {
            e.printStackTrace();
        }

    }

    public void updateSyncedTransactions(com.tenqube.visual_third.model.api.TransactionRequest transactionRequest) {
        ArrayList<String> ids = new ArrayList<>();
        ArrayList<com.tenqube.visual_third.model.api.TransactionRequest.Transaction> transactions = transactionRequest.getTransactions();

        for(com.tenqube.visual_third.model.api.TransactionRequest.Transaction tran : transactions) {
            ids.add(tran.getIdentifier());
        }
        transactionDao.updateSyncedTransactions(ids);
    }

    public void updateTransaction(UpdateTransactionRequest updateTransactionRequest) {
        transactionDao.updateTransaction(updateTransactionRequest);
    }

    public void updateRetrySearch(ArrayList<SearchCompanyRequest.Transaction> transactions, boolean shouldRetry) {
        ArrayList<String> ids = new ArrayList<>();

        for(SearchCompanyRequest.Transaction tran : transactions) {
            ids.add(tran.getIdentifier());
        }
        transactionDao.updateRetryTransactions(ids, shouldRetry);


    }

    public int loadCardId(String cardName, int cardType, int cardSubType) {
        int cardId = cardDao.getCardId(cardName, cardType, cardSubType);
        if(cardId == -1) {
            cardId = cardDao.insert(cardName, cardType, cardSubType);
        }
        return cardId;
    }


    /**
     * 파싱된 결과 로컬 디비에 저장하기
     * 1. 카드 아이디 구하기 없으면 저장
     * 2. 사용자 카테고리 아이디
     * 3. 내역 저장하기
     * @param transactions 파싱된 내역
     */
    public void mergeTransactions(ArrayList<tenqube.parser.model.Transaction> transactions) {

        try {
            for(tenqube.parser.model.Transaction parsedTran : transactions) {
                // getCardId
                int cardId = loadCardId(parsedTran.cardName, parsedTran.cardType, parsedTran.cardSubType);

                // getUserCateId
                int userCateId = userCategoryDao.getCategoryId(parsedTran.dwType == Constants.DWType.DEPOSIT.ordinal() ?
                        "901010" : "101010");

                // merge transaction
                transactionDao.mergeTransaction(parsedTran, cardId, userCateId);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }


    }

    /**
     * 검색 후 업데이트 할 부분
     * @param response 검색된 결과 값
     */
    public void updateTransactions(SearchCompanyResponse response) {

        try {
            // userCateId 구하기
            for(SearchCompanyResponse.TranCompany tranCompany : response.getResults()) {
                int userCateId = userCategoryDao.getCategoryId(tranCompany.getCategory().getRepCode());
                transactionDao.updateTransaction(tranCompany, userCateId);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }

    }

    public void updateTransaction(SearchCompanyResponse.TranCompany tranCompany, boolean isAll) {

        try {
            // userCateId 구하기
            int userCateId = userCategoryDao.getCategoryId(tranCompany.getCategory().getRepCode());

            if(isAll) {
                transactionDao.updateTransactionByKeyword(tranCompany, userCateId);
            } else {
                transactionDao.updateTransaction(tranCompany, userCateId);

            }
        } catch (Exception e) {
            e.printStackTrace();
        }

    }

}
