package com.tenqube.visual_third.api;

import android.content.Context;

import com.google.gson.Gson;
import com.tenqube.visual_third.manager.PrefManager;
import com.tenqube.visual_third.model.api.SearchCompanyRequest;
import com.tenqube.visual_third.model.api.SearchCompanyResponse;
import com.tenqube.visual_third.model.api.SignUpRequest;
import com.tenqube.visual_third.model.api.SignUpResponse;
import com.tenqube.visual_third.model.api.SyncCategoryResponse;
import com.tenqube.visual_third.model.api.TransactionRequest;
import com.tenqube.visual_third.repository.VisualRepository;

import java.io.IOException;
import java.util.concurrent.TimeUnit;

import okhttp3.OkHttpClient;
import retrofit2.Call;
import retrofit2.Callback;
import retrofit2.Response;
import retrofit2.Retrofit;
import retrofit2.converter.gson.GsonConverterFactory;
import tenqube.parser.OnNetworkResultListener;
import tenqube.parser.model.ParsingRule;

import static tenqube.parser.core.ParserService.mIsDebug;
import static tenqube.parser.util.LogUtil.LOGI;

public class VisualApi {

    public static final String TAG = VisualApi.class.getSimpleName();

    public static VisualApi mInstance;

    private Context context;
    private VisualApiService visualApiService;

    private PrefManager prefManager;
    private VisualRepository repository;

    public static VisualApi getInstance(Context context){
        synchronized (VisualApi.class) {
            if(mInstance == null){
                mInstance = new VisualApi(context.getApplicationContext());
            }
        }
        return mInstance;
    }

    private VisualApi(Context context) {
        this.context = context;
        prefManager = PrefManager.getInstance(context);
        repository = VisualRepository.getInstance(context);
    }

    private VisualApiService getApiService() {
        if (visualApiService == null) {
            visualApiService = provideRetrofit(getUrl()).create(VisualApiService.class);
        }
        return visualApiService;
    }

    private String getUrl() {
        return VisualApiService.BASE_URL + prefManager.loadStringValue(PrefManager.QUALIFIER, "dev") + "/";
    }

    private Retrofit provideRetrofit(String url) {
        return new Retrofit.Builder()
                .baseUrl(url)
                .client(provideOkHttpClient())
                .addConverterFactory(GsonConverterFactory.create(new Gson()))
                .build();
    }

    private OkHttpClient provideOkHttpClient() {
        OkHttpClient.Builder okhttpClientBuilder = new OkHttpClient.Builder();
        okhttpClientBuilder.connectTimeout(5, TimeUnit.SECONDS);
        okhttpClientBuilder.readTimeout(5, TimeUnit.SECONDS);
        okhttpClientBuilder.writeTimeout(5, TimeUnit.SECONDS);

        return okhttpClientBuilder.build();
    }

    public SignUpResponse signUp(SignUpRequest signUpInfo) {

        try {
            String apiKey = prefManager.loadStringValue(PrefManager.API_KEY, "");
            Response<SignUpResponse> response = getApiService().signUp(apiKey, signUpInfo).execute();
            if (response.isSuccessful()) {
                return response.body();
            }
        } catch (IOException e) {
            e.printStackTrace();
        }

        return null;

    }

    public void syncCategory(int version) {
        try {

            String apiKey = prefManager.loadStringValue(PrefManager.API_KEY, "");
            String authorization = prefManager.loadStringValue(PrefManager.UID, "");
            Call<SyncCategoryResponse> call = getApiService().syncCategory(apiKey, authorization, version);
            call.enqueue(new Callback<SyncCategoryResponse>() {
                @Override
                public void onResponse(Call<SyncCategoryResponse> call, Response<SyncCategoryResponse> response) {

                    if(response.isSuccessful()) {
                        SyncCategoryResponse categoryResponse = response.body();
                        if(categoryResponse != null) {
                            prefManager.saveIntValue(PrefManager.CATEGORY_VERSION, categoryResponse.getVersion());
                            if(!categoryResponse.getCategories().isEmpty())
                                repository.mergeCategory(categoryResponse.getCategories());
                        }
                    }
                }

                @Override
                public void onFailure(Call<SyncCategoryResponse> call, Throwable t) {

                }
            });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


    public ParsingRule syncParsingRule(int version) {
        try {
            String apiKey = prefManager.loadStringValue(PrefManager.API_KEY, "");
            String authorization = prefManager.loadStringValue(PrefManager.UID, "");

            Response<ParsingRule> response = getApiService().syncParsingRule(apiKey, authorization, version).execute();
            if(response.isSuccessful()) {
                return response.body();
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        return null;
    }

    public SearchCompanyResponse searchCompany(SearchCompanyRequest searchCompanyRequest) {
        try {
            String apiKey = prefManager.loadStringValue(PrefManager.SEARCH_API_KEY, "");
            String searchUrl = prefManager.loadStringValue(PrefManager.SEARCH_URL, "");
            String authorization = prefManager.loadStringValue(PrefManager.UID, "");

            Response<SearchCompanyResponse> response = getApiService().searchCompany(apiKey, authorization,
                    searchUrl, searchCompanyRequest).execute();

            if(response.isSuccessful()) {
                repository.updateRetrySearch(searchCompanyRequest.getTransactions(), false);

                return response.body();
            } else {
                if(response.code() == 499 || response.code() == 500) {

                    // retry
                    // 사용자가 변경 하지않고, retry flag true 인경우 쿼리해서 다시 보내줘야함
                    // update retry flag, isSynced = false
                    repository.updateRetrySearch(searchCompanyRequest.getTransactions(), true);

                }
                LOGI(TAG, "searchCompany" + response.code() +"msg" + response.message(), mIsDebug);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }

        return null;
    }

    private void saveTransactions(TransactionRequest transactionRequest, Callback<Void> callback) {
        String apiKey = prefManager.loadStringValue(PrefManager.API_KEY, "");
        String authorization = prefManager.loadStringValue(PrefManager.UID, "");
        Call<Void> call = getApiService().saveTransactions(apiKey, authorization, transactionRequest);
        call.enqueue(callback);
    }

    public void syncTransactions(final OnNetworkResultListener callback) {
        final TransactionRequest transactionRequest = repository.loadNotSyncedTransactions();
        if(transactionRequest != null) {
            saveTransactions(transactionRequest, new Callback<Void>() {
                @Override
                public void onResponse(Call<Void> call, Response<Void> response) {

                    LOGI("saveTran", "saveTransactions" + response.code() + "msg" + response.message(), mIsDebug);
                    if(response.isSuccessful() || response.code() == 400) {
                        repository.updateSyncedTransactions(transactionRequest);
                    }
                    if(callback != null) callback.onResult(true);
                }

                @Override
                public void onFailure(Call<Void> call, Throwable t) {

                    LOGI("saveTran", "saveTransactions onFailure", mIsDebug);
                    if(callback != null) callback.onResult(false);
                }
            });
        } else {

            LOGI("saveTran", "saveTransactions transactionRequest is null", mIsDebug);
            if(callback != null) callback.onResult(false);
        }
    }

}
