package com.tenqube.visual_third.core;


import android.content.Context;
import android.database.Cursor;
import android.net.Uri;
import android.os.Bundle;
import android.os.Handler;
import android.os.Message;

import com.tenqube.visual_third.manager.PrefManager;

import java.util.ArrayList;
import java.util.Calendar;

import tenqube.parser.BulkSmsAdapter;
import tenqube.parser.OnNetworkResultListener;
import tenqube.parser.constants.Constants;
import tenqube.parser.core.ParserService;
import tenqube.parser.model.SMS;
import tenqube.parser.model.Transaction;

import static com.tenqube.visual_third.core.VisualWebActivity.FINISH;
import static com.tenqube.visual_third.core.VisualWebActivity.PROGRESS;
import static com.tenqube.visual_third.util.Utils.getConvertedDate;

class BulkLoader implements BulkSmsAdapter {

    private final Context mContext;
    private Cursor mCursor;
    private final Handler mHandler;
    private final OnSmsLoadFinished onSmsLoadFinished;

    private ParserService parserService;
    private VisualApi api;
    private PrefManager prefManager;

    /**
     * @param context 컨텍스트
     * @param handler 진행상태 메세지 전송 handler
     */
    public BulkLoader(Context context,
                      Handler handler,
                      OnSmsLoadFinished onSmsLoadFinished) {

        this.mContext = context;
        this.mHandler = handler;
        this.onSmsLoadFinished = onSmsLoadFinished;

        prefManager = PrefManager.getInstance(context);
        parserService = ParserService.getInstance(context);
        parserService.setDebugMode(true);
        api = VisualApi.getInstance(context);




    }

    public void doParsing() {

        // cursor 초기화
        initCursor();

        // syncParsingRule()
        parserService.syncParsingRule(api.syncParsingRule());

        // parseBulk
        parserService.parseBulk(this);

    }

    private void initCursor() {

        try {
            Uri uri = Uri.parse("content://sms/inbox");
            mCursor = mContext.getContentResolver().query(uri, null, getWhere(), null, "date asc");
            if (mCursor != null) {
                mCursor.moveToFirst();
            }
        } catch (Exception e) {
            onError(1);
        }

    }

    private String getWhere() {
        Calendar calendar = Calendar.getInstance();
        long time = prefManager.loadLongValue(PrefManager.SIGN_UP_TIME, 0);
        if(time != 0)
            calendar.setTimeInMillis(time);
        calendar.add(Calendar.MONTH, -6);
        return  "date > " + calendar.getTimeInMillis() + " and date <= " + time;
    }

    @Override
    public int getSmsCount() {
        return mCursor == null ? 0 : mCursor.getCount();
    }

    @Override
    public SMS getSmsAt(int n) {
        return getSMS();
    }

    private SMS getSMS() {
        SMS sms = null;
        if (mCursor != null) {
            sms = new SMS(mCursor.getInt(mCursor.getColumnIndex("_id")),
                mCursor.getString(mCursor.getColumnIndex("body")),
                mCursor.getString(mCursor.getColumnIndex("address")),
                mCursor.getString(mCursor.getColumnIndex("address")),
                getConvertedDate(mCursor.getLong(mCursor.getColumnIndex("date"))),
                Constants.SMSType.SMS.ordinal());
            mCursor.moveToNext();
        }

        return sms;
    }

    /**
     * @param now 현재 내역 수
     * @param total 전체 내역 수
     */
    @Override
    public void onProgress(int now, int total) {
        //ui 변경
        int percent = ((now * 100) / total) + 1;

        Bundle data = new Bundle();
        data.putInt("percent", percent);
        data.putInt("now", now);
        data.putInt("total", total);


        sendHandlerMsg(data,
                PROGRESS,
                percent);
    }

    /**
     * 구버전의 경우 v1형태에 맞게 구성후 검색서버 반영되면 변경 처리
     * doGrouping
     * @param transactions 파싱된 지출내역 목록
     * @param callback 성공여부 콜백 메소드
     */
    @Override
    public void sendToServerTransactions(ArrayList<Transaction> transactions, final OnNetworkResultListener callback) {

        // 서버 전송
        api.searchCompany(transactions, callback);

    }



    @Override
    public void onCompleted() {

        sendHandlerMsg(null, FINISH, 0);

        if(mCursor != null)
            mCursor.close();

        if(onSmsLoadFinished != null)
            onSmsLoadFinished.onFinished(false);
    }



    @Override
    public void onError(final int resultCode) {

        //파싱룰 동기화가 필요한경우
        sendHandlerMsg(null, FINISH, 0);

        if(mCursor != null)
            mCursor.close();

        if(onSmsLoadFinished != null)
            onSmsLoadFinished.onFinished(true);
    }

    private void sendHandlerMsg(Bundle data, int what, int arg1) {

        if(mHandler != null) {
            Message hMsg = mHandler.obtainMessage();
            if(data != null) hMsg.setData(data);
            hMsg.what = what;
            hMsg.arg1 = arg1;
            mHandler.sendMessage(hMsg);
        }
    }


    public interface OnSmsLoadFinished {
        void onFinished(boolean isError);
    }


}
