package com.tenqube.visual_third.manager;

import com.tenqube.visual_third.api.VisualApi;
import com.tenqube.visual_third.model.api.VersionResponse;

import tenqube.parser.core.ParserService;
import tenqube.parser.model.ParsingRule;

import static com.tenqube.visual_third.Constants.ANALYSIS_VERSION;
import static com.tenqube.visual_third.Constants.CATEGORY_VERSION;
import static com.tenqube.visual_third.Constants.CLIP_CATEGORY_VERSION;
import static com.tenqube.visual_third.Constants.NOTIFICATION_VERSION;
import static com.tenqube.visual_third.Constants.RULE_VERSION;
import static com.tenqube.visual_third.api.VisualApi.API_KEY;
import static com.tenqube.visual_third.manager.PrefManager.PARSING_RULE_VERSION;

public class ResourceManager {

    private final VisualApi visualApi;
    private final PrefManager prefManager;
    private final ParserService parserService;

    private static ResourceManager mInstance = null;

    public static ResourceManager getInstance(VisualApi visualApi, PrefManager prefManager, ParserService parserService) {

        synchronized (ResourceManager.class) {
            if (mInstance == null) {
                mInstance = new ResourceManager(visualApi, prefManager, parserService);
            }
        }
        return mInstance;
    }

    private ResourceManager(VisualApi visualApi, PrefManager prefManager, ParserService parserService) {
        this.visualApi = visualApi;
        this.prefManager = prefManager;
        this.parserService = parserService;
    }

    public void sync() {
        new Thread(new Runnable() {
            @Override
            public void run() {

                try {
                    String[] authInfos = visualApi.getAuthInfo();

                    VersionResponse versionResponse = visualApi.syncVersion(authInfos[API_KEY]);

                    syncCategory(authInfos, versionResponse);

                    syncClipCategory(authInfos, versionResponse);

                    syncAnalysis(authInfos, versionResponse);

                    syncParsingRule(authInfos, versionResponse);

                    syncNotification(authInfos, versionResponse);

                } catch (Exception e) {
                    e.printStackTrace();
                }

            }
        }).start();

    }

    public void syncNotification() {
        new Thread(new Runnable() {
            @Override
            public void run() {

                try {
                    String[] authInfos = visualApi.getAuthInfo();

                    VersionResponse versionResponse = visualApi.syncVersion(authInfos[API_KEY]);

                    syncNotification(authInfos, versionResponse);

                } catch (Exception e) {
                    e.printStackTrace();
                }

            }
        }).start();

    }

    private void syncCategory(String[] authInfos, VersionResponse versionResponse) {
        int clientVersion = prefManager.loadIntValue(PrefManager.CATEGORY_VERSION, CATEGORY_VERSION);
        if(versionResponse != null && versionResponse.getCategory() > clientVersion) {
            visualApi.syncCategory(authInfos, clientVersion, versionResponse.getCategory());
        }
    }

    private void syncClipCategory(String[] authInfos, VersionResponse versionResponse) {
        int clientVersion = prefManager.loadIntValue(PrefManager.CLIP_CATEGORY_VERSION, CLIP_CATEGORY_VERSION);
        if(versionResponse != null && versionResponse.getClipCategory() > clientVersion) {
            visualApi.syncClipCategory(authInfos, clientVersion, versionResponse.getClipCategory());
        }
    }

    private void syncAnalysis(String[] authInfos, VersionResponse versionResponse) {
        int clientVersion = prefManager.loadIntValue(PrefManager.ANALYSIS_VERSION, ANALYSIS_VERSION);
        if(versionResponse != null && versionResponse.getAnalysis() > clientVersion) {
            visualApi.syncAnalysis(authInfos, clientVersion, versionResponse.getAnalysis());
        }
    }

    private void syncParsingRule(String[] authInfos, VersionResponse versionResponse) {

        int clientVersion = prefManager.loadIntValue(PARSING_RULE_VERSION, RULE_VERSION);

        if(versionResponse != null && versionResponse.getParsingRule() > clientVersion) {
            ParsingRule parsingRule = visualApi.syncParsingRule(authInfos, clientVersion, versionResponse.getParsingRule());
            if(parsingRule != null) {
                parserService.syncParsingRule(parsingRule);
                prefManager.saveIntValue(PARSING_RULE_VERSION, parsingRule.ruleVersion);
            }
        }
    }

    private void syncNotification(String[] authInfos, VersionResponse versionResponse) {

        int clientVersion = prefManager.loadIntValue(PrefManager.NOTIFICATION_VERSION, NOTIFICATION_VERSION);
        if(versionResponse != null && versionResponse.getNotification() > clientVersion) {
            visualApi.syncNotification(authInfos, versionResponse.getNotification());
        }
    }
}
