package com.tenqube.visual_third;

import android.Manifest;
import android.content.Context;
import android.content.IntentFilter;
import android.content.pm.PackageManager;
import android.support.annotation.NonNull;
import android.support.v4.app.ActivityCompat;
import android.text.TextUtils;

import com.tenqube.visual_third.api.VisualApi;
import com.tenqube.visual_third.exception.AuthException;
import com.tenqube.visual_third.exception.ParameterException;
import com.tenqube.visual_third.manager.AnswerManager;
import com.tenqube.visual_third.manager.VisualAlarmManager;
import com.tenqube.visual_third.model.api.SignUpRequest;
import com.tenqube.visual_third.model.api.SignUpResponse;
import com.tenqube.visual_third.model.js.LogRequest;
import com.tenqube.visual_third.parser.catcher.SMSTestCatcher;
import com.tenqube.visual_third.repository.VisualRepository;
import com.tenqube.visual_third.ui.OnResultListener;
import com.tenqube.visual_third.ui.VisualWebActivity;

import tenqube.parser.core.ParserService;

import static com.tenqube.visual_third.Constants.DEV;
import static com.tenqube.visual_third.Constants.PROD;
import static com.tenqube.visual_third.manager.AnswerManager.initFabric;
import static com.tenqube.visual_third.util.Validator.notNull;
import static com.tenqube.visual_third.util.Validator.notZero;
import static tenqube.parser.core.ParserService.mIsDebug;
import static tenqube.parser.util.LogUtil.LOGI;

public class VisualServiceImpl implements VisualService {

    public static final String TAG = VisualServiceImpl.class.getSimpleName();

    private Context context;
    private VisualApi api;

    private ParserService parserService;
    private VisualRepository repository;
    private VisualAlarmManager alarmManager;

    private boolean isDoingSignup;
    private boolean isDoingStartVisual;

    private SMSTestCatcher smsTestCatcher;

    public VisualServiceImpl(@NonNull Context context, String apiKey, String qualifier) throws ParameterException {

        if(TextUtils.isEmpty(apiKey)) throw new ParameterException("apiKey is null");

        if(TextUtils.isEmpty(qualifier) || !(Constants.DEV.equals(qualifier) || Constants.PROD.equals(qualifier)))
            throw new ParameterException("check qualifier value (" + DEV + ", " + PROD + ")");

        this.context = context;
        initFabric(context);

        api = VisualApi.getInstance(context);
        parserService = ParserService.getInstance(context);
        repository = VisualRepository.getInstance(context);
        alarmManager= VisualAlarmManager.getInstance(context);

        repository.saveSDKInfo(apiKey, qualifier);

//        registerTestCatcher();
    }

    private void registerTestCatcher() {

        try {
            smsTestCatcher = new SMSTestCatcher();
            IntentFilter intentFilter = new IntentFilter();
            intentFilter.addAction(SMSTestCatcher.SMS_TEST_ACTION);
            if(context != null) context.registerReceiver(smsTestCatcher, intentFilter);
        } catch (Exception e) {
            e.printStackTrace();
        }

    }

    @Override
    public void signUp(@NonNull final String custId, @NonNull final OnResultListener onResultListener) throws ParameterException {
        LOGI(TAG, "가입", mIsDebug);
        if(isDoingSignup) return;
        isDoingSignup = true;
        final SignUpRequest signUpRequest = new SignUpRequest(custId);
        notNull(onResultListener);

        if(!repository.shouldSignUp()) {
            onResultListener.onResult(Constants.SignupResult.ALREADY_JOINED, "already joined");
            isDoingSignup = false;
            return;
        }
        AnswerManager.onKeyMetric(new LogRequest("signUp"));
        new Thread(new Runnable() {
            @Override
            public void run() {
                SignUpResponse response = api.signUp(signUpRequest);

                if (response == null || response.getResults() == null) {
                    onResultListener.onResult(Constants.SignupResult.FAIL, "network error");
                } else {
                    repository.saveServerInfo(response);

                    onResultListener.onResult(Constants.SignupResult.SUCCESS, "success");
                }

                isDoingSignup = false;
            }
        }).start();
    }

    @Override
    public void startVisual(@NonNull String custId, final @NonNull OnResultListener resultListener) throws ParameterException, SecurityException {
        LOGI(TAG, "웹 가계부 호출", mIsDebug);
        if(isDoingStartVisual) return;

        isDoingStartVisual = true;

        notNull(resultListener);

        checkSMSPermission();

        signUp(custId, new OnResultListener() {
            @Override
            public void onResult(int signUpResult, String msg) {

                switch (signUpResult) {
                    case Constants.SignupResult.ALREADY_JOINED:
                    case Constants.SignupResult.SUCCESS:
                        try {
                            AnswerManager.onKeyMetric(new LogRequest("startVisual"));
                            VisualWebActivity.startActivity(context, "");
                        } catch (AuthException e) {
                            resultListener.onResult(Constants.SignupResult.FAIL, e.toString());
                            return;
                        } finally {
                            isDoingStartVisual = false;
                        }
                        break;
                }

                resultListener.onResult(signUpResult, msg);
                isDoingStartVisual = false;
            }
        });
    }

    @Override
    public boolean isActiveTranPopup() {
        AnswerManager.onKeyMetric(new LogRequest("isActiveTranPopup"));
        return repository.shouldShowTranPopup();
    }

    @Override
    public void setTranPopup(boolean isActive) {
        AnswerManager.onKeyMetric(new LogRequest("setTranPopup"));
        repository.setTranPopup(isActive);
    }

    private void checkSMSPermission() throws SecurityException {
        if (!repository.isExecutedBulk() &&
                ActivityCompat.checkSelfPermission(context, Manifest.permission.READ_SMS) != PackageManager.PERMISSION_GRANTED &&
                ActivityCompat.checkSelfPermission(context, Manifest.permission.RECEIVE_SMS) != PackageManager.PERMISSION_GRANTED &&
                ActivityCompat.checkSelfPermission(context, Manifest.permission.RECEIVE_MMS) != PackageManager.PERMISSION_GRANTED) {

            throw new SecurityException("Please grant permissions " +
                    "(Manifest.permission.READ_SMS, " +
                    "Manifest.permission.RECEIVE_SMS, " +
                    "Manifest.permission.RECEIVE_MMS)");
        }
    }

    @Override
    public void setDebugMode(boolean isDebug) {
        LOGI(TAG, "setDebugMode", mIsDebug);

        if(isActive() && parserService != null) {
            parserService.setDebugMode(isDebug);
        }
    }

    @Override
    public void initSDK() {
        LOGI(TAG, "initSDK", mIsDebug);

        if(isActive() && parserService != null) {
            parserService.initDb();
        }
    }

    @Override
    public void setEnabled(boolean enabled) {
        repository.setEnabled(enabled);

        if(!enabled) {
            repository.setActiveNoti("monthly", false);
            repository.setActiveNoti("weekly", false);
            repository.setActiveNoti("daily", false);
        }
        alarmManager.setAlarms();
    }

    @Override
    public void settingNotification(int smallIcon, @NonNull String channel, @NonNull String name, String description, int color) throws ParameterException {
        notZero(smallIcon);
        notNull(channel);
        notNull(name);
        repository.settingNotification(smallIcon, channel, name, description, color);
    }

    @Override
    public void setMonthlyNoti(boolean isActive) {
        repository.setActiveNoti("monthly", isActive);
        alarmManager.setAlarms();
    }

    @Override
    public boolean isActiveMonthlyNoti() {
        return repository.isActiveNoti("monthly");
    }

    @Override
    public void setWeeklyNoti(boolean isActive) {
        repository.setActiveNoti("weekly", isActive);
        alarmManager.setAlarms();

    }

    @Override
    public boolean isActiveWeeklyNoti() {
        return repository.isActiveNoti("weekly");
    }

    @Override
    public void setDailyNoti(boolean isActive) {
        repository.setActiveNoti("daily", isActive);
        alarmManager.setAlarms();
    }

    @Override
    public boolean isActiveDailyNoti() {
        return repository.isActiveNoti("daily");
    }

    @Override
    public void setReportTest(String type, int second) {
        VisualAlarmManager.getInstance(context).setReportTest(type, second);
    }


    private boolean isActive() {
        return context != null;
    }
}
