package com.tenqube.visual_third.core;

import android.content.Context;

import com.google.gson.Gson;
import com.tenqube.visual_third.OnResultListener;
import com.tenqube.visual_third.manager.PrefManager;
import com.tenqube.visual_third.model.SearchCompanyRequest;
import com.tenqube.visual_third.model.SearchCompanyResponse;
import com.tenqube.visual_third.model.SignUpRequest;
import com.tenqube.visual_third.model.SignUpResponse;
import com.tenqube.visual_third.model.TransactionRequest;
import com.tenqube.visual_third.util.Mapper;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.concurrent.TimeUnit;

import okhttp3.OkHttpClient;
import retrofit2.Call;
import retrofit2.Callback;
import retrofit2.Response;
import retrofit2.Retrofit;
import retrofit2.converter.gson.GsonConverterFactory;
import tenqube.parser.OnNetworkResultListener;
import tenqube.parser.model.ParsingRule;
import tenqube.parser.model.Transaction;

import static com.tenqube.visual_third.manager.PrefManager.PARSING_RULE_VERSION;
import static com.tenqube.visual_third.manager.PrefManager.UID;

class VisualApi {

    public static final String TAG = VisualApi.class.getSimpleName();

    public static VisualApi mInstance;

    private Context context;
    private VisualApiService visualApiService;
    private SearchApiService searchApiService;

    private PrefManager prefManager;


    public static VisualApi getInstance(Context context){
        synchronized (VisualApi.class) {
            if(mInstance == null){
                mInstance = new VisualApi(context.getApplicationContext());
            }

        }
        return mInstance;
    }

    private VisualApi(Context context) {
        this.context = context;
        prefManager = PrefManager.getInstance(context);
    }

    private VisualApiService getApiService() {
        if (visualApiService == null) {
            visualApiService = provideRetrofit(getUrl(), false).create(VisualApiService.class);

        }
        return visualApiService;
    }

    private SearchApiService getSearchApiService() {
        if (searchApiService == null) {
            searchApiService = provideRetrofit(prefManager.loadStringValue(PrefManager.SEARCH_URL, ""), true).create(SearchApiService.class);

        }
        return searchApiService;
    }


    private String getUrl() {
        return VisualApiService.BASE_URL + VisualApiService.QUALIFIER;
    }

    private Retrofit provideRetrofit(String url, boolean isSearch) {
        return new Retrofit.Builder()
                .baseUrl(url)
                .client(provideOkHttpClient(isSearch))
                .addConverterFactory(GsonConverterFactory.create(new Gson()))
                .build();
    }

    private OkHttpClient provideOkHttpClient(boolean isSearch) {
        OkHttpClient.Builder okhttpClientBuilder = new OkHttpClient.Builder();
        okhttpClientBuilder.connectTimeout(5, TimeUnit.SECONDS);
        okhttpClientBuilder.readTimeout(5, TimeUnit.SECONDS);
        okhttpClientBuilder.writeTimeout(5, TimeUnit.SECONDS);
        okhttpClientBuilder.addInterceptor(isSearch ? new AddSearchHeaderInterceptor(prefManager) : new AddHeaderInterceptor(prefManager));

        return okhttpClientBuilder.build();
    }


    public void signUp(SignUpRequest signUpInfo, final OnResultListener listener) {

        Call<SignUpResponse> call = getApiService().signUp(signUpInfo);
        call.enqueue(new Callback<SignUpResponse>() {
            @Override
            public void onResponse(Call<SignUpResponse> call, Response<SignUpResponse> response) {

                if (response.isSuccessful()) {
                    // login request succeed, new token generated
                    SignUpResponse signUpResponse = response.body();
                    if(signUpResponse != null) {
                        prefManager.saveStringValue(PrefManager.SEARCH_URL, signUpResponse.getResults().getSearch().getUrl());
                        prefManager.saveStringValue(PrefManager.SEARCH_API_KEY, signUpResponse.getResults().getSearch().getApiKey());
                        prefManager.saveStringValue(UID, signUpResponse.getResults().getAuthorization().getSdk());
                        prefManager.saveLongValue(PrefManager.SIGN_UP_TIME, Calendar.getInstance().getTimeInMillis());
                        return;
                    }
                }

                listener.onSuccess(false);
            }

            @Override
            public void onFailure(Call<SignUpResponse> call, Throwable t) {
                listener.onSuccess(false);
            }
        });

    }



    public ParsingRule syncParsingRule() {
        try {
            Response<ParsingRule> response = getApiService().syncParsingRule(prefManager.loadIntValue(PARSING_RULE_VERSION, -1)).execute();
            if(response.isSuccessful()) {
                ParsingRule parsingRule = response.body();
                if(parsingRule != null)
                    prefManager.saveIntValue(PARSING_RULE_VERSION, parsingRule.ruleVersion);
                return response.body();
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        return null;
    }

    public SearchCompanyResponse searchCompany(ArrayList<Transaction> transactions) {
        try {
            Response<SearchCompanyResponse> response = getSearchApiService().searchCompany(new SearchCompanyRequest(Mapper.toSearchTransaction(transactions))).execute();
            if(response.isSuccessful()) {
                SearchCompanyResponse searchCompanyResponse = response.body();
                // db update
                // transaction.update(searchCompanyResponse, transactions);

                return searchCompanyResponse;
            }
        } catch (IOException e) {
            e.printStackTrace();
        }

        return null;
    }


    public void searchCompany(ArrayList<Transaction> transactions, final OnNetworkResultListener callback) {
        Call<SearchCompanyResponse> call = getSearchApiService().searchCompany(new SearchCompanyRequest(Mapper.toSearchTransaction(transactions)));
        call.enqueue(new Callback<SearchCompanyResponse>() {
            @Override
            public void onResponse(Call<SearchCompanyResponse> call, Response<SearchCompanyResponse> response) {
                SearchCompanyResponse searchCompanyResponse = response.body();
                // identifier 기준으로 저장하기 company 및 카테고리 정보 저장하기
                // db update
                // transaction.update(searchCompanyResponse, transactions);

                if(callback != null) callback.onResult(true);
            }

            @Override
            public void onFailure(Call<SearchCompanyResponse> call, Throwable t) {
                if(callback != null) callback.onResult(false);

            }
        });
    }


    public void saveTransactions(TransactionRequest transactionRequest, Callback<Void> callback) {
        Call<Void> call = getApiService().saveTransactions(transactionRequest);
        call.enqueue(callback);
    }



}
