package com.tenqube.visual_third.ui;

import android.Manifest;
import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.graphics.Color;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.os.Message;
import android.support.annotation.NonNull;
import android.support.v4.content.LocalBroadcastManager;
import android.support.v4.widget.SwipeRefreshLayout;
import android.support.v7.widget.CardView;
import android.text.TextUtils;
import android.view.Gravity;
import android.view.View;
import android.webkit.WebView;
import android.webkit.WebViewClient;
import android.widget.FrameLayout;
import android.widget.LinearLayout;
import android.widget.Toast;

import com.tenqube.visual_third.Callback;
import com.tenqube.visual_third.Constants;
import com.tenqube.visual_third.R;
import com.tenqube.visual_third.VisualServiceImpl;
import com.tenqube.visual_third.analysis.AnalysisServiceImpl;
import com.tenqube.visual_third.api.VisualApi;
import com.tenqube.visual_third.exception.AuthException;
import com.tenqube.visual_third.exception.ParameterException;
import com.tenqube.visual_third.manager.AdManager;
import com.tenqube.visual_third.manager.AnswerManager;
import com.tenqube.visual_third.manager.ExcelManager;
import com.tenqube.visual_third.manager.PrefManager;
import com.tenqube.visual_third.manager.ResourceManager;
import com.tenqube.visual_third.manager.StatusManager;
import com.tenqube.visual_third.manager.VisualAlarmManager;
import com.tenqube.visual_third.model.js.LogRequest;
import com.tenqube.visual_third.model.parser.PopupData;
import com.tenqube.visual_third.model.ui.TransactionPopupInfo;
import com.tenqube.visual_third.parser.loader.BulkLoader;
import com.tenqube.visual_third.repository.VisualRepository;
import com.tenqube.visual_third.util.PathUtil;
import com.tenqube.visual_third.util.Utils;
import com.tenqube.visual_third.util.permission.AfterPermissionGranted;
import com.tenqube.visual_third.util.permission.PermissionUtil;
import com.tenqube.visual_third.web.ActionImpl;
import com.tenqube.visual_third.web.ErrorImpl;
import com.tenqube.visual_third.web.LogImpl;
import com.tenqube.visual_third.web.RepoImpl;
import com.tenqube.visual_third.web.SystemImpl;
import com.tenqube.visual_third.web.UiImpl;
import com.tenqube.visual_third.web.VisualInterface;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.File;
import java.util.List;

import tenqube.parser.core.ParserService;

import static com.tenqube.visual_third.manager.AnswerManager.initFabric;
import static com.tenqube.visual_third.util.Utils.isNotiEnabled;
import static com.tenqube.visual_third.util.Utils.isSDKDisabled;
import static com.tenqube.visual_third.util.Validator.notNull;
import static com.tenqube.visual_third.util.WebViewHelper.getBaseUrl;
import static com.tenqube.visual_third.util.WebViewHelper.getProgressUrl;
import static com.tenqube.visual_third.util.WebViewHelper.settings;
import static tenqube.parser.core.ParserService.mIsDebug;
import static tenqube.parser.util.LogUtil.LOGI;

public class VisualWebActivity extends BaseActivity implements TimePickerFragment.Callback,
        DatePickerFragment.Callback,
        PermissionUtil.PermissionCallbacks,
        ActivityCallback{

    public static final String ARG_PATH = "ARG_PATH";

    public static final String TAG = VisualServiceImpl.class.getSimpleName();

    public static final int PERMISSION_REQUEST_CODE = 0;
    public static final int NOTIFICATION_REQUEST_CODE = 10;
    public static final int VISUAL_REQUEST_CODE = 512;

    public static final int FINISH = 10;
    public static final int PROGRESS = 11;
    public static final int SHOW_PROGRESS_BAR = 12;
    public static final int ERROR = 13;
    private PrefManager prefManager;
    private VisualApi api;
    private VisualRepository repository;
    private ResourceManager resourceManager;
    private VisualAlarmManager alarmManager;
    private boolean isPageLoaded;
    private String mFailingUrl;

    private String path;
    private WebView webview;
    private BulkLoader bulkLoader;

    private VisualInterface.Repo repoInterface;
    private VisualInterface.UI uiInterface;
    private VisualInterface.System systemInterface;
    private VisualInterface.Error errorInterface;
    private VisualInterface.Log logInterface;
    private VisualInterface.Action actionInterface;

    private SwipeRefreshLayout swipeRefreshLayout;
    private LinearLayout errorView;
    private boolean isError;
    private FrameLayout webContainer;
    private CardView adContainer;

    private boolean isLoading;

    @SuppressLint({"JavascriptInterface", "AddJavascriptInterface", "SetJavaScriptEnabled"})
    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        try {
            if(isSDKDisabled(getApplicationContext())) {
                throw new Exception("SDK disabled");
            }

            initFabric(VisualWebActivity.this);
            AnswerManager.onKeyMetric(new LogRequest("VisualWebActivity"));
            api = VisualApi.getInstance(getApplicationContext());
            repository = VisualRepository.getInstance(getApplicationContext());
            prefManager = PrefManager.getInstance(getApplicationContext());
            resourceManager = ResourceManager.getInstance(api, prefManager, ParserService.getInstance(getApplicationContext()));
            alarmManager = VisualAlarmManager.getInstance(getApplicationContext());

            path = getIntent().getStringExtra(ARG_PATH);
            if(path == null) path = "";
            LOGI("PATH", path, true);
            setContentView(R.layout.activity_visual_web);

            swipeRefreshLayout = findViewById(R.id.swipe_refresh_layout);
            swipeRefreshLayout.setEnabled(false);
            swipeRefreshLayout.setColorSchemeResources(
                    R.color.colorPopupRed,
                    R.color.colorPopupRed,
                    R.color.colorPopupRed);

            swipeRefreshLayout.setOnRefreshListener(new SwipeRefreshLayout.OnRefreshListener() {
                @Override
                public void onRefresh() {
                    swipeRefreshLayout.setRefreshing(false);
                    AnalysisServiceImpl.shouldRefresh = true;
                    webview.reload();

                }
            });
            webContainer = findViewById(R.id.web_container);
            //error view 처리
            errorView = findViewById(R.id.error_container);
            errorView.setVisibility(View.GONE);

            webview = findViewById(R.id.visual_web_view);

            // javascriptInterface
            errorInterface = new ErrorImpl(this, webview);
            logInterface = new LogImpl(this, webview, errorInterface);

            repoInterface = new RepoImpl(this,
                    webview,
                    errorInterface,
                    repository,
                    VisualApi.getInstance(getApplicationContext()),
                    new AnalysisServiceImpl(this, ""), alarmManager);
            uiInterface = new UiImpl(this,
                    webview,
                    errorInterface,
                    new BottomDialog(this),
                    AdManager.getInstance(getApplicationContext()));

            systemInterface = new SystemImpl(this, webview, errorInterface);
            actionInterface = new ActionImpl(this, webview);

            webview.addJavascriptInterface(errorInterface, "visualError");
            webview.addJavascriptInterface(repoInterface, "visualRepo");
            webview.addJavascriptInterface(uiInterface, "visualUI");
            webview.addJavascriptInterface(systemInterface, "visualSystem");
            webview.addJavascriptInterface(logInterface, "visualLog");
            webview.addJavascriptInterface(actionInterface, "visualAction");

            // settings
            settings(webview, repository.isDev());

            // error
            webview.setWebViewClient(new WebViewClient() {
                public void onReceivedError(WebView view, int errorCode, String description, String failingUrl) {
                    mFailingUrl = failingUrl;
                    loadErrorUrl();
                }
            });

            if(shouldRequestOverlayPermission()) {
                VisualPermissionActivity.startActivity(this, PERMISSION_REQUEST_CODE);
            } else {
                startVisual();
            }

        } catch (Exception e) {
            finish();
        }

        StatusManager.getInstance().setOnFinishListener(new OnFinishListener() {
            @Override
            public void onFinish() {
                finish();
            }
        });
    }

    private void startVisual() {
        loadUrl(path);

        if(!prefManager.isEnabled(PrefManager.BULK_EXECUTED, false)) { // 벌크 파싱 진행

            LOGI(TAG, "벌크 파싱 진행", mIsDebug);
            startBulkParsing();

        } else {

            alarmManager.setAlarms();
            // 리소스 동기화
            resourceManager.sync();
        }
    }

    @Override
    public void exportExcel(Callback<Boolean> callback) {
        this.completion = callback;
        exportExcel();
    }

    @Override
    public Context getContext() {
        return getApplicationContext();
    }

    @Override
    public Activity getActivity() {
        return VisualWebActivity.this;
    }

    public void onPagedLoaded() {
        runOnUiThread(new Runnable() {
            @Override
            public void run() {
                isPageLoaded = true;
            }
        });
    }

    private void loadUrl(String path) {
        if(webview != null) {
            isPageLoaded = false;
            if(path == null) path = "";
            String url = getBaseUrl(prefManager) + path;
            webview.loadUrl(url);
        }
    }

    private void loadErrorUrl() {
        isPageLoaded = false;
        if(swipeRefreshLayout != null) swipeRefreshLayout.setEnabled(false);
        errorView.setVisibility(View.VISIBLE);
    }

    private BroadcastReceiver transactionReceiver = new BroadcastReceiver() {

        @Override
        public void onReceive(Context context, final Intent intent) {

            try {
                final int tranId = intent.getIntExtra("tranId", 0);

                webview.post(new Runnable() {
                    @Override
                    public void run() {
                        if(tranId != 0)
                            webview.loadUrl("javascript:window.onTransactionReceived" + "(" + tranId + ");");
                    }
                });

            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    };

    @Override
    public void onResume() {
        super.onResume();
        LocalBroadcastManager.getInstance(this).registerReceiver(transactionReceiver, new IntentFilter(Constants.ACTION_TRANSACTION_RECEIVED));
        repository.setVisualActive(true);
        overridePendingTransition(0, 0);
    }

    @Override
    public void onPause() {
        super.onPause();
        LocalBroadcastManager.getInstance(this).unregisterReceiver(transactionReceiver);
        repository.setVisualActive(false);
        overridePendingTransition(0, 0);
    }

    private Handler handler = new Handler(new Handler.Callback() { // 핸들러로 전달받은 값 progress 값 세팅해주기
        @Override
        public boolean handleMessage(final Message msg) {


            try {
                switch (msg.what) {
                    case PROGRESS:
                        if(webview != null) {
                            Bundle bundle = msg.getData();
                            if(bundle != null) {
                                webview.loadUrl("javascript:window.onProgress" + "(" + bundle.getInt("now") + "," + bundle.getInt("total") + ");");
                            }
                        }
                        break;

                    case SHOW_PROGRESS_BAR:
                        isLoading = true;
                        loadProgressUrl();
                        break;
                    case FINISH:
                        onBulkFinish(true);
                        break;


                    case ERROR:
                    default:
                        onBulkFinish(false);
                        break;
                }
            } catch (Exception e) {
                onBulkFinish(false);
            }

            return false;
        }
    });


    private void onBulkFinish(boolean isSuccess) {
        isLoading = false;
        prefManager.saveBoolean(PrefManager.BULK_EXECUTED, true);
        if(webview != null) {
            startVisual();
        }
        if(completion != null) {
            completion.onDataLoaded(isSuccess);
            completion = null;
        }

    }

    private void loadProgressUrl() {
        if(webview != null) {
            isPageLoaded = false;
            String webUrl = getProgressUrl(prefManager, Constants.ProgressType.BULK);
            webview.loadUrl(webUrl);
        }
    }

    /**
     * 1. 퍼미션 체크
     * 2. 프로그레스 다이얼로그 show
     * 3. 지난 6개월 문자 파싱 시작
     */
    private void startBulkParsing() {

        new Thread(new Runnable() {
            @Override
            public void run() {
                try {
                    handler.sendEmptyMessage(SHOW_PROGRESS_BAR);

                    LOGI(TAG, "startBulkParsing start", mIsDebug);
                    bulkLoader = new BulkLoader(VisualWebActivity.this, handler, new BulkLoader.OnSmsLoadFinished() {
                        @Override
                        public void onFinished(boolean isError) {
                            try {
                                if(handler != null) handler.sendEmptyMessage(FINISH);
                            } catch (Exception e) {
                                e.printStackTrace();
                            }
                        }
                    });

                    bulkLoader.doParsing();

                } catch (Exception e) {
                    handler.sendEmptyMessage(ERROR);
                }

            }
        }).start();
    }

    @Override
    protected void onActivityResult(int requestCode, int resultCode, final Intent data) {
        super.onActivityResult(requestCode, resultCode, data);

        switch (requestCode) {
            case PERMISSION_REQUEST_CODE:

                if(shouldRequestOverlayPermission()) {
                    finish();
                } else {
                    startVisual();
                }

                break;
            case NOTIFICATION_REQUEST_CODE:
                boolean isEnabled = isNotiEnabled(getApplicationContext());

                JSONObject jsonObject = new JSONObject();
                try {
                    jsonObject.put("isNotiEnabled", isEnabled);
                    String json = jsonObject.toString();
                    if(systemInterface != null) systemInterface.onNotiCallback(json);

                } catch (JSONException e) {
                    e.printStackTrace();
                }
                break;

        }
    }

    @Override
    public void onBackPressed() {

        try {
            if(isPageLoaded && uiInterface != null) {
                uiInterface.onFinish();
            } else {
                super.onBackPressed();
            }
        } catch (Exception e) {
            super.onBackPressed();
        }
    }

    @Override
    public void onCalendar(String date, String callback) {
        uiInterface.onCalendar(date, callback);
    }

    public void onScrollChanged(final int pos) {
        return;
    }

    public void setRefreshEnabled(final boolean enabled) {
        swipeRefreshLayout.post(new Runnable() {
            @Override
            public void run() {
                swipeRefreshLayout.setEnabled(enabled);
            }
        });
    }

    @Override
    public void addAd(View view) {
        try {
            webContainer.addView(createCardView(view));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private CardView createCardView(View view) {
        adContainer = new CardView(this);
        adContainer.setCardElevation(getResources().getDimension(R.dimen.elevation));
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            adContainer.setElevation(getResources().getDimension(R.dimen.elevation));
        }

        // Set the CardView layoutParams
        FrameLayout.LayoutParams params = new FrameLayout.LayoutParams(
                FrameLayout.LayoutParams.WRAP_CONTENT,
                FrameLayout.LayoutParams.WRAP_CONTENT
        );



        params.setMargins(Utils.dpToPx(10), Utils.dpToPx(16), Utils.dpToPx(10), Utils.dpToPx(16));
        params.gravity = Gravity.BOTTOM;
        adContainer.setLayoutParams(params);

        // Set CardView corner radius
        adContainer.setRadius(Utils.dpToPx(13));

        // Set a background color for CardView
        adContainer.setCardBackgroundColor(Color.parseColor("#00000000"));


        adContainer.addView(view);

        return adContainer;

    }

    public void hideAd() {
        try {
            runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    if(adContainer != null) {
                        webContainer.removeView(adContainer);
                    }

                }
            });

        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    public TransactionPopupInfo getPopUpInfo() {
        return null;
    }

    @Override
    public void setStatusBarColor(String color) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            try {
                getWindow().setStatusBarColor(Color.parseColor(color));
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    public void reload() {
        if(webview != null)  {
            webview.post(new Runnable() {
                @Override
                public void run() {
                    webview.reload();
                }
            });
        }
    }

    public void retry() {
        if(webview != null)  {
            webview.post(new Runnable() {
                @Override
                public void run() {
                    webview.loadUrl(mFailingUrl);
                }
            });
        }
    }

    public static String[] PERMISSIONS_STORAGE = {Manifest.permission.READ_EXTERNAL_STORAGE, Manifest.permission.WRITE_EXTERNAL_STORAGE};
    public static final int STORAGE = 1;

    public Callback<Boolean> completion;

    @AfterPermissionGranted(STORAGE)
    public void exportExcel() {
        if (PermissionUtil.hasPermissions(
                this, PERMISSIONS_STORAGE)) {

            ExcelManager.getInstance(VisualWebActivity.this).export(new ExcelManager.Callback() {
                @Override
                public void onCompleted(File file) {

                    boolean isSuccess = file != null;
                    if(isSuccess) {
                        // 인텐트 공유
                        String applicationId = prefManager.loadStringValue(PrefManager.APPLICATION_ID,
                                getApplicationContext().getPackageName());
                        try {
                            skipFlag = true;

                            Intent intent = new Intent(Intent.ACTION_SEND);
                            intent.setType("application/octet-stream");
                            intent.addFlags(Intent.FLAG_ACTIVITY_NO_USER_ACTION);

                            intent.putExtra(Intent.EXTRA_STREAM, Utils.getUri(getApplicationContext(), applicationId, file));
                            startActivity(Intent.createChooser(intent, "공유"));
                            Toast.makeText(getApplicationContext(),getString(R.string.export_excel_success_msg, file.getName()), Toast.LENGTH_SHORT).show();

                        } catch (Exception e){
                            e.printStackTrace();
                            String log = "pkgName:" + applicationId  + "\n FileProvider Error";
                            Toast.makeText(getApplicationContext(), log, Toast.LENGTH_SHORT).show();
                            completion.onDataLoaded(false);
                            return;
                        }
                    }
                    completion.onDataLoaded(isSuccess);
                }
            });

        } else {

            PermissionUtil.requestPermissions(
                    this,
                    STORAGE,
                    PERMISSIONS_STORAGE);
        }
    }

    @Override
    public void onPermissionsGranted(int requestCode, List<String> list) {

        if (requestCode == STORAGE) {
            exportExcel();
        }
    }

    @Override
    public void onPermissionsDenied(int requestCode, List<String> list) {
        Toast.makeText(getApplicationContext(), "엑셀 파일 내보내기를 위해 외부 저장소에 대한 권한이 필요합니다. 권한이 없는 경우 엑셀 파일 내보내기 사용이 불가합니다.", Toast.LENGTH_SHORT).show();
    }

    @Override
    public void onRequestPermissionsResult(int requestCode, String[] permissions, int[] grantResults) {
        super.onRequestPermissionsResult(requestCode, permissions, grantResults);

        // Forward results to EasyPermissions
        PermissionUtil.onRequestPermissionsResult(requestCode, permissions, grantResults, this);
    }

    public static void startActivity(Activity activity, String path) throws AuthException {
        if(TextUtils.isEmpty(PrefManager.getInstance(activity).loadStringValue(PrefManager.UID, "")))
            throw new AuthException("Need to sign up");

        BaseActivity.skipFlag = true;
        Intent intent = new Intent(activity, VisualWebActivity.class);
        intent.putExtra(ARG_PATH, path);
        intent.addFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP |
                Intent.FLAG_ACTIVITY_REORDER_TO_FRONT);
        activity.startActivityForResult(intent, VISUAL_REQUEST_CODE);
    }

    public static void startActivity(Context context, String path) throws AuthException {
        if(TextUtils.isEmpty(PrefManager.getInstance(context).loadStringValue(PrefManager.UID, "")))
            throw new AuthException("Need to sign up");

        BaseActivity.skipFlag = true;
        Intent intent = new Intent(context, VisualWebActivity.class);
        intent.putExtra(ARG_PATH, path);

        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK |
                Intent.FLAG_ACTIVITY_CLEAR_TOP |
                Intent.FLAG_ACTIVITY_REORDER_TO_FRONT);
        context.startActivity(intent);
    }

    public static void startActivity(@NonNull Context context, @NonNull PopupData popupData) throws AuthException, ParameterException {

        notNull(context);
        notNull(popupData);

        if(TextUtils.isEmpty(PrefManager.getInstance(context).loadStringValue(PrefManager.UID, "")))
            throw new AuthException("Need to sign up");
        BaseActivity.skipFlag = true;

        Intent intent = new Intent(context, VisualWebActivity.class);
        intent.putExtra(ARG_PATH, PathUtil.makePath(popupData));
        intent.addFlags(Intent.FLAG_ACTIVITY_NO_USER_ACTION);

        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK |
                Intent.FLAG_ACTIVITY_CLEAR_TOP |
                Intent.FLAG_ACTIVITY_REORDER_TO_FRONT);
        context.startActivity(intent);
    }

    public boolean shouldRequestOverlayPermission() {
        return PermissionUtil.shouldRequestOverlayPermission(getApplicationContext()) && repository.shouldShowTranPopup();
    }

}
