package com.tenqube.visual_third.manager;

import android.content.Context;
import android.content.Intent;
import android.graphics.drawable.Drawable;
import android.net.Uri;
import android.os.Handler;
import android.support.annotation.NonNull;
import android.text.TextUtils;
import android.view.LayoutInflater;
import android.view.View;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;

import com.bumptech.glide.Glide;
import com.bumptech.glide.request.target.BitmapImageViewTarget;
import com.google.android.gms.ads.AdListener;
import com.google.android.gms.ads.AdLoader;
import com.google.android.gms.ads.AdRequest;
import com.google.android.gms.ads.MobileAds;
import com.google.android.gms.ads.VideoOptions;
import com.google.android.gms.ads.formats.NativeAdOptions;
import com.google.android.gms.ads.formats.UnifiedNativeAd;
import com.google.android.gms.ads.formats.UnifiedNativeAdView;
import com.tenqube.visual_third.OnAdLoadedListener;
import com.tenqube.visual_third.R;
import com.tenqube.visual_third.api.VisualApi;
import com.tenqube.visual_third.entity.Advertisement;
import com.tenqube.visual_third.model.api.AdResponse;
import com.tenqube.visual_third.model.api.VersionResponse;
import com.tenqube.visual_third.repository.VisualRepository;
import com.tenqube.visual_third.ui.VisualWebActivity;
import com.tenqube.visual_third.util.GoOpenMarketUtil;
import com.tenqube.visual_third.util.Utils;

import java.util.ArrayList;

import static com.tenqube.visual_third.api.VisualApi.API_KEY;

public class AdManager {

    public static final String AD_ID = "ca-app-pub-1003660361092577~4657260383";
    public static final String UNIT_ID = "ca-app-pub-1003660361092577/4309119488";

    public static final String TEST_UNIT_ID = "ca-app-pub-3940256099942544/2247696110";
    private static AdManager mInstance = null;
    private VisualRepository repository;
    private VisualApi callApi;
    private Context mContext;
    private View defaultView;
    private View adContainer;
    private UnifiedNativeAdView mAdView;

    public static AdManager getInstance(Context ctx) {

        synchronized (AdManager.class) {
            if (mInstance == null) {
                mInstance = new AdManager(ctx.getApplicationContext());
            }
        }
        return mInstance;
    }

    private AdManager(Context context) {
        this.mContext = context;

        MobileAds.initialize(context.getApplicationContext(), AD_ID);
        this.callApi = VisualApi.getInstance(context);
        this.repository = VisualRepository.getInstance(context);
    }

    /**
     * 1. Visual Api 호출
     * 2. 만족하는 광고 체크
     * 3. 없는경우 구글 admob 호출
     * 4. 로드 실패한경우 디폴트 광고 호출
     * @param listener
     */
    public void loadAd(@NonNull OnAdLoadedListener listener) {
        loadVisualAd(listener);
    }

    /**
     * 1. 버전 체크
     * 2. 버전 다른 경우 동기화 요청
     * 3. 동기화
     * 4. 광고 로드
     * 5. 조건에 만족하는지 체크
     * 6. 만족하는 광고
     * @param listener
     */
    private void loadVisualAd(final OnAdLoadedListener listener) {

        try {
            String[] authInfos = callApi.getAuthInfo();

            VersionResponse versionResponse = callApi.syncVersion(authInfos[API_KEY]);

            // version
            int version = repository.loadAdVersion();
            if(versionResponse != null && versionResponse.getAd() > version) {
                AdResponse adResponse = callApi.getAds(versionResponse.getAd(), authInfos[API_KEY]);
                if(adResponse != null) {
                    defaultView = null;
                    repository.mergeAdvertisement(adResponse, versionResponse.getAd());
                }
            }
            // loadAd
            ArrayList<Advertisement> ads = repository.loadAds();

            // isSatisfied
            final ArrayList<Advertisement> satisfiedAds = new ArrayList<>();

            boolean isSatisfied;
            int priority = -1;
            for(Advertisement ad : ads) {

                if(priority != -1 && priority != ad.getPriority())
                    break;

                isSatisfied = repository.isSatisfied(ad.getQuery());

                if(isSatisfied) {
                    satisfiedAds.add(ad);
                }

                priority = ad.getPriority();
            }

            new Handler(mContext.getMainLooper()).post(new Runnable() {
                public void run() {
                    // main thread 처리
                    if(!satisfiedAds.isEmpty()) {
                        // priority
                        Advertisement advertisement = satisfiedAds.get(Utils.getRandomNumber(satisfiedAds.size()));
                        populateView(advertisement);
                        listener.onAdLoaded(adContainer);

                    } else {
                        if(mAdView == null) {
                            loadGoogleAd(listener);
                        } else {
                            listener.onAdLoaded(mAdView);
                        }
                        loadGoogleAd(null);
                    }
                }
            });

        } catch (Exception e) {
            loadGoogleAd(listener);
        }
    }

    private void populateView(final Advertisement advertisement) {

        adContainer = LayoutInflater.from(mContext).inflate(R.layout.ad_view, null, false);
        ImageView imageTextView = adContainer.findViewById(R.id.image);
        LinearLayout adTextContainer = adContainer.findViewById(R.id.ad_text_container);

        adContainer.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                goIntent(advertisement);
            }
        });

        if(!TextUtils.isEmpty(advertisement.getImage())) {
            imageTextView.setVisibility(View.VISIBLE);
            Glide.with(mContext).load(advertisement.getImage()).into(imageTextView);

            adTextContainer.setVisibility(View.GONE);
        } else {
            imageTextView.setVisibility(View.GONE);
            adTextContainer.setVisibility(View.VISIBLE);

            ImageView iconImageView = adContainer.findViewById(R.id.icon_image);
            Glide.with(mContext).load(advertisement.getIconImage()).into(iconImageView);

            TextView labelTextView = adContainer.findViewById(R.id.label);
            labelTextView.setText(advertisement.getLabel());
            TextView titleTextView = adContainer.findViewById(R.id.title);
            titleTextView.setText(advertisement.getTitle());
            TextView contentTextView = adContainer.findViewById(R.id.content);
            contentTextView.setText(advertisement.getContent());
            TextView linktoTextView = adContainer.findViewById(R.id.linkto);
            linktoTextView.setText(advertisement.getLinkToStr());

        }
    }

    private void populateUnifiedNativeAdView(UnifiedNativeAd nativeAd, UnifiedNativeAdView adView) {

        adView.setHeadlineView(adView.findViewById(R.id.title));
        adView.setIconView(adView.findViewById(R.id.icon_image));
        adView.setBodyView(adView.findViewById(R.id.content));
        adView.setCallToActionView(adView.findViewById(R.id.linkto));


        // The headline is guaranteed to be in every UnifiedNativeAd.
        ((TextView) adView.getHeadlineView()).setText(nativeAd.getHeadline());
        // These assets aren't guaranteed to be in every UnifiedNativeAd, so it's important to
        // check before trying to display them.
        if (nativeAd.getBody() == null) {
            adView.getBodyView().setVisibility(View.INVISIBLE);
        } else {
            adView.getBodyView().setVisibility(View.VISIBLE);
            ((TextView) adView.getBodyView()).setText(nativeAd.getBody());
        }

        if (nativeAd.getCallToAction() == null) {
            adView.getCallToActionView().setVisibility(View.INVISIBLE);
        } else {
            adView.getCallToActionView().setVisibility(View.VISIBLE);
            ((TextView) adView.getCallToActionView()).setText(nativeAd.getCallToAction());
        }

        if (nativeAd.getIcon() == null) {
            adView.getIconView().setVisibility(View.GONE);
        } else {
            ((ImageView) adView.getIconView()).setImageDrawable(
                    nativeAd.getIcon().getDrawable());
            adView.getIconView().setVisibility(View.VISIBLE);
        }

//        if (nativeAd.getPrice() == null) {
//            adView.getPriceView().setVisibility(View.INVISIBLE);
//        } else {
//            adView.getPriceView().setVisibility(View.VISIBLE);
//            ((TextView) adView.getPriceView()).setText(nativeAd.getPrice());
//        }
//
//        if (nativeAd.getStore() == null) {
//            adView.getStoreView().setVisibility(View.INVISIBLE);
//        } else {
//            adView.getStoreView().setVisibility(View.VISIBLE);
//            ((TextView) adView.getStoreView()).setText(nativeAd.getStore());
//        }
//
//        if (nativeAd.getStarRating() == null) {
//            adView.getStarRatingView().setVisibility(View.INVISIBLE);
//        } else {
//            ((RatingBar) adView.getStarRatingView())
//                    .setRating(nativeAd.getStarRating().floatValue());
//            adView.getStarRatingView().setVisibility(View.VISIBLE);
//        }
//
//        if (nativeAd.getAdvertiser() == null) {
//            adView.getAdvertiserView().setVisibility(View.INVISIBLE);
//        } else {
//            ((TextView) adView.getAdvertiserView()).setText(nativeAd.getAdvertiser());
//            adView.getAdvertiserView().setVisibility(View.VISIBLE);
//        }

        // This method tells the Google Mobile Ads SDK that you have finished populating your
        // native ad view with this native ad. The SDK will populate the adView's MediaView
        // with the media content from this native ad.
        adView.setNativeAd(nativeAd);

    }

    /**
     * Creates a request for a new native ad based on the boolean parameters and calls the
     * corresponding "populate" method when one is successfully returned.
     *
     */
    private void loadGoogleAd(final OnAdLoadedListener listener) {

        AdLoader.Builder builder = new AdLoader.Builder(mContext, UNIT_ID);

        builder.forUnifiedNativeAd(new UnifiedNativeAd.OnUnifiedNativeAdLoadedListener() {
            @Override
            public void onUnifiedNativeAdLoaded(UnifiedNativeAd unifiedNativeAd) {

                UnifiedNativeAdView adView = (UnifiedNativeAdView) LayoutInflater.from(mContext)
                        .inflate(R.layout.admob_view, null);

                populateUnifiedNativeAdView(unifiedNativeAd, adView);

                if(listener != null) {
                    listener.onAdLoaded(adView);
                    mAdView = null;
                } else {
                    mAdView = adView;
                }
            }

        });

        VideoOptions videoOptions = new VideoOptions.Builder()
                .setStartMuted(true)
                .build();

        NativeAdOptions adOptions = new NativeAdOptions.Builder()
                .setVideoOptions(videoOptions)
                .build();

        builder.withNativeAdOptions(adOptions);

        AdLoader adLoader = builder.withAdListener(new AdListener() {
            @Override
            public void onAdFailedToLoad(int errorCode) {
                mAdView = null;
                if(defaultView == null) loadDefaultAd();
                if(listener != null) listener.onAdLoaded(defaultView);
            }
        }).build();

        adLoader.loadAd(new AdRequest.Builder().build());
    }

    private void loadDefaultAd() {
        Advertisement defaultAd = repository.loadAd(0);
        if(defaultAd == null) {
            defaultAd = new Advertisement(0, "", "광고", "", "visual", "app", "", "http://images.myqlip.com/banner.webp", "", 1, "");
        }

        defaultView = LayoutInflater.from(mContext).inflate(R.layout.ad_default_view, null, false);

        final Advertisement finalDefaultAd = defaultAd;
        defaultView.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                goIntent(finalDefaultAd);
            }
        });

        final ImageView imageView = defaultView.findViewById(R.id.image);
        Glide.with(mContext)
                .load(defaultAd.getImage())
                .asBitmap()
                .into(new BitmapImageViewTarget(imageView) {
                    @Override
                    public void onLoadFailed(Exception e, Drawable errorDrawable) {
                        imageView.setImageResource(R.drawable.banner);
                    }
                });

    }

    private void goIntent(Advertisement ad) {

        try {

            if("app".equals(ad.getLinkToType())) {

                VisualWebActivity.startActivity(mContext, "");

            } else if("market".equals(ad.getLinkToType())) {

                GoOpenMarketUtil.openOnGooglePlayMarket(mContext, ad.getLinkTo());

            } else if("web".equals(ad.getLinkToType())) {
                Intent i = new Intent(Intent.ACTION_VIEW);
                i.setData(Uri.parse(ad.getLinkTo()));
                mContext.startActivity(i);
            }

        } catch (Exception e) {
            e.printStackTrace();
        }
    }
}
