package com.tenqube.visual_third.manager;

import android.app.Notification;
import android.app.NotificationChannel;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.net.Uri;
import android.os.Build;

import androidx.core.app.NotificationCompat;
import androidx.core.app.NotificationManagerCompat;
import androidx.core.app.TaskStackBuilder;

import com.tenqube.visual_third.Constants;
import com.tenqube.visual_third.R;
import com.tenqube.visual_third.entity.VisualNotification;
import com.tenqube.visual_third.repository.RepositoryHolder;
import com.tenqube.visual_third.repository.VisualRepository;
import com.tenqube.visual_third.ui.VisualWebActivity;
import com.tenqube.visual_third.util.Utils;

import java.util.Calendar;

import static com.tenqube.visual_third.manager.PrefManager.NOTI_CHANNEL;
import static com.tenqube.visual_third.manager.PrefManager.NOTI_COLOR;
import static com.tenqube.visual_third.manager.PrefManager.NOTI_ICON_RES;
import static com.tenqube.visual_third.util.Utils.DECIMAL_FORMAT;
import static com.tenqube.visual_third.util.Utils.getYMD;
import static com.tenqube.visual_third.util.Utils.isSDKDisabled;
import static com.tenqube.visual_third.util.Utils.threeComma;

public class VisualNotificationManager {

    public static final String TAG = VisualNotificationManager.class.getSimpleName();
    public static final int NOTI_ID = 1080;
    private VisualRepository repository;
    private PrefManager prefManager;
    private Context context;

    private static VisualNotificationManager mInstance = null;

    public static VisualNotificationManager getInstance(Context ctx) {
        synchronized (VisualNotificationManager.class) {
            if (mInstance == null) {
                mInstance = new VisualNotificationManager(ctx.getApplicationContext());
            }
        }
        return mInstance;
    }

    private VisualNotificationManager(Context context) {
        this.context = context;
        repository = RepositoryHolder.getInstance(context).getVisualRepository();
        prefManager = PrefManager.getInstance(context);
    }

    public void showNotification(VisualNotification notification) {
        Utils.LOGD("showNotification", "notification: start");

        if(isSDKDisabled(context)) return;

        String[] dates = getDateRanges(notification); // 날짜 범위

        double sum = repository.getSum(dates[0], dates[1]);
        notifyNotification(notification.getTicker().contains("%") ?
                        String.format(notification.getTicker(), threeComma(sum))
                :
                notification.getTicker(),
                notification.getTitle(),
                notification.getContent().contains("%") ?
                        String.format(notification.getContent(), threeComma(sum))
                        :
                        notification.getContent(),
                getStackBuilder(getPath(notification.getName())));

    }

    private String getPath(String name) {
        switch (name) {
            case Constants.NotificationType.MONTHLY:

                return "&monthly=true";

            case Constants.NotificationType.WEEKLY_FRI:
            case Constants.NotificationType.WEEKLY_MON:

                return "&weekly=true ";

            default:
                return "&daily=true";
        }
    }

    private String[] getDateRanges(VisualNotification notification) {
        String fromAt = "";
        String toAt = "";

        Calendar calendar = Calendar.getInstance();
        calendar.set(Calendar.HOUR, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);

        long currentTime = calendar.getTimeInMillis();

        int year = calendar.get(Calendar.YEAR);
        int month = calendar.get(Calendar.MONTH) + 1;
        int day = calendar.get(Calendar.DATE);

        int fromYear, fromMonth;

        switch (notification.getName()) {
            case "monthly":
                fromYear = year;
                fromMonth = month - 1;

                if(fromMonth == 0) {
                    fromYear = year - 1;
                    fromMonth = 12;
                }
                fromAt = fromYear + "-" + DECIMAL_FORMAT.format(fromMonth) + "-01";// 지난달 1일 'ymd'
                toAt = year + "-" + DECIMAL_FORMAT.format(month) + "-01";// 지난달 1일 'ymd'
                break;

            case "weekly_mon":
                // 지난 주 월 ~ 이번주 월

                calendar.set(Calendar.DAY_OF_WEEK, Calendar.MONTH);
                if(currentTime < calendar.getTimeInMillis()) {
                    calendar.add(Calendar.WEEK_OF_YEAR, -2);
                } else if (currentTime == calendar.getTimeInMillis()) {
                    calendar.add(Calendar.WEEK_OF_YEAR, -1);
                }

                fromAt = getYMD(calendar); // 지난주 월 'ymd'
                calendar.add(Calendar.WEEK_OF_YEAR, 1);
                toAt = getYMD(calendar); // 이번주 월 'ymd'

                break;

            case "weekly_fri":
                calendar.set(Calendar.DAY_OF_WEEK, Calendar.MONTH);
                if(currentTime < calendar.getTimeInMillis()) {
                    calendar.add(Calendar.WEEK_OF_YEAR, -1);
                }
                fromAt = getYMD(calendar); // 지난주 월 'ymd'

                calendar.add(Calendar.WEEK_OF_YEAR, 1);
                toAt = getYMD(calendar); // 이번주 월 'ymd'

                break;

            case "daily":
                fromAt = Utils.getYMD(calendar);
                calendar.add(Calendar.DATE, 1);
                toAt = Utils.getYMD(calendar);
                break;
        }

        return new String []{fromAt, toAt};
    }

    private void createNotificationChannel() {
        // Create the NotificationChannel, but only on API 26+ because
        // the NotificationChannel class is new and not in the support library
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            int importance = NotificationManager.IMPORTANCE_DEFAULT;
            NotificationChannel channel = new NotificationChannel(prefManager.loadStringValue(NOTI_CHANNEL, "visual"),
                    prefManager.loadStringValue(PrefManager.NOTI_CHANNEL_NAME, "visual"),
                    importance);

            // Register the channel with the system; you can't change the importance
            // or other notification behaviors after this
            NotificationManager notificationManager = context.getSystemService(NotificationManager.class);
            if(notificationManager != null) notificationManager.createNotificationChannel(channel);
        }
    }

    private boolean hasChannel() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {

            NotificationManager notificationManager = context.getSystemService(NotificationManager.class);
            if(notificationManager != null) return notificationManager.getNotificationChannel(prefManager.loadStringValue(NOTI_CHANNEL, "visual")) != null;

        }

        return true;
    }

    public Notification getNotification() {
        if(hasChannel()) {

            PendingIntent pi = PendingIntent.getActivity(context, 0, new Intent(context, VisualWebActivity.class), 0);
            Utils.LOGD("notifyNotification", "channel 존재여부 true");

            int color = prefManager.loadIntValue(NOTI_COLOR, 0);

            NotificationCompat.Builder mBuilder =
                    new NotificationCompat.Builder(context, prefManager.loadStringValue(NOTI_CHANNEL, NOTI_CHANNEL))
                            .setSmallIcon(prefManager.loadIntValue(NOTI_ICON_RES, R.drawable.ic_status))
                            .setAutoCancel(true)
                            .setContentIntent(pi)
                            .setDefaults(Notification.DEFAULT_LIGHTS | Notification.DEFAULT_SOUND)
                            .setVibrate(new long[]{0L}); // vibrate off


            if(color != 0) mBuilder.setColor(color);

            return mBuilder.build();
        } else {
            return null;
        }
    }

    public void dismiss() {
        NotificationManagerCompat notificationManager = NotificationManagerCompat.from(context);
        notificationManager.cancel(NOTI_ID);
    }

    private void notifyNotification(String ticker,
                                    String title,
                                    String content,
                                    TaskStackBuilder stackBuilder) {



        if(hasChannel()) {

            PendingIntent pi = stackBuilder.getPendingIntent(NOTI_ID, PendingIntent.FLAG_UPDATE_CURRENT);

//        createNotificationChannel();
            Utils.LOGD("notifyNotification", "channel 존재여부 true");

            int color = prefManager.loadIntValue(NOTI_COLOR, 0);

            NotificationCompat.Builder mBuilder =
                    new NotificationCompat.Builder(context, prefManager.loadStringValue(NOTI_CHANNEL, NOTI_CHANNEL))
                            .setSmallIcon(prefManager.loadIntValue(NOTI_ICON_RES, R.drawable.ic_status))
                            .setContentTitle(title)
                            .setContentText(content)
                            .setTicker(ticker)
                            .setAutoCancel(true)
                            .setContentIntent(pi)
                            .setDefaults(Notification.DEFAULT_LIGHTS | Notification.DEFAULT_SOUND)
                            .setVibrate(new long[]{0L}) // vibrate off
                            .setStyle(new NotificationCompat.BigTextStyle()
                                    .bigText(content));

            if(color != 0) mBuilder.setColor(color);

            NotificationManagerCompat notificationManager = NotificationManagerCompat.from(context);
            notificationManager.notify(NOTI_ID, mBuilder.build());
        } else {
            Utils.LOGD("notifyNotification", "channel 존재여부 false");

        }
    }

    private Intent getDeepLinkIntent(String path) {

        Intent intent = new Intent();
        intent.putExtra(VisualWebActivity.ARG_PATH, path);
        String deepLink = "visual://popup";
        intent.setAction(Intent.ACTION_VIEW);
        intent.setData(Uri.parse(deepLink));
        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK |
                Intent.FLAG_ACTIVITY_CLEAR_TOP |
                Intent.FLAG_ACTIVITY_REORDER_TO_FRONT);

        return intent;
    }

    private TaskStackBuilder getStackBuilder(String path) {


//        if(TextUtils.isEmpty(deepLink)) {
//            VisualWebActivity.startActivity(TransactionPopupActivity.this, path);
//        } else {
//            Intent intent = new Intent(Intent.ACTION_VIEW, Uri.parse(deepLink));
//            intent.putExtra(VisualWebActivity.ARG_PATH, path);
//            startActivity(intent);
//        }
        Intent intent = new Intent(context, VisualWebActivity.class);
        intent.putExtra(VisualWebActivity.ARG_PATH, path);
        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK |
                Intent.FLAG_ACTIVITY_CLEAR_TOP |
                Intent.FLAG_ACTIVITY_REORDER_TO_FRONT);

        TaskStackBuilder stackBuilder = TaskStackBuilder.create(context);
        stackBuilder.addNextIntent(intent);
        return stackBuilder;
    }

}
