package com.tenqube.visual_third.ui;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.app.AlertDialog;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.res.Configuration;
import android.graphics.Color;
import android.net.Uri;
import android.os.Bundle;
import android.provider.Settings;
import android.text.TextUtils;
import android.view.Gravity;
import android.view.SoundEffectConstants;
import android.view.View;
import android.webkit.ConsoleMessage;
import android.webkit.WebChromeClient;
import android.webkit.WebSettings;
import android.webkit.WebView;
import android.webkit.WebViewClient;
import android.widget.FrameLayout;
import android.widget.LinearLayout;
import android.widget.Toast;

import androidx.annotation.Nullable;
import androidx.appcompat.app.AppCompatActivity;
import androidx.cardview.widget.CardView;
import androidx.localbroadcastmanager.content.LocalBroadcastManager;
import androidx.swiperefreshlayout.widget.SwipeRefreshLayout;

import com.tenqube.visual_third.ActivityCallback;
import com.tenqube.visual_third.Constants;
import com.tenqube.visual_third.R;
import com.tenqube.visual_third.VisualServiceImpl;
import com.tenqube.visual_third.analysis.AnalysisServiceImpl;
import com.tenqube.visual_third.exception.AuthException;
import com.tenqube.visual_third.manager.AdManager;
import com.tenqube.visual_third.manager.PrefManager;
import com.tenqube.visual_third.manager.SecretKeyManager;
import com.tenqube.visual_third.manager.VisualAlarmManager;
import com.tenqube.visual_third.manager.migration.Migration;
import com.tenqube.visual_third.manager.migration.MigrationCallback;
import com.tenqube.visual_third.manager.sms.SmsLoader;
import com.tenqube.visual_third.model.ui.TransactionPopupInfo;
import com.tenqube.visual_third.parser.SyncHolder;
import com.tenqube.visual_third.parser.executor.BulkCallback;
import com.tenqube.visual_third.parser.executor.BulkExecutor;
import com.tenqube.visual_third.repository.RepositoryHolder;
import com.tenqube.visual_third.repository.ResourceRepository;
import com.tenqube.visual_third.repository.VisualRepository;
import com.tenqube.visual_third.util.AppExecutors;
import com.tenqube.visual_third.util.Utils;
import com.tenqube.visual_third.util.VisualWebView;
import com.tenqube.visual_third.util.permission.AfterPermissionGranted;
import com.tenqube.visual_third.util.permission.PermissionUtil;
import com.tenqube.visual_third.web.ActionImpl;
import com.tenqube.visual_third.web.Completion;
import com.tenqube.visual_third.web.ErrorImpl;
import com.tenqube.visual_third.web.LogImpl;
import com.tenqube.visual_third.web.RepoImpl;
import com.tenqube.visual_third.web.SystemImpl;
import com.tenqube.visual_third.web.UiImpl;
import com.tenqube.visual_third.web.VisualInterface;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.List;

import tenqube.parser.core.ParserService;

import static com.tenqube.visual_third.util.Utils.fixInputMethod;
import static com.tenqube.visual_third.util.Utils.isNotiEnabled;
import static com.tenqube.visual_third.util.Utils.isSDKDisabled;
import static com.tenqube.visual_third.util.WebViewHelper.getBaseUrl;
import static com.tenqube.visual_third.util.WebViewHelper.getProgressUrl;
import static com.tenqube.visual_third.util.permission.PermissionUtil.PERMISSIONS_SMS;
import static com.tenqube.visual_third.util.permission.PermissionUtil.PERMISSIONS_STORAGE;


public class VisualWebActivity extends AppCompatActivity implements
        TimePickerFragment.Callback,
        DatePickerFragment.Callback,
        PermissionUtil.PermissionCallbacks,
        ActivityCallback,
        BulkCallback {

    public static final String ARG_PATH = "ARG_PATH";
    public static final String ACTION_BROADCAST_TRANSACTION = "ACTION_BROADCAST_TRANSACTION";

    public static final String TAG = VisualServiceImpl.class.getSimpleName();

    public static final int NOTIFICATION_REQUEST_CODE = 10;
    public static final int VISUAL_REQUEST_CODE = 512;
    public static final int PERMISSION_REQUEST_CODE = 0;


    private BulkExecutor bulkExecutor;

    @Nullable
    private VisualRepository repository;

    @Nullable
    private ResourceRepository resourceRepository;
    private VisualAlarmManager alarmManager;
    private Migration migration;
    private boolean isPageLoaded;

    private String mFailingUrl;

    private String baseUrl = "";
    private String webUrl = "";

    private String path;

    @Nullable
    private VisualWebView webview;

    private VisualInterface.Repo repoInterface;
    private VisualInterface.UI uiInterface;
    private VisualInterface.System systemInterface;
    private VisualInterface.Error errorInterface;
    private VisualInterface.Log logInterface;
    private VisualInterface.Action actionInterface;

    private SwipeRefreshLayout swipeRefreshLayout;

    @Nullable
    private LinearLayout errorView;
    private boolean isError;

    private boolean isLoading;

    private PrefManager prefManager;

    private long startTime;

    // 광고
    private FrameLayout webContainer;
    private CardView adContainer;
    private boolean isActivityFirst;
    private BottomDialog bottomDialog;

    @SuppressLint({"JavascriptInterface", "AddJavascriptInterface", "SetJavaScriptEnabled"})
    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        try {
//
            isActivityFirst = true;
            Utils.LOG("onCreate", "start VisualActivity");
            AnalysisServiceImpl.shouldRefresh = true;
            startTime = System.currentTimeMillis();
            checkSDK();
            setContentView(R.layout.activity_visual_web);
            parseArg();
            createInstances();
            setUpWebContainer();
            setupSwipeRefreshView();
            setupErrorView();
            setupWebView();
            start();

        } catch (Exception e) {
            Utils.LOG("onCreate", "start VisualActivity" + e.toString());
            finish();
        }
    }

    private void checkSDK() throws Exception {
        Utils.LOG("checkSDK", "start");
        if(isSDKDisabled(getApplicationContext())) {
            throw new Exception("SDK disabled");
        }
    }

    private void parseArg() {
        Utils.LOG("parseArg", "start");

        path = getIntent().getStringExtra(ARG_PATH);
        if(path == null) {
            path = "";
        }
    }

    private void setUpWebContainer() {
        Utils.LOG("setUpWebContainer", "start");

        webContainer = findViewById(R.id.web_container);
    }

    private void createInstances() {

        Utils.LOG("createInstances", "start");

        prefManager = PrefManager.getInstance(getApplicationContext());

        repository = RepositoryHolder.getInstance(getApplicationContext()).getVisualRepository();

        resourceRepository = RepositoryHolder.getInstance(getApplicationContext()).getResourceRepository();

        alarmManager = VisualAlarmManager.getInstance(getApplicationContext());

        migration = new Migration(prefManager,
                repository,
                new SmsLoader(getApplicationContext()),
                ParserService.getInstance(getApplicationContext()),
                new AppExecutors());

    }

    private void setupSwipeRefreshView() {

        Utils.LOG("setupSwipeRefreshView", "start");

        swipeRefreshLayout = findViewById(R.id.swipe_refresh_layout);
        swipeRefreshLayout.setEnabled(false);
        swipeRefreshLayout.setColorSchemeResources(
                R.color.colorPopupRed,
                R.color.colorPopupRed,
                R.color.colorPopupRed);

        swipeRefreshLayout.setOnRefreshListener(new SwipeRefreshLayout.OnRefreshListener() {
            @Override
            public void onRefresh() {
                swipeRefreshLayout.setRefreshing(false);
                AnalysisServiceImpl.shouldRefresh = true;
                if(webview != null)
                    webview.reload();

            }
        });
    }

    private void setupErrorView() {

        Utils.LOG("setupErrorView", "start");

        //error view 처리
        errorView = findViewById(R.id.error_container);
        errorView.setVisibility(View.GONE);

        findViewById(R.id.retry).setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                retry();
            }
        });

    }

    private void setupWebView() {
        Utils.LOG("setupWebView", "start");

        webview = findViewById(R.id.visual2_web_view);
        webview.setLayerType(View.LAYER_TYPE_HARDWARE, null);
        webview.playSoundEffect(SoundEffectConstants.CLICK);

        // javascriptInterface
        errorInterface = new ErrorImpl(this, webview);
        logInterface = new LogImpl(this, webview, errorInterface);

        bottomDialog = new BottomDialog(this);
        repoInterface = new RepoImpl(this, webview, errorInterface, repository, new AnalysisServiceImpl(getApplicationContext(), ""), alarmManager);
        uiInterface = new UiImpl(this, webview, errorInterface, bottomDialog, AdManager.getInstance(getApplicationContext()));
        systemInterface = new SystemImpl(this, webview, errorInterface);
        actionInterface = new ActionImpl(this, webview);

        webview.setBackgroundColor(Color.TRANSPARENT);

        webview.addJavascriptInterface(errorInterface, "visualError");
        webview.addJavascriptInterface(repoInterface, "visualRepo");
        webview.addJavascriptInterface(uiInterface, "visualUI");
        webview.addJavascriptInterface(systemInterface, "visualSystem");
        webview.addJavascriptInterface(logInterface, "visualLog");
        webview.addJavascriptInterface(actionInterface, "visualAction");

        // settings
        webview.getSettings().setJavaScriptEnabled(true);
        webview.getSettings().setDomStorageEnabled(true);
        webview.getSettings().setDatabaseEnabled(true);
        webview.getSettings().setLoadWithOverviewMode(true);
        webview.getSettings().setUseWideViewPort(true);
        webview.getSettings().setTextZoom(100);

        if(repository != null) {
            if(repository.isDev()) {
                webview.getSettings().setCacheMode(WebSettings.LOAD_NO_CACHE);
                WebView.setWebContentsDebuggingEnabled(true);
            } else {
                WebView.setWebContentsDebuggingEnabled(false);

            }
        }

        webview.setWebChromeClient(new WebChromeClient() {
            @Override
            public void onProgressChanged(WebView view, int newProgress) {

                Utils.LOG("onProgressChanged", "newProgress: " + newProgress + "");
                super.onProgressChanged(view, newProgress);
            }

            @Override
            public boolean onConsoleMessage(ConsoleMessage consoleMessage) {
                Utils.LOG("onConsoleMessage", "consoleMessage: " + consoleMessage.message() + "");
                return super.onConsoleMessage(consoleMessage);
            }
        });


        webview.setWebViewClient(new WebViewClient() {

            public void onReceivedError(WebView view, int errorCode, String description, String failingUrl) {
                Utils.LOG("onReceivedError", "수행시간: " + ((System.currentTimeMillis() - startTime) / 1000) + "초");
                isError = true;
                mFailingUrl = failingUrl;
                loadErrorUrl();
            }

            @Override
            public void onPageFinished(WebView view, String url) {
                if (!isError && errorView != null) errorView.setVisibility(View.GONE);

                isError = false;
                Utils.LOG("onPageFinished", "수행시간: " + ((System.currentTimeMillis() - startTime) / 1000) + "초, url:" + url);
                super.onPageFinished(view, url);
            }
        });
    }

    private void start() {
        Utils.LOG("start", "start");
        startVisual();
    }

    public void addAd(final View view) {
        Utils.LOG("addAd", "start");

        try {
            webContainer.addView(createCardView(view));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private CardView createCardView(View view) {

        Utils.LOG("createCardView", "start");

        adContainer = new CardView(this);

        // Set the CardView layoutParams
        FrameLayout.LayoutParams params = new FrameLayout.LayoutParams(
                FrameLayout.LayoutParams.WRAP_CONTENT,
                FrameLayout.LayoutParams.WRAP_CONTENT
        );

        params.setMargins(Utils.dpToPx(10), Utils.dpToPx(16), Utils.dpToPx(10), Utils.dpToPx(16));
        params.gravity = Gravity.BOTTOM;
        adContainer.setLayoutParams(params);

        // Set CardView corner radius
        adContainer.setRadius(Utils.dpToPx(13));

        // Set a background color for CardView
        adContainer.setCardBackgroundColor(Color.parseColor("#00000000"));

        adContainer.addView(view);

        return adContainer;

    }

    public void hideAd() {
        Utils.LOG("hideAd", "start");

        try {
            runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    if(adContainer != null) {
                        webContainer.removeView(adContainer);
                    }
                }
            });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    public TransactionPopupInfo getPopUpInfo() {
        return null;
    }

    @Override
    public void setStatusBarColor(String color) {

        Utils.LOG("setStatusBarColor", "start");

        try {
            getWindow().setStatusBarColor(Color.parseColor(color));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


    private BroadcastReceiver transactionReceiver = new BroadcastReceiver() {

        @Override
        public void onReceive(Context context, final Intent intent) {

            try {

                Utils.LOG("transactionReceiver", "start");

                final int tranId = intent.getIntExtra("tranId", 0);
                webview.post(new Runnable() {
                    @Override
                    public void run() {
                        if(tranId != 0)
                            webview.loadUrl("javascript:window.onTransactionReceived" + "(" + tranId + ");");
                    }
                });

            } catch (Exception e) {
            }
        }
    };

    private void startVisual() {

        Utils.LOG("startVisual", "start");

        if(repository != null) {
            if(repository.shouldBulk()) { // 벌크 파싱 진행
                migration.saveMigratedVersion(Migration.VERSION);
                // default parsing
                loadProgressUrl(Constants.ProgressType.DEFAULT); //기본 로딩 빠르게 실행하기
                // 벌크 파싱 시작
                startBulkParsing(true);
            } else {
                migration();
            }
        }
    }

    private void migration() {

        migration.card(new MigrationCallback() {
            @Override
            public void onStart() {
                // 벌크 url
                Utils.LOG("migration", "start");

                if(webview != null) {
                    loadProgressUrl(Constants.ProgressType.MIGRATION);
                }

            }

            @Override
            public void onProgress(final int current, final int total) {
                Utils.LOG("migration", "current" + current + "/ total" + total);

                if(webview != null) {
                    webview.loadUrl("javascript:window.onProgress" + "(" + current + "," + total + ");");
                }
            }

            @Override
            public void onCompleted() {

                Utils.LOG("migration", "onCompleted");

                loadUrl(path);

                // 알람 설정
                alarmManager.setAlarms();

                // 리소스 동기화
                if(resourceRepository != null)
                    resourceRepository.sync();
            }
        });
    }


    /**
     * 1. 퍼미션 체크
     * 2. 프로그레스 다이얼로그 show
     * 3. 지난 6개월 문자 파싱 시작
     */
    private void startBulkParsing(final boolean isInit) {
        Utils.LOG("startBulkParsing", "isInit:" + isInit);

        isLoading = true;
        bulkExecutor = RepositoryHolder.getInstance(getApplicationContext()).getBulkExecutor();
        bulkExecutor.setCallback(this);
        bulkExecutor.start(isInit);

    }

    public void changeStatusBarColor(String color) {

        Utils.LOG("changeStatusBarColor", "color:" + color);
        try {
            getWindow().setStatusBarColor(Color.parseColor(color));
        } catch (Exception e) {
        }
    }

    public void onPagedLoaded() {
        runOnUiThread(new Runnable() {
            @Override
            public void run() {
                isPageLoaded = true;
            }
        });
    }

    private void loadUrl(String path) {

        Utils.LOG("loadUrl", "path:" + path);

        if(webview != null && !isLoading) {
            isPageLoaded = false;
            if(path == null) path = "";
            webUrl = prefManager.loadStringValue(PrefManager.WEB_URL, "");
            if(TextUtils.isEmpty(webUrl)){
                Utils.LOG("finish", "TextUtils.isEmpty(webUrl)");
                finish();
                return;
            }
            baseUrl = getBaseUrl(webUrl);

            Utils.LOG("loadUrl", "수행시간: " + ((System.currentTimeMillis() - startTime) / 1000) + "초, url:" + baseUrl);

            webview.loadUrl(baseUrl + path);
        }
    }

    @Override
    public void onSyncError() {
        if(webview != null) {
            webview.postDelayed(new Runnable() {
                @Override
                public void run() {
                    //onSyncError
                    if(webview != null)
                        webview.loadUrl("javascript:window.onSyncError();");
                }
            }, 200);

        }
    }

    @Override
    public void onBulkFinish(final boolean isSuccess,final boolean shouldReload) {

        isLoading = false;
        Utils.LOG("BulkHandler", "onBulkFinish");
        AnalysisServiceImpl.shouldRefresh = true;
        if(repository != null) {
            repository.saveBulk();
        }

        if(webview != null && shouldReload) {
            startVisual();
        }
        if(completion != null) {
            completion.onDataLoaded(isSuccess);
            completion = null;
        }
    }

    @Override
    public void loadProgressUrl(Constants.ProgressType progressType) {
        Utils.LOG("loadProgressUrl", "progressType:" + progressType);
        if(webview != null) {
            isPageLoaded = false;
            baseUrl = getProgressUrl(prefManager, progressType);
            webview.loadUrl(baseUrl);
        }
    }

    @Override
    public void onProgress(int now, int total) {
        if(webview != null) {
            webview.loadUrl("javascript:window.onProgress" + "(" + now + "," + total + ");");
        }
    }

    private void loadErrorUrl() {

        Utils.LOG("loadErrorUrl", "start");

        isPageLoaded = false;
        if(swipeRefreshLayout != null) swipeRefreshLayout.setEnabled(false);
        if(errorView != null) errorView.setVisibility(View.VISIBLE);
    }

    @Override
    protected void onActivityResult(int requestCode, int resultCode, final Intent data) {
        super.onActivityResult(requestCode, resultCode, data);

        if (requestCode == NOTIFICATION_REQUEST_CODE) {
            boolean isEnabled = isNotiEnabled(getApplicationContext());

            JSONObject jsonObject = new JSONObject();
            try {
                jsonObject.put("isNotiEnabled", isEnabled);
                String json = jsonObject.toString();
                if (systemInterface != null) systemInterface.onNotiCallback(json);

            } catch (JSONException e) {

            }
        } else if (requestCode == PERMISSION_REQUEST_CODE) {
            startVisual();
        }
    }

    @Override
    public void onBackPressed() {
        Utils.LOG("onBackPressed", "start");

        try {
            if(isPageLoaded && uiInterface != null) {
                uiInterface.onFinish();
            } else {
                super.onBackPressed();
            }

        } catch (Exception e) {
            super.onBackPressed();
        }
    }

    @Override
    public void onResume() {
        super.onResume();

        if(webview != null)
            webview.onResume();

        if(!isActivityFirst && webview != null)
            webview.reload();

        if(isActivityFirst) { // oncreate 체크 함수
            isActivityFirst = false;
        }
        LocalBroadcastManager.getInstance(this).registerReceiver(transactionReceiver, new IntentFilter(ACTION_BROADCAST_TRANSACTION));

        if(repository != null)
            repository.setVisualActive(true);
        overridePendingTransition(0, 0);
    }

    private void showPopup() {

        Utils.LOG("showPopup", "start");

        AlertDialog.Builder builder = new AlertDialog.Builder(this);
        builder.setTitle(getString(R.string.dialog_title))
                .setMessage(getString(R.string.dialog_message))
                .setCancelable(false)
                .setPositiveButton(getString(R.string.dialog_setting), new DialogInterface.OnClickListener() {
                    @Override
                    public void onClick(DialogInterface dialog, int which) {
                        goToAppSetting();
                        dialog.cancel();

                        Utils.LOG("showPopup setting", "finish");

                        finish();
                    }
                })
                .setNegativeButton(getString(R.string.dialog_cancel), new DialogInterface.OnClickListener() {
                    @Override
                    public void onClick(DialogInterface dialog, int which) {

                        Utils.LOG("showPopup cancel", "finish");

                        finish();
                    }
                });

        AlertDialog dialog = builder.create();
        dialog.show();
    }

    public void goToAppSetting() {

        Utils.LOG("goToAppSetting", "start");

        Uri packageUri = Uri.parse("package:" + getApplicationContext().getPackageName());
        Intent intent = new Intent(Settings.ACTION_APPLICATION_DETAILS_SETTINGS, packageUri);
        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK | Intent.FLAG_ACTIVITY_EXCLUDE_FROM_RECENTS);
        startActivity(intent);
    }

    @Override
    public void onPause() {
        super.onPause();

        if(webview != null)
            webview.onPause();

        Utils.LOG("onPause", "start");

        LocalBroadcastManager.getInstance(this).unregisterReceiver(transactionReceiver);

        if(repository != null)
            repository.setVisualActive(false);
        overridePendingTransition(0, 0);
    }

    @Override
    public void onCalendar(String date, String callback) {

        Utils.LOG("onCalendar", "date" + date + "callback" + callback);

        uiInterface.onCalendar(date, callback);
    }

    public void onScrollChanged(final int pos) {
    }

    public void setRefreshEnabled(final boolean enabled) {
        swipeRefreshLayout.post(new Runnable() {
            @Override
            public void run() {
                swipeRefreshLayout.setEnabled(enabled);
            }
        });
    }

    public void reload() {
        if(webview != null)  {
            webview.post(new Runnable() {
                @Override
                public void run() {
                    webview.reload();
                }
            });
        }
    }

    public void retry() {
        if(webview != null)  {
            webview.post(new Runnable() {
                @Override
                public void run() {
                    webview.loadUrl(mFailingUrl);
                }
            });
        }
    }


    public static final int STORAGE = 1;

    public Completion<Boolean> completion;

    public void exportExcel(Completion<Boolean> completion) {
        this.completion = completion;
        exportExcel();
    }

    @Override
    public Context getContext() {
        return getApplicationContext();
    }

    @Override
    public AppCompatActivity getActivity() {
        return this;
    }

    @Override
    public void restore(Completion<Boolean> completion) {
        this.completion = completion;
        startBulkParsing(false);
    }

    @AfterPermissionGranted(STORAGE)
    public void exportExcel() {
        if (PermissionUtil.hasPermissions(
                this, PERMISSIONS_STORAGE)) {


        } else {
            PermissionUtil.requestPermissions(
                    this,
                    STORAGE,
                    PERMISSIONS_STORAGE);
        }
    }

    @Override
    public void onPermissionsGranted(int requestCode, List<String> list) {

        if (requestCode == STORAGE) {
            exportExcel();
        }
    }

    @Override
    public void onPermissionsDenied(int requestCode, List<String> list) {
        Toast.makeText(getApplicationContext(), "엑셀 파일 내보내기를 위해 외부 저장소에 대한 권한이 필요합니다. 권한이 없는 경우 엑셀 파일 내보내기 사용이 불가합니다.", Toast.LENGTH_SHORT).show();
    }

    @Override
    public void onRequestPermissionsResult(int requestCode, String[] permissions, int[] grantResults) {
        super.onRequestPermissionsResult(requestCode, permissions, grantResults);

        // Forward results to EasyPermissions
        PermissionUtil.onRequestPermissionsResult(requestCode, permissions, grantResults, this);
    }

    public static void startActivity(Activity activity, String path) throws AuthException {
        if(TextUtils.isEmpty(SecretKeyManager.getInstance(activity).getKey(PrefManager.ACCESS_TOKEN)))
            throw new AuthException("Need to sign up");

        Intent intent = new Intent(activity, VisualWebActivity.class);
        intent.putExtra(ARG_PATH, path);
        intent.addFlags(Intent.FLAG_ACTIVITY_SINGLE_TOP|
                Intent.FLAG_ACTIVITY_CLEAR_TOP |
                Intent.FLAG_ACTIVITY_REORDER_TO_FRONT);
        activity.startActivityForResult(intent, VISUAL_REQUEST_CODE);
    }

    public static void startActivity(Context context, String path) throws AuthException {
        if(TextUtils.isEmpty(SecretKeyManager.getInstance(context).getKey(PrefManager.ACCESS_TOKEN)))
            throw new AuthException("Need to sign up");

        Intent intent = new Intent(context, VisualWebActivity.class);
        intent.putExtra(ARG_PATH, path);
        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK |
                Intent.FLAG_ACTIVITY_CLEAR_TOP |
                Intent.FLAG_ACTIVITY_REORDER_TO_FRONT);
        context.startActivity(intent);
    }

    @Override
    protected void onRestoreInstanceState(Bundle savedInstanceState) {
        Utils.LOG("onRestoreInstanceState", "");
        try {
            if (PermissionUtil.hasPermissions(
                    this, PERMISSIONS_SMS)) {

                Utils.LOG("onRestoreInstanceState", "finish");

                finish();
            } else {
                showPopup();
            }
            super.onRestoreInstanceState(savedInstanceState);
        } catch (Exception e) {

            Utils.LOG("onRestoreInstanceState", "exception" + e.toString());
            finish();
        }

    }

    @Override
    public void onConfigurationChanged(Configuration newConfig) {
        Utils.LOG("onConfigurationChanged", "");
        super.onConfigurationChanged(newConfig);

        Utils.LOG("onConfigurationChanged", "finish");

        finish();

    }

    @Override
    protected void onDestroy() {
        Utils.LOG("onDestroy", "");
        destroyWebView();
        destroy();
        super.onDestroy();
        fixInputMethod(getApplicationContext());
    }

    private void destroyWebView() {
        if(webview != null) {

            webview.removeAllViews();
            webview.getSettings().setJavaScriptEnabled(false);
            webview.removeJavascriptInterface("visualError");
            webview.removeJavascriptInterface("visualRepo");
            webview.removeJavascriptInterface("visualUI");
            webview.removeJavascriptInterface("visualSystem");
            webview.removeJavascriptInterface("visualLog");
            webview.removeJavascriptInterface("visualAction");
        }
    }


    private void destroy() {
        SyncHolder.getInstance().setSyncService(null);

        if(bulkExecutor != null)
            bulkExecutor.setCallback(null);
        //bulkExecutor는 destroy 되면 안되고 벌크파싱 종료시에 자동 destroy 됩니다.

        alarmManager = null;

        repoInterface = null;
        uiInterface = null;
        systemInterface = null;
        errorInterface = null;
        logInterface = null;
        actionInterface = null;
        swipeRefreshLayout = null;
        errorView = null;
        prefManager = null;

        webContainer = null;

        adContainer = null;

        bottomDialog = null;
        webview = null;


    }
}
