package com.tenqube.visual_third.analysis;

import android.text.TextUtils;

import com.tenqube.visual_third.Constants;
import com.tenqube.visual_third.entity.Condition;
import com.tenqube.visual_third.entity.JoinedContent;
import com.tenqube.visual_third.model.analysis.AggregationValue;
import com.tenqube.visual_third.model.analysis.Analysis;
import com.tenqube.visual_third.model.analysis.CAC;
import com.tenqube.visual_third.model.analysis.CategoryAvg;
import com.tenqube.visual_third.model.analysis.Morning;
import com.tenqube.visual_third.model.analysis.Result;
import com.tenqube.visual_third.model.analysis.Transaction;
import com.tenqube.visual_third.util.Utils;

import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;

import static com.tenqube.visual_third.Constants.AVG;
import static com.tenqube.visual_third.Constants.AggregateType.CARD_TYPE;
import static com.tenqube.visual_third.Constants.AggregateType.DATE;
import static com.tenqube.visual_third.Constants.AggregateType.FRANCHISE;
import static com.tenqube.visual_third.Constants.AggregateType.KEYWORD;
import static com.tenqube.visual_third.Constants.AggregateType.LCODE;
import static com.tenqube.visual_third.Constants.AggregateType.MCODE;
import static com.tenqube.visual_third.Constants.AggregateType.MONTH;
import static com.tenqube.visual_third.Constants.CNT;
import static com.tenqube.visual_third.Constants.FROM;
import static com.tenqube.visual_third.Constants.SUM;
import static com.tenqube.visual_third.Constants.TO;
import static com.tenqube.visual_third.util.Utils.LOGD;
import static com.tenqube.visual_third.util.Utils.compareDate;
import static com.tenqube.visual_third.util.Utils.compareDateBefore;
import static com.tenqube.visual_third.util.Utils.compareDatetime;
import static com.tenqube.visual_third.util.Utils.equalsDate;
import static com.tenqube.visual_third.util.Utils.equalsDatetime;
import static com.tenqube.visual_third.util.Utils.getDailyDateRanges;
import static com.tenqube.visual_third.util.Utils.getDateRangeAnalysis;
import static com.tenqube.visual_third.util.Utils.getWeeklyDateRanges;
import static com.tenqube.visual_third.util.Utils.getYMD;
import static com.tenqube.visual_third.util.Utils.gtDate;
import static com.tenqube.visual_third.util.Utils.isEmpty;
import static com.tenqube.visual_third.util.Utils.isEvenDay;
import static com.tenqube.visual_third.util.Utils.isOddDay;
import static com.tenqube.visual_third.util.Utils.ltDate;
import static com.tenqube.visual_third.util.Utils.toCalendar;

/**
 * Created by sagwangjin on 2018. 7. 22..
 */

public class AnalysisHelper {

    static final String KEY_DELIMITER = ";";

    static HashMap<String, AggregationValue> aggregate(ArrayList<Transaction> transactions, int... keys) {
        HashMap<String, AggregationValue> results = new HashMap<>();
        AggregationValue aggregation;

        for(Transaction transaction : transactions) {

            String key = makeKey(transaction, keys);
            aggregation = results.get(key);

            if(aggregation != null) {

                aggregation.setSum(aggregation.getSum() + transaction.getTransaction().getSpentMoney());
                aggregation.setCnt(aggregation.getCnt() + 1);
                aggregation.setAvg();
                aggregation.getTranIds().add(transaction.getTransaction().getId());
            } else {
                ArrayList<Integer> tranIds = new ArrayList<>();
                tranIds.add(transaction.getTransaction().getId());
                aggregation = new AggregationValue(key, transaction.getTransaction().getSpentMoney(), 1, transaction.getTransaction().getSpentMoney(), tranIds);
                aggregation.setTitle(getTitle(transaction, keys.length > 0 ? keys[0] : 0));
            }
            results.put(key, aggregation);
        }
        return results;
    }

    private static String makeKey(Transaction transaction , int... keys) {

        ArrayList<String> aggregateKey = new ArrayList<>();

        for(int key : keys) {
            switch (key) {
                case LCODE:
                    aggregateKey.add(transaction.getCategory().getLargeCode());
                    break;

                case MONTH:
                    aggregateKey.add(getYMD(transaction.getTransaction().getSpentDate()));
                    break;

                case MCODE:
                    aggregateKey.add(transaction.getCategory().getLargeMediumCode());
                    break;

                case CARD_TYPE:
                    aggregateKey.add("" + transaction.getCard().getChangeType());
                    break;

                case FRANCHISE:
                    aggregateKey.add("" + transaction.getTransaction().getFranchise());
                    break;

                case KEYWORD:
                    aggregateKey.add("" + transaction.getTransaction().getKeyword());
                    break;

                case DATE:
                    aggregateKey.add("" + Utils.getYMD(transaction.getTransaction().getSpentDate()));
                    break;


                default:
                    aggregateKey.add("");
                    break;
            }
        }

        return TextUtils.join(KEY_DELIMITER, aggregateKey);
    }

    private static String getTitle(Transaction transaction , int key) {
        String title;
        switch (key) {

            case KEYWORD:
                title = transaction.getTransaction().getKeyword();
                break;

            case LCODE:
                title = transaction.getCategory().getLarge();
                break;

            case MCODE:
                title = transaction.getCategory().getMedium();
                break;

            case CARD_TYPE:
                title = transaction.getCategory().getMedium();
                break;

            case DATE:
                title = Utils.getYMD(transaction.getTransaction().getSpentDate());
                break;

            default:
                title = "";
                break;
        }

        return title;
    }

    static int getMonth(int before) {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.MONTH, -1 * before);
        return calendar.get(Calendar.MONTH) + 1;
    }

    static ArrayList<Transaction> filterTransactions(ArrayList<Transaction> transactions, int before, int startDay) {
        ArrayList<Transaction> filteredTransaction = new ArrayList<>();
        Calendar calendar = Calendar.getInstance();
        String[] dateRange = getDateRangeAnalysis(calendar.get(Calendar.YEAR), calendar.get(Calendar.MONTH) + 1, before, startDay);

        Utils.LOGD("filterTransactions" + before, "before:" + before + "from: " + dateRange[FROM] + "to:" + dateRange[TO]);

        for(Transaction transaction : transactions) {
            if ((equalsDate(dateRange[FROM], transaction.getTransaction().getSpentDate()) ||
                    ltDate(dateRange[FROM], transaction.getTransaction().getSpentDate())) &&
                    gtDate(dateRange[TO], transaction.getTransaction().getSpentDate())) {
                filteredTransaction.add(transaction);

//                Log.i("filterTransactions dateRange[FROM]: ", dateRange[FROM]);
//                Log.i("filterTransactions dateRange[TO]: ", dateRange[TO]);
//
//                Log.i("filterTransactions", transaction.getTransaction().getSpentDate());


            }
        }
        return filteredTransaction;
    }

    static double getSamePeriodSum(ArrayList<Transaction> transactions, int before) {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.MONTH, -1 * before);
        double sum = 0;
        for(Transaction transaction : transactions) {

            if(compareDate(transaction.getTransaction().getSpentDate(), getYMD(calendar))) {
                sum += transaction.getTransaction().getSpentMoney();
            }
        }
        return sum;
    }

    static ArrayList<Transaction> filterTransactionsDaily(ArrayList<Transaction> transactions, int before) {

        ArrayList<Transaction> filteredTransactions = new ArrayList<>();

        ArrayList<String> dateRanges = getDailyDateRanges(before);

        for(Transaction transaction : transactions) {

            if(compareDate(dateRanges.get(FROM), transaction.getTransaction().getSpentDate()) &&
                    compareDate(transaction.getTransaction().getSpentDate(), dateRanges.get(TO))){
                filteredTransactions.add(transaction);
            }
        }

        return filteredTransactions;

    }

    static ArrayList<Transaction> filterTransactionsWeekly(ArrayList<Transaction> transactions, int before) {

        ArrayList<Transaction> filteredTransactions = new ArrayList<>();
        // 지난 주 월 ~ 오늘 기간 필터추가
        ArrayList<String> dateRanges = getWeeklyDateRanges(before);

        for(Transaction transaction : transactions) {

            if(compareDate(dateRanges.get(FROM), transaction.getTransaction().getSpentDate()) &&
                    compareDate(transaction.getTransaction().getSpentDate(), dateRanges.get(TO))){
                filteredTransactions.add(transaction);
            }
        }

        return filteredTransactions;
    }

    static ArrayList<Transaction> filterTransactions12Weekly(ArrayList<Transaction> transactions) {

        ArrayList<Transaction> filteredTransactions = new ArrayList<>();
        // 지난 주 월 ~ 오늘 기간 필터추가
        String dateFrom = getWeeklyDateRanges(12).get(FROM);
        String dateTo = getWeeklyDateRanges(1).get(TO);

        LOGD("filterTransactions12Weekly", "dateFrom" + dateFrom + "dateTo" + dateTo);

        for(Transaction transaction : transactions) {

            if(compareDate(dateFrom, transaction.getTransaction().getSpentDate()) &&
                    compareDate(transaction.getTransaction().getSpentDate(), dateTo)){
                filteredTransactions.add(transaction);
            }
        }

        return filteredTransactions;
    }

    static ArrayList<Transaction> filterTransactionsMcode(ArrayList<Transaction> transactions, int... mcodes) {

        ArrayList<Transaction> filteredTransactions = new ArrayList<>();
        // 지난 주 월 ~ 오늘 기간 필터추가

        for(Transaction transaction : transactions) {
            boolean isMatched = true;
            for(int mcode : mcodes) {
                if(!String.valueOf(mcode).equals(transaction.getCategory().getLargeMediumCode())) {

                    isMatched = false;
                    break;
                }
            }
            if(isMatched) filteredTransactions.add(transaction);

        }
        return filteredTransactions;
    }

    static ArrayList<Transaction> filterTransactionsByDate(ArrayList<Transaction> transactions, String date) {

        ArrayList<Transaction> filteredTransactions = new ArrayList<>();
        // 지난 주 월 ~ 오늘 기간 필터추가

        for(Transaction transaction : transactions) {

            if(date.equals(Utils.getYMD(transaction.getTransaction().getSpentDate()))){
                filteredTransactions.add(transaction);
            }
        }

        return filteredTransactions;
    }

    static double getSumByLcode(ArrayList<Transaction> transactions, int lCode, String date) {

        double sum = 0;

        for(Transaction transaction : transactions) {

            if(String.valueOf(lCode).equals(transaction.getCategory().getLargeCode()) &&
                    compareDateBefore(Utils.getYMD(transaction.getTransaction().getSpentDate()), date)){
                sum += transaction.getTransaction().getSpentMoney();
            }
        }

        return sum;
    }

    /**
     * 오전 7 <= 시간 < 10
     * @param transactions 내역목록
     * @return 오전 시간 금액 합
     */
    static Morning getMorningSum(ArrayList<Transaction> transactions, int mcode) {

        int fromHour = 7;
        int toHour = 10;
        double sum = 0;
        ArrayList<Integer> ids = new ArrayList<>();

        for(Transaction transaction : transactions) {

            if(String.valueOf(mcode).equals(transaction.getCategory().getLargeMediumCode())) {
                Calendar calendar = toCalendar(transaction.getTransaction().getSpentDate());
                int hour = calendar.get(Calendar.HOUR_OF_DAY);
                if(fromHour <= hour && hour < toHour) {
                    sum += transaction.getTransaction().getSpentMoney();
                    ids.add(transaction.getTransaction().getId());
                }
            }
        }
        return new Morning(sum, ids);
    }

    static Transaction getMaxKeyword(ArrayList<Transaction> transactions) {
        Transaction maxTransaction = null;
        for(Transaction transaction : transactions) {

            if(maxTransaction == null || maxTransaction.getTransaction().getSpentMoney() < transaction.getTransaction().getSpentMoney()) {
                maxTransaction = transaction;
            } else if (maxTransaction.getTransaction().getSpentMoney() == transaction.getTransaction().getSpentMoney()) {
                if(compareDatetime(maxTransaction.getTransaction().getSpentDate(), transaction.getTransaction().getSpentDate())) {
                    maxTransaction = transaction;
                } else if (equalsDatetime(maxTransaction.getTransaction().getSpentDate(), transaction.getTransaction().getSpentDate())) {
                    if(maxTransaction.getTransaction().getId() < transaction.getTransaction().getId()) {
                        maxTransaction = transaction;
                    }
                }
            }
        }
        return maxTransaction;
    }

    static Transaction getMaxSpentDate(ArrayList<Transaction> transactions) {
        Transaction maxTransaction = null;
        for(Transaction transaction : transactions) {
            if(maxTransaction == null || compareDatetime(maxTransaction.getTransaction().getSpentDate(), transaction.getTransaction().getSpentDate())) {
                maxTransaction = transaction;
            } else if (compareDatetime(maxTransaction.getTransaction().getSpentDate(), transaction.getTransaction().getSpentDate())) {
                if(maxTransaction.getTransaction().getId() < transaction.getTransaction().getId()) {
                    maxTransaction = transaction;
                }
            }
        }

        return maxTransaction;
    }

    static boolean isSatisfied(Condition condition) {
        try {
            if(condition == null) return false;
            String[] values = getValues(condition.getFuncKeys().split(","));
            if(values == null || values.length == 0 || TextUtils.isEmpty(values[0])) return false;


            switch (condition.getFuncType()) {
                case "=":
                    return Double.parseDouble(condition.getStandard()) == Double.parseDouble(values[0]);
                case "<":
                    return Double.parseDouble(condition.getStandard()) < Double.parseDouble(values[0]);
                case ">":
                    return Double.parseDouble(condition.getStandard()) > Double.parseDouble(values[0]);
                case ">=":
                    return Double.parseDouble(condition.getStandard()) >= Double.parseDouble(values[0]);
                case "<=":
                    return Double.parseDouble(condition.getStandard()) <= Double.parseDouble(values[0]);
                case "!=":
                    return Double.parseDouble(condition.getStandard()) != Double.parseDouble(values[0]);
                case "function*<":
                    return Double.parseDouble(values[0]) * Double.parseDouble(condition.getStandard()) < Double.parseDouble(values[1]);
                case "function*>":
                    return Double.parseDouble(values[0]) * Double.parseDouble(condition.getStandard()) > Double.parseDouble(values[1]);
                case "date=":
                    return equalsDate(condition.getStandard(), values[0]);
                case "date<":
                    return ltDate(condition.getStandard(), values[0]);
                case "date>":
                    return gtDate(condition.getStandard(), values[0]);
                case "date<=":
                    return compareDate(condition.getStandard(), values[0]);
                case "date>=":
                    return compareDate(values[0], condition.getStandard());
                case "odd":
                    return isOddDay();
                case "even":
                    return isEvenDay();
                case "True":
                    return true;
                case "isEmpty":
                    if("0".equals(condition.getStandard())) {
                        return !TextUtils.isEmpty(values[0]);
                    } else {
                        return TextUtils.isEmpty(values[0]);
                    }
                default:
                    return false;

            }
        } catch (Exception e) {

        }
        return false;
    }

    private static String[] getValues(String[] keys)  {

        String[] funcResults = new String[keys.length];

        int i = 0;
        for(String key : keys) {
            String value = getFuncValue(Integer.parseInt(key));
            funcResults[i] = value;
            i++;
        }
        return funcResults;
    }

    static String getContent(String content, String funcKey) throws Exception {

        if(TextUtils.isEmpty(funcKey) || "none".equals(funcKey))
            return content;

        String[] funcKeys = funcKey.split(",");
        String[] values = new String[funcKey.length()];

        for (int i = 0 ; i < funcKeys.length ; i++) {
            // 데이터 포맷에 맞게 변경 해야함
            Result result = FunctionMap.values.get(Integer.parseInt(funcKeys[i].replace(" ", "")));
            if(result == null) {
                throw new Exception();
            }
            values[i] = result.getDisplayValue();
        }

        return String.format(content, (Object[]) values);

    }

    static String getValue(String image) {

        try {
            int key = Integer.parseInt(image);
            String imagePath = getFuncValue(key);
            if(!TextUtils.isEmpty(imagePath)) {
                image = imagePath;
            }
        } catch (NumberFormatException e) {
        }
        return image;
    }

    static ArrayList<Integer> getTranIds(JoinedContent content) {

        ArrayList<Integer> tranIds = new ArrayList<>();
        try {
            String[] rawKeys = content.getContent().getRawKeys().split(",");
            for(String rawKey : rawKeys) {
                ArrayList<Integer> temIds = FunctionMap.tranIds.get(Integer.parseInt(rawKey));
                if(!isEmpty(temIds)){
                    tranIds.addAll(temIds);
                }
            }

            return tranIds;
        } catch (Exception e) {

        }
        return tranIds;
    }

    private static final DecimalFormat decimalFormat = new DecimalFormat("###,###");
    private static final DecimalFormat decimalFormatWithDot = new DecimalFormat("###,###.#");

    public static String getLv0Currency(double amount) {
        if (amount < 1000) {
           return decimalFormat.format(amount) +"원";
        } else {
            return decimalFormatWithDot.format(amount/10000) +"만원";
        }
    }

    public static String getNumber(double amount) {
        return decimalFormat.format(amount) +"원";
    }

    public static String getPercent(double value) {
        if (value == 0) {
            return "0.0 %";
        } else {
            return decimalFormatWithDot.format(value) + "%";
        }
    }

    public static String getCnt(double value, String unit) {
        return decimalFormatWithDot.format(value) + unit ;
    }

    public static String getNumberFirst(double value) {
        return decimalFormatWithDot.format(value);
    }

    static String[] getAlcoholType(int cnt){
        if(cnt == 0){
            return new String[]{"비음주", "lv0_alcohol_type_no_alcohol"};
        } else if(cnt > 0 && cnt <= 2) {
            return new String[]{"건강 중시형", "lv0_alcohol_type_healthy"};
        } else if(cnt > 2 && cnt <= 5) {
            return new String[]{"스트레스 해소", "lv0_alcohol_type_stress_down"};
        } else if(cnt > 5 && cnt <= 8) {
            return new String[]{"애주가형", "lv0_alcohol_type_lover"};
        } else if(cnt > 8) {
            return new String[]{"밥보다 술", "lv0_alcohol_type_prefer_al"};
        } else {
            return new String[]{"비음주", "lv0_alcohol_type_no_alcohol"};
        }
    }

    static String[] getCoffeeCountry(double weekAvg) {
        if (23 < weekAvg) {
            return new String[]{"노르웨이", "25잔/1주일"};
        } else if (20 < weekAvg && weekAvg <= 23) {
            return new String[]{"스위스", "22잔/1주일"};
        } else if (17 < weekAvg && weekAvg <= 20) {
            return new String[]{"캐나다", "19잔/1주일"};
        } else if (14 < weekAvg && weekAvg <= 17) {
            return new String[]{"브라질", "16잔/1주일"};
        } else if (11.8 < weekAvg && weekAvg <= 14) {
            return new String[]{"미국", "13잔/1주일"};
        } else if (8.5 < weekAvg && weekAvg <= 11.8) {
            return new String[]{"일본", "11잔/1주일"};
        } else if (5.5 < weekAvg && weekAvg <= 8.5) {
            return new String[]{"한국", "6잔/1주일"};
        } else if (3 < weekAvg && weekAvg <= 5.5) {
            return new String[]{"러시아", "5잔/1주일"};
        } else if (2.2 < weekAvg && weekAvg <= 3) {
            return new String[]{"태국", "3잔/1주일"};
        } else if (0.97 < weekAvg && weekAvg <= 2.2) {
            return new String[]{"이집트", "2잔/1주일"};
        } else {
            return new String[]{"인도", "1잔 이하/1주일"};
        }
    }


    static double getAggregate(HashMap<String, AggregationValue> map, String key, int agType) {
        double result = 0;

        try {
            if(map != null && map.get(key) != null) {
                switch (agType) {
                    case SUM:
                        return map.get(key).getSum();
                    case CNT:
                        return map.get(key).getCnt();
                    case AVG:
                        return map.get(key).getAvg();
                }
            }
        } catch (Exception e) {

        }
        return result;

    }

    static ArrayList<Integer> getAggregatedTranIds(HashMap<String, AggregationValue> map, String key) {
        try {
            if(map != null && map.get(key) != null) {
                return map.get(key).getTranIds();
            }
        } catch (Exception e) {

        }
        return new ArrayList<>();
    }

    static double getPercentValue(double first, double divider) {
        if(divider == 0) return 0;

        return ((first - divider) * 100) / Math.abs(divider);
    }

    static void descCate(ArrayList<CategoryAvg> values) {
        Collections.sort(values, new Comparator<CategoryAvg>() {
            public int compare(CategoryAvg f1, CategoryAvg f2) {

                return f1.getSum() > f2.getSum() ? -1 : 1;
            }
        });
    }

    static void desc(ArrayList<AggregationValue> values) {
        Collections.sort(values, new Comparator<AggregationValue>() {
            public int compare(AggregationValue f1, AggregationValue f2) {

                return f1.getSum() > f2.getSum() ? -1 : 1;
            }
        });
    }

    static void descAnalysis(ArrayList<Analysis> values) {
        Collections.sort(values, new Comparator<Analysis>() {
            public int compare(Analysis f1, Analysis f2) {

                return f1.getCategoryPriority() > f2.getCategoryPriority() ? -1 : 1;
            }
        });
    }

    static void descCAC(ArrayList<CAC> values) {
        Collections.sort(values, new Comparator<CAC>() {
            public int compare(CAC f1, CAC f2) {

                return f1.getSum() > f2.getSum() ? -1 : 1;
            }
        });
    }

    public static void asc(ArrayList<CAC> values) {
        Collections.sort(values, new Comparator<CAC>() {
            public int compare(CAC f1, CAC f2) {

                return f1.getSum() > f2.getSum() ? 1 : -1;
            }
        });
    }

    static String getLargeImages(String lcode, boolean isRandom) {
        try {
            String[] images = getLargeImages(Integer.parseInt(lcode));

            List<String> items = Arrays.asList(images);

            if(isRandom) {
                Collections.shuffle(items);
            }

            return items.get(0);
        } catch (Exception e) {

        }
        return "";
    }

    private static String[] getLargeImages(int lcode) {
        switch (lcode) {
            case Constants.LCODE.FOOD:
                return new String[]{
                        "lv0_food_else_1",
                        "lv0_mid_food_chinese",
                        "lv0_mid_food_dining_place",
                        "lv0_mid_food_fam_re",
                        "lv0_mid_food_fastfood_1",
                        "lv0_mid_food_gogi",
                        "lv0_mid_food_japanese",
                        "lv0_mid_food_korean",
                        "lv0_mid_food_late_night",
                        "lv0_mid_food_western",
                        "lv0_mid_chicken"

                };
            case Constants.LCODE.CAFE:
                return new String[]{
                        "lv0_cafe_else_1",
                        "lv0_mid_cafe_bakery",
                        "lv0_mid_cafe_coffee",
                        "lv0_mid_cafe_dessert",
                        "lv0_mid_cafe_donut"

                };

            case Constants.LCODE.ALCOHOL:
                return new String[]{
                        "lv0_alcohol_else",
                        "lv0_alcohol_soolzip_yesterday",

                        "lv0_mid_alcohol_soolzip",
                        "lv0_mid_alcohol_entertainment"

                };
            case Constants.LCODE.MART:
                return new String[]{
                        "lv0_mart_else",
                        "lv0_mid_mart_grocery",
                        "lv0_mid_mart_conveni",
                        "lv0_mid_mart_dailyitems",
                        "lv0_mid_mart_events_flower",
                        "lv0_mid_mart_housework",
                        "lv0_mid_mart_kids_1",
                        "lv0_mid_mart_pets"

                };

            case Constants.LCODE.ONLINE:
                return new String[]{
                        "lv0_online_else",
                        "lv0_mid_online_homeshop"

                };
            case Constants.LCODE.SHOPPING:
                return new String[] {
                        "lv0_shopping_else",
                        "lv0_mid_shopping_sportbrand"

                };
            case Constants.LCODE.FINANCE:
                return new String[] {
                        "lv0_finance_else",
                        "lv0_mid_finance_insurance",
                        "lv0_mid_finance_tax"


                };
            case Constants.LCODE.HEALTHCARE:
                return new String[] {
                        "lv0_healthcare_else",
                        "lv0_mid_healthcare_clinic",
                        "lv0_mid_healthcare_dentist",
                        "lv0_mid_healthcare_healthproduct",
                        "lv0_mid_healthcare_pharmacy"

                };

            case Constants.LCODE.BEAUTY:

                return new String[] {
                        "lv0_beauty_else_1",
                        "lv0_mid_beauty_nail",
                        "lv0_mid_beauty_hairshop",
                        "lv0_mid_beauty_makeup",
                        "lv0_mid_beauty_skin"

                };
            case Constants.LCODE.LIVINGS:
                return new String[] {
                        "lv0_livings_else",
                        "lv0_mid_livings_telecom"

                };
            case Constants.LCODE.EDUCATION:
                return new String[] {
                        "lv0_education_else",
                        "lv0_mid_education_academy",
                        "lv0_mid_education_school"


                };

            case Constants.LCODE.CULTURE:
                return new String[] {
                        "lv0_culture_else",
                        "lv0_mid_culture_movies",
                        "lv0_mid_culture_books",
                        "lv0_mid_culture_hobby"


                };
            case Constants.LCODE.TRANSPORT:

                return new String[] {
                        "lv0_transportation_else",
                        "lv0_mid_transportation_gas",
                        "lv0_mid_transportation_taxi",
                        "lv0_mid_transportation_car",
                        "lv0_mid_transportation_masstransit"

                };
            case Constants.LCODE.LEPORTS:
                return new String[] {
                        "lv0_leports_else"};
            case Constants.LCODE.TRAVEL:
                return new String[] {
                        "lv0_travel_else",
                        "lv0_mid_travel_sightseeing",
                        "lv0_mid_travel_hotel"
                };

            case Constants.LCODE.FAMILY_EVENT:
                return new String[] {
                        "lv0_family_event_else"};
            default:
                return new String[] {"lv0_uncate_else"};
        }
    }

    static String getMidImg(String mcode) {

        switch (mcode) {

            case "6613":
                return "lv0_mid_travel_hotel";

            case "6614":
                return "lv0_mid_travel_sightseeing";

            case "6212":
                return "lv0_mid_transportation_gas";

            case "6216":
                return "lv0_mid_transportation_taxi";

            case "6213":
                return "lv0_mid_transportation_car";

            case "6214":
                return "lv0_mid_transportation_masstransit";

            case "5611":
                return "lv0_mid_culture_movies";

            case "5612":
                return "lv0_mid_culture_books";

            case "5615":
                return "lv0_mid_culture_hobby";

            case "5411":
                return "lv0_mid_education_academy";

            case "5414":
                return "lv0_mid_education_school";

            case "5211":
                return "lv0_mid_livings_telecom";

            case "4614":
                return "lv0_mid_beauty_nail";

            case "4611":
                return "lv0_mid_beauty_hairshop";

            case "4612":
                return "lv0_mid_beauty_skin";

            case "4610":

                return "lv0_mid_beauty_makeup";

            case "4413":

                return "lv0_mid_healthcare_clinic";

            case "4412":
                return "lv0_mid_healthcare_dentist";

            case "4416":
                return "lv0_mid_healthcare_healthproduct";

            case "4414":
                return "lv0_mid_healthcare_pharmacy";

            case "4212":
                return "lv0_mid_finance_insurance";

            case "4211":
                return "lv0_mid_finance_tax";

            case "3614":
                return "lv0_mid_shopping_sportbrand";

            case "3411":
                return "lv0_mid_online_homeshop";

            case "3216":
                return "lv0_mid_mart_grocery";

            case "3211":
                return "lv0_mid_mart_conveni";

            case "3213":
                return "lv0_mid_mart_dailyitems";

            case "3217":
                return "lv0_mid_mart_events_flower";

            case "3212":
                return "lv0_mid_mart_housework";

            case "3214":
                return "lv0_mid_mart_kids_1";

            case "3215":
                return "lv0_mid_mart_pets";

            case "2613":
                return "lv0_mid_alcohol_soolzip";

            case "2611":
                return "lv0_mid_alcohol_entertainment";

            case "2414":
                return "lv0_mid_cafe_bakery";

            case "2411":
                return "lv0_mid_cafe_coffee";

            case "2415":
                return "lv0_mid_cafe_dessert";

            case "2416":
                return "lv0_mid_cafe_donut";

            case "2215":
                return "lv0_mid_food_chinese";
            case "2217":
                return "lv0_mid_food_dining_place";
            case "2213":
                return "lv0_mid_food_fam_re";
            case "2212":
                return "lv0_mid_food_fastfood_1";
            case "2222":
                return "lv0_mid_food_gogi";
            case "2216":
                return "lv0_mid_food_japanese";

            case "2211":
                return "lv0_mid_food_korean";
            case "2224":
                return "lv0_mid_food_late_night";
            case "2218":
                return "lv0_mid_food_western";

            case "2214":
                return "lv0_mid_chicken";

            default:
                return getLargeImages(mcode.substring(0, 2), false);
        }
    }

    static double getFuncAmount(int key) {
        Result result = FunctionMap.values.get(key);
        if(result == null) return 0;
        return result.getAmount();
    }

    private static String getFuncValue(int key) {
        Result result = FunctionMap.values.get(key);
        if(result == null) return "";
        return result.getValue();
    }

    static ArrayList<Integer> getTranIds(ArrayList<Transaction> transactions) {
        ArrayList<Integer> tranIds = new ArrayList<>();

        for(Transaction transaction : transactions) {
            tranIds.add(transaction.getTransaction().getId());
        }
        return tranIds;
    }


}
