package com.tenqube.visual_third.manager.migration;

import android.text.TextUtils;
import android.util.SparseArray;

import com.tenqube.visual_third.entity.JoinedTransaction;
import com.tenqube.visual_third.manager.PrefManager;
import com.tenqube.visual_third.manager.sms.SmsLoader;
import com.tenqube.visual_third.model.parser.MigrationCard;
import com.tenqube.visual_third.repository.VisualRepository;
import com.tenqube.visual_third.util.AppExecutors;
import com.tenqube.visual_third.util.Utils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import tenqube.parser.Parser;
import tenqube.parser.constants.Constants;
import tenqube.parser.model.RegData;
import tenqube.parser.model.SMS;
import tenqube.parser.util.Validator;

import static com.tenqube.visual_third.manager.PrefManager.MIGRATION_VERSION;

public class Migration {

    public static final int VERSION = 1;

    private final PrefManager prefManager;
    private final VisualRepository visualRepository;
    private final SmsLoader smsLoader;
    private final Parser parser;
    private final AppExecutors appExecutors;

    private int total = 100;
    private int current;

    public Migration(PrefManager prefManager,
                     VisualRepository visualRepository,
                     SmsLoader smsLoader,
                     Parser parser,
                     AppExecutors appExecutors) {
        this.prefManager = prefManager;
        this.visualRepository = visualRepository;
        this.smsLoader = smsLoader;
        this.parser = parser;
        this.appExecutors = appExecutors;
    }

    /**
     * 파싱된 sms 정보를 불러와서 카드정보를 마이그레이션 합니다.
     */
    public void card(final MigrationCallback callback) {

        appExecutors.diskIO().execute(new Runnable() {
            @Override
            public void run() {

                // check already migrate()
                if(!checkAlreadyMigrated()) {
                    Utils.LOG("card", "start");

                    appExecutors.mainThread().execute(new Runnable() {
                        @Override
                        public void run() {
                            callback.onStart();
                        }
                    });
                    // getTransactions

                    onProgress(callback, 1, total);

                    HashMap<String, ArrayList<SMS>> smsMap = getSmsMap();

                    ArrayList<JoinedTransaction> transactions = getTransactions();

                    onProgress(callback,10, total);

                    // getRegDatas
                    SparseArray<RegData> regData = getRegData(transactions);
                    onProgress(callback,20, total);
                    // parsing
                    ArrayList<MigrationCard> parsedResults = parsing(transactions, regData, smsMap);

                    onProgress(callback,30, total);

                    // saveCards
                    setCardId(parsedResults);
                    onProgress(callback,50, total);

                    // updateTransactionsByCards
                    updateTransactionsByCards(parsedResults);

                    onProgress(callback,70, total);


                    // removeCards except 현금
                    // 매칭 되지 않은 카드정보는 살려야함
                    removeCardsNotUsedExceptCache();


                    onProgress(callback,90, total);

                    // saveMigrationVersion
                    saveMigratedVersion(VERSION);

                    onProgress(callback,100, total);

                }

                appExecutors.mainThread().execute(new Runnable() {
                    @Override
                    public void run() {
                        callback.onCompleted();
                    }
                });
            }
        });
    }

    private void onProgress(final MigrationCallback callback, final int current, final int total) {
        appExecutors.mainThread().execute(new Runnable() {
            @Override
            public void run() {
                callback.onProgress(current, total);
            }
        });
    }

    private boolean checkAlreadyMigrated() {
        return prefManager.loadIntValue(MIGRATION_VERSION, 0) == VERSION;
    }

    private HashMap<String, ArrayList<SMS>> getSmsMap() {
        HashMap<String, ArrayList<SMS>> smsMap = new HashMap<>();

        ArrayList<SMS> smsList = smsLoader.findAll();

        for(SMS sms : smsList) {

            ArrayList<SMS> smsItems = smsMap.get(sms.getSmsDate());

            if(smsItems == null) {
                smsItems = new ArrayList<>();
            }

            smsItems.add(sms);

            smsMap.put(sms.getSmsDate(), smsItems);

        }

        return smsMap;
    }

    private String makeSmsKey(SMS sms) {
        return sms.getSmsDate();
    }

    private ArrayList<JoinedTransaction> getTransactions() {
        Utils.LOG("getTransactions", "start");
        return visualRepository.loadJoinedTransactions();
    }

    private SparseArray<RegData> getRegData(ArrayList<JoinedTransaction> transactions) {
        Utils.LOG("getRegData", "start");

        ArrayList<Integer> regIds = new ArrayList<>();

        for(JoinedTransaction transaction : transactions) {
            regIds.add(transaction.getTransaction().getRegId());
        }

        if(regIds.isEmpty()) {
            return new SparseArray<>();
        } else {
           SparseArray<RegData> hashMap = new SparseArray<>();

            ArrayList<RegData> results = parser.getParsingRule(regIds);

            for(RegData reg : results) {
                hashMap.put(reg.regId, reg);
            }

            return hashMap;
        }
    }

    private ArrayList<MigrationCard> parsing(ArrayList<JoinedTransaction> transactions,
                                             SparseArray<RegData> regData,
                                             HashMap<String, ArrayList<SMS>> smsMap) {
        Utils.LOG("Migration parsing", "start");


        Utils.LOG("Migration parsing", "transactions size" + transactions.size());

        ArrayList<MigrationCard> results = new ArrayList<>();

        for(JoinedTransaction transaction : transactions) {

            ArrayList<SMS> smsList = smsMap.get(transaction.getTransaction().getSmsDate());

            if(smsList != null) {
                for(SMS sms : smsList) {
                    RegData reg = regData.get(transaction.getTransaction().getRegId());
                    if(reg != null) {

                        String fullSms = transformFullSMS(sms.getFullSms());
                        Pattern pattern = Pattern.compile(reg.regExpression, Pattern.CASE_INSENSITIVE | Pattern.DOTALL | Pattern.MULTILINE | Pattern.UNICODE_CASE);
                        Matcher matcher = pattern.matcher(fullSms);

                        if(matcher.matches()) {

                            //cardName
                            String parsedCardName = getCardName(getParsedData(reg.cardName, matcher));
                            String cardName = TextUtils.isEmpty(parsedCardName) || parsedCardName.contains("null") ?
                                    Constants.KEYWORD_EMPTY : parsedCardName.trim();

                            if(cardName == null) {
                                cardName = "";
                            }

                            //cardNum
                            String cardNum = getCardNum(reg.cardNum, matcher);
                            if(cardNum == null) {
                                cardNum = "";
                            }

                            //cardType
                            int cardType = getCardType(reg.cardType, matcher);

                            //cardSubtype
                            int cardSubType = getCardSubType(reg.cardSubType, matcher);

                            results.add(new MigrationCard(transaction.getTransaction().getIdentifier(), cardName,
                                    cardNum,
                                    cardType,
                                    cardSubType));

                            break;
                        }

                    }
                }
            }

        }
        Utils.LOG("parsing", "results size" + results.size());

        return results;
    }

    private void removeCardsNotUsedExceptCache() {
        // 매칭되지 않은 카드아이디 추출
        Utils.LOG("removeCardsNotUsedExceptCache", "start");

        ArrayList<Integer> cardIds = new ArrayList<>();

        ArrayList<JoinedTransaction> transactions = visualRepository.loadJoinedTransactions();

        for(JoinedTransaction transaction : transactions) {
            cardIds.add(transaction.getCard().getId());
        }

        HashSet<Integer> usedUniqueCardIds = new HashSet<>(cardIds); // 삭제 하면 안됩니다.

        Utils.LOG("removeCardsNotUsedExceptCache", "usedUniqueCardIds size" + usedUniqueCardIds);

        visualRepository.deleteCardsByNotInIds(new ArrayList(usedUniqueCardIds)); // 현재 연결되지 않은 카드정보를 초기화 합니다.

    }


    private ArrayList<MigrationCard> setCardId(ArrayList<MigrationCard> cards) {
        // 저장 하면서 identifier 별 카드 아이디를
        Utils.LOG("setCardId", "start" + cards.size());

        for(MigrationCard card: cards) {
            card.cardId = visualRepository.loadCardId(card.getCardFullName(), card.cardType, card.cardSubType);
        }

        return cards;
    }

    private void updateTransactionsByCards(ArrayList<MigrationCard> cards) {

        visualRepository.updateTransactionCards(cards);
    }

    private int getCardType(String cardType, Matcher matcher) {
        //CardType

        return !TextUtils.isEmpty(cardType)&&Validator.isNumber(cardType)?
                getCardTypeStr(getMatcher(Integer.parseInt(cardType), matcher))
                :
                getCardTypeStr(cardType);

    }

    private int getCardTypeStr(String cardType) {

        if (TextUtils.isEmpty(cardType))
            return Constants.CardType.CARD.ordinal();

        if ("체크".equals(cardType)) {

            return Constants.CardType.CHECK.ordinal();

        } else if ("은행".equals(cardType)) {

            return Constants.CardType.BANK_ACCOUNT.ordinal();

        } else {

            return Constants.CardType.CARD.ordinal();

        }
    }

    private int getCardSubType(String cardSubType, Matcher matcher) {
        //CardSubType

        return !TextUtils.isEmpty(cardSubType)&&Validator.isNumber(cardSubType)?
                getCardSubTypeStr(getMatcher(Integer.parseInt(cardSubType), matcher))
                :
                getCardSubTypeStr(cardSubType);

    }

    private int getCardSubTypeStr(String cardSubType) {

        if (TextUtils.isEmpty(cardSubType))
            return Constants.CardSubType.NORMAL.ordinal();

        if ("법인".equals(cardSubType)) {

            return Constants.CardSubType.CORPORATION.ordinal();

        } else if ("가족".equals(cardSubType)) {

            return Constants.CardSubType.FAMILY.ordinal();

        } else {

            return Constants.CardSubType.NORMAL.ordinal();

        }
    }

    private String getCardNumUntilFour(String cardNum){

        if(TextUtils.isEmpty(cardNum))
            return "";

        cardNum = cardNum.replace("(","");
        cardNum = cardNum.replace(")","");
        cardNum = cardNum.replace("-","");
        cardNum = cardNum.replace("X","*");
        cardNum = cardNum.replace("#","*");
        cardNum = cardNum.replaceAll(" ","");
        return  cardNum.length()>4?
                "("+cardNum.substring(cardNum.length()-4)+")"
                :
                "("+cardNum+")";
    }

    private String getCardNum(String cardNum, Matcher matcher) {

        return !TextUtils.isEmpty(cardNum)&& Validator.isNumber(cardNum)?
                getCardNumUntilFour(getMatcher(Integer.parseInt(cardNum), matcher)).trim():
                "";

    }

    private String getCardName(String cardName) {

        if(TextUtils.isEmpty(cardName))
            return "";

        return cardName.replace("카드", "").replace("-","");

    }

    private String getParsedData(String regStr, Matcher matcher) {

        StringBuilder valueStr = new StringBuilder();
        if (!TextUtils.isEmpty(regStr)) {
            if (regStr.contains(",")) {
                String[] subMapping =  regStr.split(",");
                for (String sub : subMapping) {
                    if (!TextUtils.isEmpty(sub)) {

                        valueStr.append(Validator.isNumber(sub) ?
                                getMatcher(Integer.parseInt(sub), matcher)
                                :
                                sub) ;
                    }
                }
            } else {

                valueStr.append(Validator.isNumber(regStr) ?
                        getMatcher(Integer.parseInt(regStr), matcher)
                        :
                        regStr);
            }
        }

        return valueStr.toString();

    }

    private String getMatcher(int pos, Matcher matcher) {

        try {
            return matcher.group(pos);
        } catch (IndexOutOfBoundsException e) {
            return "";
        }
    }


    public void saveMigratedVersion(int version) {
        prefManager.saveIntValue(MIGRATION_VERSION, version);

    }

    private  String transformFullSMS(String fullSMS){

        if(fullSMS == null)fullSMS = "None";
        fullSMS = fullSMS.replace("[FW]", "");
        fullSMS = fullSMS.replace("　", " ");
        fullSMS = fullSMS.replace("↵"," ");
        fullSMS = fullSMS.replace("FW>", "");
        fullSMS = fullSMS.replace("듀>","");
        fullSMS = fullSMS.replace("[재전송]","");
        fullSMS = fullSMS.replace("[투]","");
        fullSMS = fullSMS.replace("[Web발신]\n", "");
        fullSMS = fullSMS.replace("[Web발신]", "");
        fullSMS = fullSMS.replace("[Web 발신]", "");
        fullSMS = fullSMS.replace("[web 발신]", "");
        fullSMS = fullSMS.replace("[web발신]", "");
        fullSMS = fullSMS.replace("[WEB발신]", "");
        fullSMS = fullSMS.replace("[WEB 발신]", "");
        fullSMS = fullSMS.replace("(재전송)", "");
        fullSMS = fullSMS.replace("[", " ");
        fullSMS = fullSMS.replace("]", " ");
        if(fullSMS.length()>4&&
                fullSMS.substring(0,4).contains("FW")){
            fullSMS = fullSMS.replace("FW", "");
        }

        return fullSMS.replace("\n", " ");
    }
}
