package com.tenqube.visual_third;

import android.app.Activity;
import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.tenqube.visual_third.exception.AuthException;
import com.tenqube.visual_third.exception.ParameterException;
import com.tenqube.visual_third.manager.AnswerManager;
import com.tenqube.visual_third.manager.VisualAlarmManager;
import com.tenqube.visual_third.model.js.LogRequest;
import com.tenqube.visual_third.repository.RepositoryHolder;
import com.tenqube.visual_third.repository.VisualRepository;
import com.tenqube.visual_third.ui.OnResultListener;
import com.tenqube.visual_third.ui.VisualWebActivity;
import com.tenqube.visual_third.util.Validator;

import tenqube.parser.core.ParserService;

import static com.tenqube.visual_third.manager.AnswerManager.initFabric;
import static com.tenqube.visual_third.util.Validator.notNull;
import static com.tenqube.visual_third.util.Validator.notZero;

public class VisualServiceImpl implements VisualService {

    public static final String TAG = VisualServiceImpl.class.getSimpleName();

    private Context context;

    private ParserService parserService;
    private VisualRepository repository;
    private VisualAlarmManager alarmManager;

    public VisualServiceImpl(@NonNull Context context,
                             @NonNull String apiKey,
                             @NonNull String qualifier,
                             @NonNull String authority) throws ParameterException {

        Validator.notNull(context);

        Validator.isNotEmpty(apiKey);

        Validator.isNotEmpty(authority);

        Validator.in(qualifier, Constants.DEV, Constants.PROD);

        this.context = context;

        repository = RepositoryHolder.getInstance(context).getVisualRepository();

        repository.saveSDKInfo(apiKey, qualifier, authority);

        parserService = ParserService.getInstance(context);

        alarmManager = VisualAlarmManager.getInstance(context);
        initFabric(context);

    }

    @Override
    public void startVisual(final @Nullable Activity activity,
                            final @NonNull String uid,
                            final @NonNull String path,
                            final @Nullable Boolean isAgree,
                            final @NonNull OnResultListener resultListener) throws ParameterException {

        Validator.isNotEmpty(uid);
        Validator.notNull(resultListener);
        Validator.notNull(path);

        repository.signUp(uid, isAgree, new OnResultListener() {
            @Override
            public void onResult(int signUpResult, String msg) {
                try {
                    //가입 성공시에만 isAgree 값에 맞게 설정정보 업데이트 처리
                    if (signUpResult == Constants.SignupResult.SUCCESS ||
                    signUpResult == Constants.SignupResult.ALREADY_JOINED) {
                        AnswerManager.onKeyMetric(new LogRequest("startVisual"));
                        if (activity == null) {
                            VisualWebActivity.startActivity(context, path);
                        } else {
                            VisualWebActivity.startActivity(activity, path);
                        }
                    }

                } catch (AuthException e) {
                    signUpResult = Constants.SignupResult.SERVER_ERROR;
                    msg = e.toString();
                } finally {
                    resultListener.onResult(signUpResult, msg);
                }
            }
        });

    }

    @Override
    public void setVisualCallback(VisualCallback callback) {
        repository.setVisualCallback(callback);
    }

    @Override
    public void signOut(Callback<Boolean> callback) {
        repository.signOut(new Callback<Boolean>() {
            @Override
            public void onDataLoaded(Boolean value) {
                alarmManager.setAlarms();

            }
        });
    }

    @Override
    public void settingNotification(int smallIcon, @NonNull String channel, int color) throws ParameterException {
        notZero(smallIcon);
        notNull(channel);
        repository.settingNotification(smallIcon, channel, color);
    }

    @Override
    public boolean isActiveTranPopup() {
        AnswerManager.onKeyMetric(new LogRequest("isActiveTranPopup"));
        return repository.isActiveTranPopup();
    }

    @Override
    public void setTranPopup(boolean isActive) {
        AnswerManager.onKeyMetric(new LogRequest("setTranPopup"));
        repository.setTranPopup(isActive);
    }

    @Override
    public void setReportAlarm(Constants.ReportAlarmType type, boolean isActive) {
        repository.setActiveNoti(type.name().toLowerCase(), isActive);
        alarmManager.setAlarms();
    }

    @Override
    public boolean isActiveReportAlarm(Constants.ReportAlarmType type) {
        return repository.isActiveNoti(type.name().toLowerCase());
    }

    @Override
    public void setReportTest(Constants.ReportAlarmType type, int second) {
        alarmManager.setReportTest(type.name().toLowerCase(), second);
    }

    @Override
    public void setLogger(boolean isActive) {
        if(isActive() && parserService != null) {
            parserService.setDebugMode(isActive);
        }
    }

    @Override
    public void initSDK() {
        if(isActive() && parserService != null) {
            parserService.initDb();
        }
    }

    @Override
    public void setEnabled(boolean enabled) {
        repository.setEnabled(enabled);
        if(!enabled) {
            repository.setNotiAll(false);
            alarmManager.setAlarms();
        }
    }

    @Override
    public boolean isJoined() {
        return repository.isJoined();
    }

    @Override
    public void setAppNoti(boolean isActive) {
        repository.setAppNoti(isActive);
        alarmManager.setAlarms();
    }

    @Override
    public boolean isActiveAppNoti() {
        return repository.isAppNoti();
    }

    private boolean isActive() {
        return context != null;
    }
}
