package com.tenqube.visual_third.analysis;

import android.content.Context;
import android.util.SparseArray;

import com.tenqube.visual_third.Constants;
import com.tenqube.visual_third.entity.Condition;
import com.tenqube.visual_third.entity.JoinedContent;
import com.tenqube.visual_third.model.analysis.Analysis;
import com.tenqube.visual_third.model.analysis.Transaction;
import com.tenqube.visual_third.repository.AnalysisRepository;
import com.tenqube.visual_third.repository.RepositoryHolder;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;

import static com.tenqube.visual_third.analysis.AnalysisHelper.descAnalysis;
import static com.tenqube.visual_third.analysis.AnalysisHelper.getContent;
import static com.tenqube.visual_third.analysis.AnalysisHelper.getValue;
import static com.tenqube.visual_third.util.Utils.isEmpty;

public class AnalysisServiceImpl implements AnalysisService {


    private final Context context;
    private final AnalysisRepository repository;

    private ArrayList<Analysis> results = new ArrayList<>();

    private long time;
    public static boolean shouldRefresh;
    private String layer;

    public AnalysisServiceImpl(Context context, String layer) {
        this.context = context;
        this.repository = RepositoryHolder.getInstance(context).getAnalysisRepository();
        this.layer = layer;
    }

    @Override
    public ArrayList<Analysis> loadAnalysisList(final ArrayList<Transaction> transactions) {

        FunctionMap functionMap = null;
        try {

            initCache();

            // 캐시가 존재하면 ? 바로 결과값 리턴
            if(!isEmpty(results)) return results;

            time = System.currentTimeMillis();

            functionMap = new FunctionMap();

            // 구한값을 가지고 계산이 필요한 부분 처리 해서 Funcmap 값 세팅
            functionMap.calculate(transactions);

            ArrayList<JoinedContent> allContents = repository.loadContents();

            ArrayList<JoinedContent> categoryContents = new ArrayList<>();

            Iterator<JoinedContent> it = allContents.iterator();
            while (it.hasNext()) {
                JoinedContent joinedContent = it.next();
                if (joinedContent.getContent().getlCode() == 0) {
                    categoryContents.add(joinedContent);
                    it.remove();
                }
            }

            // 조건 체크하는 분석 정보 필터
            allContents = getSatisfiedContents(allContents);

            // 조건에 만족하는 분석 내용과 데이터 합치기
            SparseArray<ArrayList<Analysis>> analysisByCategory = mergeContentsByCategory(allContents);

            analysisByCategory = functionMap.loadCommonCategory(analysisByCategory, categoryContents);

            results = selectAnalysis(analysisByCategory);

            descAnalysis(results);

        } catch (Exception e) {

        } finally {
            FunctionMap.values.clear();
            FunctionMap.tranIds.clear();
        }

        return results;
    }

    private ArrayList<Analysis> selectAnalysis(SparseArray<ArrayList<Analysis>> analysisByCategory) {

        ArrayList<Analysis> selectedAnalysis = new ArrayList<>();

//        if(TextUtils.isEmpty(layer) || Constants.DEV.equals(layer)) {
//            for(int i = 0 ; i < analysisByCategory.size() ; i++) {
//                ArrayList<Analysis> categoryAnalysis = analysisByCategory.get(analysisByCategory.keyAt(i));
//                if(!isEmpty(categoryAnalysis)) selectedAnalysis.addAll(categoryAnalysis);
//            }
//        } else {
            // 월간
        ArrayList<Analysis> monthly = analysisByCategory.get(Constants.LCODE.MONTHLY);
        analysisByCategory.remove(Constants.LCODE.MONTHLY);


        if(!isEmpty(monthly)) {
            Collections.shuffle(monthly);
            selectedAnalysis.add(monthly.get(0));
        }

        // 주간
        ArrayList<Analysis> weekly = analysisByCategory.get(Constants.LCODE.WEEKLY);
        analysisByCategory.remove(Constants.LCODE.WEEKLY);

        if(!isEmpty(weekly)){
            Collections.shuffle(weekly);
            selectedAnalysis.add(weekly.get(0));
        }

        // 카테고리별
        for(int i = 0 ; i < analysisByCategory.size() ; i++) {
            int lcode = analysisByCategory.keyAt(i);
            ArrayList<Analysis> categoryAnalysis = analysisByCategory.get(lcode);

            if(!isEmpty(categoryAnalysis)) {
                Collections.shuffle(categoryAnalysis);
                selectedAnalysis.add(categoryAnalysis.get(0));
            }
        }
        analysisByCategory.clear();

        return selectedAnalysis;
    }

    /**
     * 캐쉬 초기화 조건
     * 1. 내역 수신시
     * 2. 내역 편집시
     * 3. 60 초
     */
    private void initCache() {
        if(shouldRefresh || isTimeOver()) {
            shouldRefresh = false;
            time = 0;
            results.clear();
        }
    }

    private boolean isTimeOver() {
        return ((System.currentTimeMillis() - time) / 1000) > 60;
    }


    private ArrayList<JoinedContent> getSatisfiedContents(ArrayList<JoinedContent> contents) {

        ArrayList<JoinedContent> results = new ArrayList<>();

        for(JoinedContent content : contents) {

            // 조건 체크
            boolean isSatisfied = true;
            ArrayList<Condition> conditions = content.getConditions();

            if(!isEmpty(conditions)) {
                for(Condition condition : conditions) {
                    isSatisfied = AnalysisHelper.isSatisfied(condition);
                    if(!isSatisfied) break;
                }
            }
            if(isSatisfied) {
                results.add(content);
            }
        }
        return results;
    }

    private SparseArray<ArrayList<Analysis>> mergeContentsByCategory(ArrayList<JoinedContent> contents) {

        SparseArray<ArrayList<Analysis>> analysisMap = new SparseArray<>();

        ArrayList<Analysis> analyses;

        for(JoinedContent content : contents) {

            try {
                String image = getValue(content.getContent().getImage());
                String label = getValue(content.getContent().getLabel());

                String lContent = getContent(content.getContent().getlContent(), content.getContent().getlKeys());
                String mContent = getContent(content.getContent().getmContent(), content.getContent().getmKeys());

                Analysis analysis = new Analysis(
                        content.getContent().getId(),
                        image,
                        label,
                        lContent,
                        mContent,
                        mergeTranIds(content.getContent().getRawKeys()),
                        content.getContent().getCategoryPriority()
                        );

                analyses = analysisMap.get(content.getContent().getlCode());
                if(analyses == null) {
                    analyses = new ArrayList<>();
                }
                analyses.add(analysis);

                analysisMap.put(content.getContent().getlCode(), analyses);
            } catch (Exception e) {

            }
        }

        return analysisMap;

    }

    private ArrayList<Integer> mergeTranIds(String rawKeys) {

        ArrayList<Integer> tranIds = new ArrayList<>();
        String[] keys = rawKeys.split(",");
        for(String key : keys) {
            ArrayList<Integer> tempTranIds = FunctionMap.tranIds.get(Integer.parseInt(key));
            if(tempTranIds != null)
                tranIds.addAll(tempTranIds);
        }
        return tranIds;
    }

}
