package com.tenqube.visual_third.api;

import android.content.Context;
import android.support.annotation.NonNull;

import com.google.gson.Gson;
import com.tenqube.visual_third.manager.PrefManager;
import com.tenqube.visual_third.model.api.AdResponse;
import com.tenqube.visual_third.model.api.AnalysisResponse;
import com.tenqube.visual_third.model.api.CurrencyResponse;
import com.tenqube.visual_third.model.api.ParsingRuleKey;
import com.tenqube.visual_third.model.api.SearchCompanyRequest;
import com.tenqube.visual_third.model.api.SearchCompanyResponse;
import com.tenqube.visual_third.model.api.SignUpRequest;
import com.tenqube.visual_third.model.api.SignUpResponse;
import com.tenqube.visual_third.model.api.SyncCategoryResponse;
import com.tenqube.visual_third.model.api.TransactionRequest;
import com.tenqube.visual_third.model.api.VersionResponse;
import com.tenqube.visual_third.model.js.AdRequest;
import com.tenqube.visual_third.repository.AnalysisRepository;
import com.tenqube.visual_third.repository.VisualRepository;
import com.tenqube.visual_third.util.Utils;

import java.util.Calendar;
import java.util.concurrent.TimeUnit;

import okhttp3.OkHttpClient;
import retrofit2.Call;
import retrofit2.Callback;
import retrofit2.Response;
import retrofit2.Retrofit;
import retrofit2.converter.gson.GsonConverterFactory;
import tenqube.parser.OnNetworkResultListener;
import tenqube.parser.model.ParsingRule;

import static com.tenqube.visual_third.util.Utils.isEmpty;

public class VisualApi {

    public static final String TAG = VisualApi.class.getSimpleName();

    public static VisualApi mInstance;

    private Context context;
    private VisualApiService visualApiService;

    private PrefManager prefManager;
    private VisualRepository repository;
    private AnalysisRepository analysisRepository;

    public static final int API_KEY = 0;
    public static final int AUTHORIZATION = 1;

    public static VisualApi getInstance(Context context){
        synchronized (VisualApi.class) {
            if(mInstance == null) {
                mInstance = new VisualApi(context.getApplicationContext());
            }
        }
        return mInstance;
    }

    private VisualApi(Context context) {
        this.context = context;
        prefManager = PrefManager.getInstance(context);
        repository = VisualRepository.getInstance(context);
        analysisRepository = AnalysisRepository.getInstance(context);
    }

    private VisualApiService getApiService() {
        if (visualApiService == null) {
            visualApiService = provideRetrofit(getUrl()).create(VisualApiService.class);
        }
        return visualApiService;
    }

    public String[] getAuthInfo() {
        return new String[]{prefManager.loadStringValue(PrefManager.API_KEY, ""),
                prefManager.loadStringValue(PrefManager.UID, "")
        };
    }

    private String[] getSearchAuthInfo() {
        return new String[]{prefManager.loadStringValue(PrefManager.SEARCH_API_KEY, ""),
                prefManager.loadStringValue(PrefManager.UID, "")
        };
    }

    private String getUrl() {
        return VisualApiService.BASE_URL + prefManager.loadStringValue(PrefManager.QUALIFIER, "dev") + "/";
    }

    private Retrofit provideRetrofit(String url) {
        return new Retrofit.Builder()
                .baseUrl(url)
                .client(provideOkHttpClient())
                .addConverterFactory(GsonConverterFactory.create(new Gson()))

                .build();
    }

    private OkHttpClient provideOkHttpClient() {
        OkHttpClient.Builder okhttpClientBuilder = new OkHttpClient.Builder();
        okhttpClientBuilder.connectTimeout(5, TimeUnit.SECONDS);
        okhttpClientBuilder.readTimeout(5, TimeUnit.SECONDS);
        okhttpClientBuilder.writeTimeout(5, TimeUnit.SECONDS);
        return okhttpClientBuilder.build();
    }

    public SignUpResponse signUp(SignUpRequest signUpInfo) {

        try {
            Response<SignUpResponse> response = getApiService().signUp(getAuthInfo()[API_KEY], signUpInfo).execute();
            if (response.isSuccessful()) {
                return response.body();
            }
        } catch (Exception e) {

            e.printStackTrace();
        }

        return null;
    }

    public VersionResponse syncVersion(String apiKey) {

        try {
            Response<VersionResponse> response = getApiService().syncVersion(apiKey).execute();
            if (response.isSuccessful()) {
                return response.body();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }

        return null;
    }

    public void syncCategory(String[] authInfo, int version) {
        try {
            Response<SyncCategoryResponse> response = getApiService().syncCategory(authInfo[API_KEY], authInfo[AUTHORIZATION], version).execute();
            if (response.isSuccessful()) {
                SyncCategoryResponse categoryResponse = response.body();
                if (categoryResponse != null) {
                    prefManager.saveIntValue(PrefManager.CATEGORY_VERSION, categoryResponse.getVersion());
                    if (!categoryResponse.getCategories().isEmpty())
                        repository.mergeCategory(categoryResponse.getCategories());
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public ParsingRule syncParsingRule(int version) {
        try {
            String[] authInfo = getAuthInfo();
            Response<ParsingRule> response = getApiService().syncParsingRule(authInfo[API_KEY], authInfo[AUTHORIZATION], version).execute();
            if(response.isSuccessful()) {
                return response.body();
            }

        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    public ParsingRuleKey syncParsingRuleKey() {
        try {
            String[] authInfo = getAuthInfo();
            Response<ParsingRuleKey> response = getApiService().syncParsingRuleKey(authInfo[API_KEY]).execute();
            if(response.isSuccessful()) {
                return response.body();
            }

        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    public void syncAnalysis(String[] authInfo, int version) {

        try {
            Response<AnalysisResponse> response = getApiService().syncAnalysis(authInfo[API_KEY], authInfo[AUTHORIZATION], version).execute();
            if(response.isSuccessful()) {
                AnalysisResponse analysisResponse = response.body();
                if(analysisResponse != null) {
                    prefManager.saveIntValue(PrefManager.ANALYSIS_VERSION, analysisResponse.getVersion());
                    if(!isEmpty(analysisResponse.getContents()))
                        analysisRepository.mergeAnalysis(analysisResponse.getContents());
                }
            }

        } catch (Exception e) {
            e.printStackTrace();
        }

    }

    public SearchCompanyResponse searchCompany(@NonNull  SearchCompanyRequest searchCompanyRequest) {
        try {

            String[] searchAuthInfo = getSearchAuthInfo();
            String searchUrl = prefManager.loadStringValue(PrefManager.SEARCH_URL, "");

            Response<SearchCompanyResponse> response = getApiService().searchCompany(searchAuthInfo[API_KEY], searchAuthInfo[AUTHORIZATION],
                    searchUrl, searchCompanyRequest).execute();

            if(response.isSuccessful()) {
                repository.updateRetrySearch(searchCompanyRequest.getTransactions(), false);

                return response.body();
            } else {
                if(response.code() == 499 || response.code() == 500) {
                    // retry
                    // 사용자가 변경 하지않고, retry flag true 인경우 쿼리해서 다시 보내줘야함
                    // update retry flag, isSynced = false
                    repository.updateRetrySearch(searchCompanyRequest.getTransactions(), true);

                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }

        return null;
    }

    private void saveTransactions(@NonNull TransactionRequest transactionRequest, Callback<Void> callback) {
        String[] authInfo = getAuthInfo();

        Call<Void> call = getApiService().saveTransactions(authInfo[API_KEY], authInfo[AUTHORIZATION], transactionRequest);
        call.enqueue(callback);
    }

    public void syncTransactions(final OnNetworkResultListener callback) {
        final TransactionRequest transactionRequest = repository.loadNotSyncedTransactions();
        if(transactionRequest != null && !isEmpty(transactionRequest.getTransactions())) {
            saveTransactions(transactionRequest, new Callback<Void>() {
                @Override
                public void onResponse(Call<Void> call, Response<Void> response) {

                    if(response.isSuccessful() || response.code() == 400) {
                        repository.updateSyncedTransactions(transactionRequest);
                        repository.deleteTransactions();
                    }
                    if(callback != null) callback.onResult(true);
                }

                @Override
                public void onFailure(Call<Void> call, Throwable t) {

                    if(callback != null) callback.onResult(true);
                }
            });
        } else {
            if(callback != null) callback.onResult(true);
        }
    }

    public double getCurrencyRate(String from , String to, double localRate) {
        try {
            String[] authInfo = getAuthInfo();

            Response<CurrencyResponse> response = getApiService().callCurrencyRate(authInfo[API_KEY], authInfo[AUTHORIZATION],
                    from.toLowerCase(), to.toLowerCase()).execute();

            if(response.isSuccessful()) {
                CurrencyResponse currencyResponse = response.body();
                if(currencyResponse != null) {
                    repository.mergeCurrency(from, to, currencyResponse.getRate(), Utils.getStringDateAsYYYYMMddHHmmss(Calendar.getInstance()));
                    return currencyResponse.getRate();
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }

        return localRate;
    }

    public void getAds(final AdRequest adRequest, Callback<AdResponse> callback) {

        try {

            String[] authInfo = getAuthInfo();
            Call<AdResponse> call = getApiService().getAds(authInfo[API_KEY], adRequest.getWhere());
            call.enqueue(callback);

        } catch (Exception e) {
            e.printStackTrace();
        }

    }

}
