package com.tenqube.visual_third.ui;

import android.Manifest;
import android.annotation.SuppressLint;
import android.app.Activity;
import android.app.FragmentManager;
import android.content.Context;
import android.content.Intent;
import android.os.Bundle;
import android.os.Handler;
import android.os.Message;
import android.support.annotation.NonNull;
import android.support.v4.content.LocalBroadcastManager;
import android.support.v4.widget.SwipeRefreshLayout;
import android.text.TextUtils;
import android.webkit.WebSettings;
import android.webkit.WebView;
import android.webkit.WebViewClient;
import android.widget.Toast;

import com.tenqube.visual_third.R;
import com.tenqube.visual_third.VisualServiceImpl;
import com.tenqube.visual_third.analysis.AnalysisServiceImpl;
import com.tenqube.visual_third.api.VisualApi;
import com.tenqube.visual_third.exception.AuthException;
import com.tenqube.visual_third.exception.ParameterException;
import com.tenqube.visual_third.manager.AnswerManager;
import com.tenqube.visual_third.manager.PrefManager;
import com.tenqube.visual_third.manager.ResourceManager;
import com.tenqube.visual_third.manager.VisualAlarmManager;
import com.tenqube.visual_third.model.api.SearchCompanyRequest;
import com.tenqube.visual_third.model.js.LogRequest;
import com.tenqube.visual_third.model.parser.PopupData;
import com.tenqube.visual_third.parser.loader.BulkLoader;
import com.tenqube.visual_third.repository.VisualRepository;
import com.tenqube.visual_third.util.CSVHelper;
import com.tenqube.visual_third.util.PathUtil;
import com.tenqube.visual_third.util.permission.AfterPermissionGranted;
import com.tenqube.visual_third.util.permission.PermissionUtil;
import com.tenqube.visual_third.web.ActionImpl;
import com.tenqube.visual_third.web.Completion;
import com.tenqube.visual_third.web.ErrorImpl;
import com.tenqube.visual_third.web.LogImpl;
import com.tenqube.visual_third.web.RepoImpl;
import com.tenqube.visual_third.web.SystemImpl;
import com.tenqube.visual_third.web.UiImpl;
import com.tenqube.visual_third.web.VisualInterface;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.List;

import tenqube.parser.core.ParserService;

import static com.tenqube.visual_third.analysis.AnalysisServiceImpl.ACTION_REFRESHED;
import static com.tenqube.visual_third.manager.AnswerManager.initFabric;
import static com.tenqube.visual_third.util.Utils.isNotiEnabled;
import static com.tenqube.visual_third.util.Utils.isSDKDisabled;
import static com.tenqube.visual_third.util.Validator.notNull;
import static com.tenqube.visual_third.util.WebViewHelper.getBaseUrl;
import static com.tenqube.visual_third.util.WebViewHelper.settings;
import static tenqube.parser.core.ParserService.mIsDebug;
import static tenqube.parser.util.LogUtil.LOGI;

public class VisualWebActivity extends Activity implements TimePickerFragment.Callback, DatePickerFragment.Callback, PermissionUtil.PermissionCallbacks{

    public static final String ARG_PATH = "ARG_PATH";

    public static final String TAG = VisualServiceImpl.class.getSimpleName();

    public static final int NOTIFICATION_REQUEST_CODE = 10;

    public static final int FINISH = 10;
    public static final int PROGRESS = 11;
    public static final int SHOW_PROGRESS_BAR = 12;
    public static final int ERROR = 13;
    private ProgressDialogFragment progressDialog;
    private PrefManager prefManager;
    private VisualApi api;
    private VisualRepository repository;
    private ResourceManager resourceManager;
    private VisualAlarmManager alarmManager;
    private boolean isPageLoaded;
    private String mFailingUrl;

    private String path;
    private WebView webview;
    private BulkLoader bulkLoader;

    private VisualInterface.Repo repoInterface;
    private VisualInterface.UI uiInterface;
    private VisualInterface.System systemInterface;
    private VisualInterface.Error errorInterface;
    private VisualInterface.Log logInterface;
    private VisualInterface.Action actionInterface;

    private SwipeRefreshLayout swipeRefreshLayout;

    @SuppressLint({"JavascriptInterface", "AddJavascriptInterface", "SetJavaScriptEnabled"})
    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        try {
            if(isSDKDisabled(getApplicationContext())) {
                throw new Exception("SDK disabled");
            }

            initFabric(VisualWebActivity.this);
            AnswerManager.onKeyMetric(new LogRequest("VisualWebActivity"));
            api = VisualApi.getInstance(getApplicationContext());
            repository = VisualRepository.getInstance(getApplicationContext());
            prefManager = PrefManager.getInstance(getApplicationContext());
            resourceManager = ResourceManager.getInstance(api, prefManager, ParserService.getInstance(getApplicationContext()));
            alarmManager = VisualAlarmManager.getInstance(getApplicationContext());

            path = getIntent().getStringExtra(ARG_PATH);
            LOGI("PATH", path, true);
            setContentView(R.layout.activity_visual_web);

            swipeRefreshLayout = findViewById(R.id.swipe_refresh_layout);
            swipeRefreshLayout.setEnabled(false);
            swipeRefreshLayout.setColorSchemeResources(
                    R.color.colorPopupRed,
                    R.color.colorPopupRed,
                    R.color.colorPopupRed);

            swipeRefreshLayout.setOnRefreshListener(new SwipeRefreshLayout.OnRefreshListener() {
                @Override
                public void onRefresh() {
                    swipeRefreshLayout.setRefreshing(false);
                    webview.reload();
                    LocalBroadcastManager.getInstance(VisualWebActivity.this).sendBroadcast(new Intent(ACTION_REFRESHED));
                }
            });

            webview = findViewById(R.id.visual_web_view);

            // javascriptInterface
            errorInterface = new ErrorImpl(this, webview);
            logInterface = new LogImpl(this, webview, errorInterface);

            repoInterface = new RepoImpl(this, webview, errorInterface, repository, api, new AnalysisServiceImpl(this, ""), alarmManager);
            uiInterface = new UiImpl(this, webview, errorInterface, new BottomDialog(this));
            systemInterface = new SystemImpl(this, webview, errorInterface);
            actionInterface = new ActionImpl(this, webview);

            webview.addJavascriptInterface(errorInterface, "visualError");
            webview.addJavascriptInterface(repoInterface, "visualRepo");
            webview.addJavascriptInterface(uiInterface, "visualUI");
            webview.addJavascriptInterface(systemInterface, "visualSystem");
            webview.addJavascriptInterface(logInterface, "visualLog");
            webview.addJavascriptInterface(actionInterface, "visualAction");

            // settings
            settings(webview);

            // error
            webview.setWebViewClient(new WebViewClient() {
                public void onReceivedError(WebView view, int errorCode, String description, String failingUrl) {
                    mFailingUrl = failingUrl;
                    loadErrorUrl();
                }
            });

            if(prefManager.loadStringValue(PrefManager.QUALIFIER, "dev").equals("dev"))
                webview.getSettings().setCacheMode(WebSettings.LOAD_NO_CACHE);

            if(!prefManager.isEnabled(PrefManager.BULK_EXECUTED, false)) { // 벌크 파싱 진행

                LOGI(TAG, "벌크 파싱 진행", mIsDebug);
                startBulkParsing();

            } else {

                alarmManager.setAlarms();

                // 리소스 동ㄱ화
                resourceManager.sync();

                // 웹 가계부 호출
                loadUrl(path);

                // 검색 실패 내역 다시 보내기
                syncSearchFailedTransactions();
            }
        } catch (Exception e) {
            finish();
        }
    }

    public void onPagedLoaded() {
        runOnUiThread(new Runnable() {
            @Override
            public void run() {
                isPageLoaded = true;
            }
        });
    }

    private void loadUrl(String path) {
        if(webview != null) {
            isPageLoaded = false;
            if(path == null) path = "";
            String url = getBaseUrl(prefManager) + path;
            webview.loadUrl(url);
        }


    }

    private void loadErrorUrl() {
        isPageLoaded = false;
        if(swipeRefreshLayout != null) swipeRefreshLayout.setEnabled(false);
        if(webview != null) webview.loadUrl("file:///android_asset/index.html");
    }

    /**
     * 검색 실패 내역 다시 보내기
     */
    private void syncSearchFailedTransactions() {
        new Thread(new Runnable() {
            @Override
            public void run() {
                SearchCompanyRequest request = repository.loadFailedSearchTransactions();

                if(request != null) api.searchCompany(request);

                api.syncTransactions(null);
            }
        }).start();

    }

    private Handler handler = new Handler(new Handler.Callback() { // 핸들러로 전달받은 값 progress 값 세팅해주기
        @Override
        public boolean handleMessage(final Message msg) {

            try {
                switch (msg.what) {
                    case PROGRESS:
                        if(progressDialog != null && progressDialog.isAdded()) {
                            Bundle bundle = msg.getData();
                            if(bundle != null) {
                                progressDialog.setProgressValue(bundle.getInt("now"), bundle.getInt("total"), bundle.getInt("percent"));
                            }
                        }
                        break;

                    case SHOW_PROGRESS_BAR:

                        try {
                            if(progressDialog == null) {
                                progressDialog = ProgressDialogFragment.newInstance(VisualWebActivity.this);
                                progressDialog.setCancelable(false);
                            }

                            if(!progressDialog.isAdded()) {
                                FragmentManager fragmentManager = getFragmentManager();
                                if(fragmentManager != null) {
                                    fragmentManager.beginTransaction().add(progressDialog, ProgressDialogFragment.TAG).commitAllowingStateLoss();
                                }
                            }

                        } catch (Exception e) {
                            progressDialog = null;
                        }

                        break;
                    case FINISH:
                    case ERROR:
                    default:
                        onBulkFinish();
                        break;
                }
            } catch (Exception e) {
                onBulkFinish();
            }

            return false;
        }
    });

    private void onBulkFinish() {
        prefManager.saveBoolean(PrefManager.BULK_EXECUTED, true);
        if(progressDialog != null && progressDialog.isAdded()) progressDialog.dismissAllowingStateLoss();
        loadUrl(path);
    }

    /**
     * 1. 퍼미션 체크
     * 2. 프로그레스 다이얼로그 show
     * 3. 지난 6개월 문자 파싱 시작
     */
    private void startBulkParsing() {

        new Thread(new Runnable() {
            @Override
            public void run() {
                try {

                    LOGI(TAG, "startBulkParsing start", mIsDebug);
                    bulkLoader = new BulkLoader(VisualWebActivity.this, handler, new BulkLoader.OnSmsLoadFinished() {
                        @Override
                        public void onFinished(boolean isError) {
                            try {
                                if(handler != null) handler.sendEmptyMessage(FINISH);
                            } catch (Exception e) {
                                e.printStackTrace();
                            }
                        }
                    });
                    // 2. 프로그레스 다이얼로그 show
                    LOGI(TAG, "2. 프로그레스 다이얼로그 show", mIsDebug);
                    handler.sendEmptyMessage(SHOW_PROGRESS_BAR);

                    // 3. 지난 6개월 문자 파싱 시작
                    LOGI(TAG, "3. 지난 6개월 문자 파싱 시작", mIsDebug);
                    bulkLoader.doParsing();

                } catch (Exception e) {
                    handler.sendEmptyMessage(ERROR);
                }

            }
        }).start();
    }

    @Override
    protected void onActivityResult(int requestCode, int resultCode, final Intent data) {
        super.onActivityResult(requestCode, resultCode, data);

        switch (requestCode) {
            case NOTIFICATION_REQUEST_CODE:
                boolean isEnabled = isNotiEnabled(getApplicationContext());

                JSONObject jsonObject = new JSONObject();
                try {
                    jsonObject.put("isNotiEnabled", isEnabled);
                    String json = jsonObject.toString();
                    if(systemInterface != null) systemInterface.onNotiCallback(json);

                } catch (JSONException e) {
                    e.printStackTrace();
                }
                break;

        }
    }

    @Override
    public void onBackPressed() {

        try {
            if(isPageLoaded && uiInterface != null) {
                uiInterface.onFinish();
            } else {
                super.onBackPressed();
            }
        } catch (Exception e) {
            super.onBackPressed();
        }
    }

    @Override
    public void onResume() {
        super.onResume();
        overridePendingTransition(0, 0);
    }

    @Override
    public void onPause() {
        super.onPause();
        overridePendingTransition(0, 0);
    }

    @Override
    public void onCalendar(String date, String callback) {
        uiInterface.onCalendar(date, callback);
    }

    public void onScrollChanged(final int pos) {
        return;
    }

    public void setRefreshEnabled(final boolean enabled) {
        swipeRefreshLayout.post(new Runnable() {
            @Override
            public void run() {
                swipeRefreshLayout.setEnabled(enabled);
            }
        });
    }

    public void reload() {
        if(webview != null)  {
            webview.post(new Runnable() {
                @Override
                public void run() {
                    webview.reload();
                }
            });
        }
    }

    public void retry() {
        if(webview != null)  {
            webview.post(new Runnable() {
                @Override
                public void run() {
                    webview.loadUrl(mFailingUrl);
                }
            });
        }
    }

    public static String[] PERMISSIONS_STORAGE = {Manifest.permission.READ_EXTERNAL_STORAGE, Manifest.permission.WRITE_EXTERNAL_STORAGE};
    public static final int STORAGE = 1;

    public Completion<Boolean> completion;

    @AfterPermissionGranted(STORAGE)
    public void exportExcel(Completion<Boolean> completion) {

        this.completion = completion;
        if (PermissionUtil.hasPermissions(
                this, PERMISSIONS_STORAGE)) {
            new CSVHelper(repository).export(completion);

        } else {
            PermissionUtil.requestPermissions(
                    this,
                    STORAGE,
                    PERMISSIONS_STORAGE);
        }
    }

    @Override
    public void onPermissionsGranted(int requestCode, List<String> list) {

        if (requestCode == STORAGE) {
            exportExcel(completion);
        }
    }

    @Override
    public void onPermissionsDenied(int requestCode, List<String> list) {
        Toast.makeText(getApplicationContext(), "엑셀 파일 내보내기를 위해 외부 저장소에 대한 권한이 필요합니다. 권한이 없는 경우 엑셀 파일 내보내기 사용이 불가합니다.", Toast.LENGTH_SHORT).show();
    }

    @Override
    public void onRequestPermissionsResult(int requestCode, String[] permissions, int[] grantResults) {
        super.onRequestPermissionsResult(requestCode, permissions, grantResults);

        // Forward results to EasyPermissions
        PermissionUtil.onRequestPermissionsResult(requestCode, permissions, grantResults, this);
    }


    public static void startActivity(Context context, String path) throws AuthException {
        if(TextUtils.isEmpty(PrefManager.getInstance(context).loadStringValue(PrefManager.UID, "")))
            throw new AuthException("Need to sign up");

        Intent intent = new Intent(context, VisualWebActivity.class);
        intent.putExtra(ARG_PATH, path);
        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK |
                Intent.FLAG_ACTIVITY_CLEAR_TOP |
                Intent.FLAG_ACTIVITY_REORDER_TO_FRONT);
        context.startActivity(intent);
    }

    public static void startActivity(@NonNull Context context, @NonNull PopupData popupData) throws AuthException, ParameterException {

        notNull(context);
        notNull(popupData);

        if(TextUtils.isEmpty(PrefManager.getInstance(context).loadStringValue(PrefManager.UID, "")))
            throw new AuthException("Need to sign up");

        Intent intent = new Intent(context, VisualWebActivity.class);
        intent.putExtra(ARG_PATH, PathUtil.makePath(popupData));
        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK |
                Intent.FLAG_ACTIVITY_CLEAR_TOP |
                Intent.FLAG_ACTIVITY_REORDER_TO_FRONT);
        context.startActivity(intent);
    }
}
