package com.tenqube.visual_third.db.dao;

import android.content.ContentValues;
import android.database.Cursor;
import android.database.sqlite.SQLiteException;
import android.text.TextUtils;
import android.util.Log;

import com.tenqube.visual_third.db.Db;
import com.tenqube.visual_third.db.QueryGenerator;
import com.tenqube.visual_third.db.ReaderContract;
import com.tenqube.visual_third.entity.Card;
import com.tenqube.visual_third.entity.Category;
import com.tenqube.visual_third.entity.JoinedTransaction;
import com.tenqube.visual_third.entity.Transaction;
import com.tenqube.visual_third.entity.UserCategory;
import com.tenqube.visual_third.model.api.SearchCompanyResponse;
import com.tenqube.visual_third.model.js.TransactionByIdsRequest;
import com.tenqube.visual_third.model.js.TransactionRequest;
import com.tenqube.visual_third.model.js.UpdateTransactionRequest;

import java.util.ArrayList;

import static com.tenqube.visual_third.db.ReaderContract.TransactionsTable.COLUMN_CARD_ID;
import static com.tenqube.visual_third.db.ReaderContract.TransactionsTable.COLUMN_CATEGORY_CODE;
import static com.tenqube.visual_third.db.ReaderContract.TransactionsTable.COLUMN_CLASS_CODE;
import static com.tenqube.visual_third.db.ReaderContract.TransactionsTable.COLUMN_COMPANY_ID;
import static com.tenqube.visual_third.db.ReaderContract.TransactionsTable.COLUMN_CURRENCY;
import static com.tenqube.visual_third.db.ReaderContract.TransactionsTable.COLUMN_DW_TYPE;
import static com.tenqube.visual_third.db.ReaderContract.TransactionsTable.COLUMN_FINISH_DATE;
import static com.tenqube.visual_third.db.ReaderContract.TransactionsTable.COLUMN_FRANCHISE;
import static com.tenqube.visual_third.db.ReaderContract.TransactionsTable.COLUMN_FULL_SMS;
import static com.tenqube.visual_third.db.ReaderContract.TransactionsTable.COLUMN_IDENTIFIER;
import static com.tenqube.visual_third.db.ReaderContract.TransactionsTable.COLUMN_INSTALLMENT_COUNT;
import static com.tenqube.visual_third.db.ReaderContract.TransactionsTable.COLUMN_IS_OFFSET;
import static com.tenqube.visual_third.db.ReaderContract.TransactionsTable.COLUMN_IS_SYNCED;
import static com.tenqube.visual_third.db.ReaderContract.TransactionsTable.COLUMN_IS_UPDATE_ALL;
import static com.tenqube.visual_third.db.ReaderContract.TransactionsTable.COLUMN_KEYWORD;
import static com.tenqube.visual_third.db.ReaderContract.TransactionsTable.COLUMN_MEMO;
import static com.tenqube.visual_third.db.ReaderContract.TransactionsTable.COLUMN_ORI_SPENT_MONEY;
import static com.tenqube.visual_third.db.ReaderContract.TransactionsTable.COLUMN_REG_ID;
import static com.tenqube.visual_third.db.ReaderContract.TransactionsTable.COLUMN_SEARCH_KEYWORD;
import static com.tenqube.visual_third.db.ReaderContract.TransactionsTable.COLUMN_SENDER;
import static com.tenqube.visual_third.db.ReaderContract.TransactionsTable.COLUMN_SHOULD_RETRY_SEARCH;
import static com.tenqube.visual_third.db.ReaderContract.TransactionsTable.COLUMN_SMS_DATE;
import static com.tenqube.visual_third.db.ReaderContract.TransactionsTable.COLUMN_SMS_TYPE;
import static com.tenqube.visual_third.db.ReaderContract.TransactionsTable.COLUMN_SPENT_DATE;
import static com.tenqube.visual_third.db.ReaderContract.TransactionsTable.COLUMN_SPENT_LATITUDE;
import static com.tenqube.visual_third.db.ReaderContract.TransactionsTable.COLUMN_SPENT_LONGITUDE;
import static com.tenqube.visual_third.db.ReaderContract.TransactionsTable.COLUMN_SPENT_MONEY;
import static com.tenqube.visual_third.db.ReaderContract.TransactionsTable.COLUMN_USER_CATEGORY_ID;
import static tenqube.parser.core.ParserService.mIsDebug;
import static tenqube.parser.util.LogUtil.LOGI;


public class TransactionDao {

    private final Db db;

    public TransactionDao(Db db) {
        this.db = db;
    }

    public int loadTranId(String identifier) {

        String query = QueryGenerator.tranId(identifier);
        Cursor c = null;

        try {
            c = db.runQuery(query);
            if(c != null) {
                if(c.moveToFirst()) {
                    while (!c.isAfterLast()) {
                        return c.getInt(0);
                    }
                }
            }
        } catch (SQLiteException e) {
            e.printStackTrace();
        } finally {
            if(c != null) {
                c.close();
            }
        }

        return 0;
    }


    public void loadAll() {

        String query = QueryGenerator.transaction();
        Cursor c = null;

        try {
            c = db.runQuery(query);
            if(c != null) {
                if(c.moveToFirst()) {
                    while (!c.isAfterLast()) {
                        LOGI("LOADALL", ReaderContract.TransactionsTable.populateModel(c).toString(), mIsDebug);
                        c.moveToNext();
                    }
                }
            }
        } catch (SQLiteException e) {
            e.printStackTrace();
        } finally {
            if(c != null) {
                c.close();
            }
        }
    }

    public Transaction loadApplyAllTran(String keyword) {

        String query = QueryGenerator.applyAllTran(keyword);
        Cursor c = null;

        try {
            c = db.runQuery(query);
            if(c != null) {
                if(c.moveToFirst()) {
                    while (!c.isAfterLast()) {
                        return ReaderContract.TransactionsTable.populateModel(c);
                    }
                }
            }
        } catch (SQLiteException e) {
            e.printStackTrace();
        } finally {
            if(c != null) {
                c.close();
            }
        }

        return null;
    }

    public JoinedTransaction loadJoinedTransaction(int tranId) {
        String query = QueryGenerator.joinedTransaction(tranId);
        Cursor c = null;

        try {
            c = db.runQuery(query);
            if(c != null) {
                if(c.moveToFirst()) {
                    while (!c.isAfterLast()) {
                        Transaction transaction = ReaderContract.TransactionsTable.populateModel(c);
                        UserCategory userCategory = ReaderContract.UserCategoryTable.populateModel(c);
                        Card card = ReaderContract.CardTable.populateModel(c);
                        Category category = ReaderContract.CategoryTable.populateModel(c);
                        return new JoinedTransaction(transaction, card, userCategory, category);
                    }
                }
            }
        } catch (SQLiteException e) {
            e.printStackTrace();
        } finally {
            if(c != null) {
                c.close();
            }
        }

        return null;
    }

    public ArrayList<JoinedTransaction> loadJoinedTransactions(TransactionRequest transactionRequest) {
        String query = QueryGenerator.joinedTransaction(transactionRequest);

        ArrayList<JoinedTransaction> results = new ArrayList<>();
        Cursor c = null;

        try {
            c = db.runQuery(query);
            if(c != null) {
                if(c.moveToFirst()) {
                    while (!c.isAfterLast()) {
                        Transaction transaction = ReaderContract.TransactionsTable.populateModel(c);
                        UserCategory userCategory = ReaderContract.UserCategoryTable.populateModel(c);
                        Card card = ReaderContract.CardTable.populateModel(c);
                        Category category = ReaderContract.CategoryTable.populateModel(c);

                        results.add(new JoinedTransaction(transaction, card, userCategory, category));
                        c.moveToNext();
                    }
                }
            }
        } catch (SQLiteException e) {
            e.printStackTrace();
        } finally {
            if(c != null) {
                c.close();
            }
        }

        return results;
    }

    public ArrayList<JoinedTransaction> loadJoinedTransactions(TransactionByIdsRequest transactionRequest) {
        String query = QueryGenerator.joinedTransaction(transactionRequest);

        ArrayList<JoinedTransaction> results = new ArrayList<>();
        Cursor c = null;

        try {
            c = db.runQuery(query);
            if(c != null) {
                if(c.moveToFirst()) {
                    while (!c.isAfterLast()) {
                        Transaction transaction = ReaderContract.TransactionsTable.populateModel(c);
                        UserCategory userCategory = ReaderContract.UserCategoryTable.populateModel(c);
                        Card card = ReaderContract.CardTable.populateModel(c);
                        Category category = ReaderContract.CategoryTable.populateModel(c);

                        results.add(new JoinedTransaction(transaction, card, userCategory, category));
                        c.moveToNext();
                    }
                }
            }
        } catch (SQLiteException e) {
            e.printStackTrace();
        } finally {
            if(c != null) {
                c.close();
            }
        }

        return results;
    }

    public ArrayList<JoinedTransaction> loadNotSyncedTransactions() {
        String query = QueryGenerator.notSyncedTransactions();

        ArrayList<JoinedTransaction> results = new ArrayList<>();
        Cursor c = null;

        try {
            c = db.runQuery(query);
            if(c != null) {
                if(c.moveToFirst()) {
                    while (!c.isAfterLast()) {
                        Transaction transaction = ReaderContract.TransactionsTable.populateModel(c);
                        UserCategory userCategory = ReaderContract.UserCategoryTable.populateModel(c);
                        Card card = ReaderContract.CardTable.populateModel(c);
                        Category category = ReaderContract.CategoryTable.populateModel(c);

                        results.add(new JoinedTransaction(transaction, card, userCategory, category));
                        c.moveToNext();
                    }
                }
            }
        } catch (SQLiteException e) {
            e.printStackTrace();
        } finally {
            if(c != null) {
                c.close();
            }
        }

        return results;
    }

    public ArrayList<JoinedTransaction> loadFailedSearchTransactions() {
        String query = QueryGenerator.failedSearchTransaction();

        ArrayList<JoinedTransaction> results = new ArrayList<>();
        Cursor c = null;

        try {
            c = db.runQuery(query);
            if(c != null) {
                if(c.moveToFirst()) {
                    while (!c.isAfterLast()) {
                        Transaction transaction = ReaderContract.TransactionsTable.populateModel(c);
                        Card card = ReaderContract.CardTable.populateModel(c);

                        results.add(new JoinedTransaction(transaction, card, null, null));
                        c.moveToNext();
                    }
                }
            }
        } catch (SQLiteException e) {
            e.printStackTrace();
        } finally {
            if(c != null) {
                c.close();
            }
        }

        return results;
    }

    /**
     * 속도 이슈 있을경우 벌크 로 진행
     * @param transaction 파싱된 내역
     */
    public void mergeTransaction(com.tenqube.visual_third.model.parser.Transaction transaction) {

        try {
            ContentValues values = new ContentValues();
            values.put(COLUMN_IDENTIFIER, transaction.getParsedTransaction().identifier);
            values.put(COLUMN_CARD_ID, transaction.getCardId());

            values.put(COLUMN_USER_CATEGORY_ID, transaction.getUserCateId());


            values.put(COLUMN_CATEGORY_CODE, transaction.getCategoryCode());
            values.put(COLUMN_SPENT_DATE, transaction.getParsedTransaction().spentDate);
            values.put(COLUMN_FINISH_DATE, transaction.getParsedTransaction().finishDate);

            values.put(COLUMN_SPENT_LATITUDE, transaction.getParsedTransaction().spentLatitude);
            values.put(COLUMN_SPENT_LONGITUDE, transaction.getParsedTransaction().spentLongitude);

            values.put(COLUMN_SPENT_MONEY, transaction.getSpentMoney());
            values.put(COLUMN_ORI_SPENT_MONEY, transaction.getParsedTransaction().spentMoney);

            values.put(COLUMN_INSTALLMENT_COUNT, transaction.getParsedTransaction().installmentCount);

            values.put(COLUMN_KEYWORD, transaction.getParsedTransaction().keyword);
            values.put(COLUMN_SEARCH_KEYWORD, transaction.getParsedTransaction().keyword);

            values.put(ReaderContract.TransactionsTable.COLUMN_REPEAT_TYPE, 0);
            values.put(COLUMN_CURRENCY, transaction.getParsedTransaction().currency);

            values.put(COLUMN_DW_TYPE, transaction.getParsedTransaction().dwType);

            values.put(COLUMN_SENDER, transaction.getParsedTransaction().sender);
            values.put(COLUMN_FULL_SMS, transaction.getParsedTransaction().fullSms);
            values.put(COLUMN_SMS_DATE, transaction.getParsedTransaction().smsDate);
            values.put(COLUMN_SMS_TYPE, transaction.getParsedTransaction().smsType);
            values.put(COLUMN_REG_ID, transaction.getParsedTransaction().regId);

            values.put(COLUMN_IS_OFFSET, transaction.getParsedTransaction().isOffset);

            values.put(COLUMN_MEMO, transaction.getParsedTransaction().memo);

            db.insertOnReplace(ReaderContract.TransactionsTable.TABLE_NAME, values);
        } catch (Exception e) {
            e.printStackTrace();
        }

    }

    public void updateSyncedTransactions(ArrayList<String> ids) {
        try {
            ContentValues values = new ContentValues();
            values.put(COLUMN_IS_SYNCED, 1);
            // 값 추가하기
            db.update(ReaderContract.TransactionsTable.TABLE_NAME, values, ReaderContract.TransactionsTable.COLUMN_IDENTIFIER +
                    " IN (" + TextUtils.join(",", ids) + ")", null);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void updateRetryTransactions(ArrayList<String> ids, boolean shouldRetry) {
        try {
            ContentValues values = new ContentValues();
            values.put(COLUMN_SHOULD_RETRY_SEARCH, shouldRetry ? 1 : 0);

            if(shouldRetry)
                values.put(COLUMN_IS_SYNCED, 0);

            // 값 추가하기
            db.update(ReaderContract.TransactionsTable.TABLE_NAME, values, ReaderContract.TransactionsTable.COLUMN_IDENTIFIER +
                    " IN (" + TextUtils.join(",", ids) + ")" , null);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     *
     * @param updateTran 업데이트 할 내역
     */
    public void updateTransaction(UpdateTransactionRequest updateTran) {

        try {
            ContentValues values = new ContentValues();
            values.put(COLUMN_USER_CATEGORY_ID, updateTran.getCateConfigId());
            values.put(COLUMN_KEYWORD, updateTran.getKeyword());
            values.put(COLUMN_SPENT_MONEY, updateTran.getAmount());
            values.put(COLUMN_CARD_ID, updateTran.getCardId());
            values.put(COLUMN_INSTALLMENT_COUNT, updateTran.getInstallmentCnt());
            values.put(COLUMN_SPENT_DATE, updateTran.getDate());
            values.put(COLUMN_MEMO, updateTran.getMemo());
            values.put(COLUMN_DW_TYPE, updateTran.isExpense() ? tenqube.parser.constants.Constants.DWType.WITHDRAW.ordinal() :
                    tenqube.parser.constants.Constants.DWType.DEPOSIT.ordinal());
            values.put(COLUMN_IS_UPDATE_ALL, updateTran.isAll() ? 1 : 0);
            values.put(COLUMN_IS_SYNCED, 0);
            String categoryCode = String.valueOf(updateTran.getlCode()) + String.valueOf(updateTran.getmCode()) + "10";
            values.put(COLUMN_CATEGORY_CODE, categoryCode);


            // 값 추가하기
            long row = db.update(ReaderContract.TransactionsTable.TABLE_NAME, values, ReaderContract.TransactionsTable.COLUMN_TRAN_ID + "= " + updateTran.getTranId(), null);
            Log.i("tag", row + "");
        } catch (Exception e) {
            e.printStackTrace();
        }

    }


    public void updateTransaction(SearchCompanyResponse.TranCompany tranCompany, int userCateId) {

        try {
            ContentValues values = new ContentValues();
            values.put(COLUMN_USER_CATEGORY_ID, userCateId);
            values.put(COLUMN_CATEGORY_CODE, tranCompany.getCategory().getCode());
            values.put(COLUMN_FRANCHISE, tranCompany.getCompany().getName());
            values.put(COLUMN_COMPANY_ID, tranCompany.getCompany().getId());
            values.put(COLUMN_CLASS_CODE, tranCompany.getClassCode());
            values.put(COLUMN_SEARCH_KEYWORD, tranCompany.getKeyword().getSearch());
            values.put(COLUMN_KEYWORD, tranCompany.getKeyword().getOri());

            // 값 추가하기
            db.update(ReaderContract.TransactionsTable.TABLE_NAME, values, COLUMN_IDENTIFIER + "= ?", new String[]{tranCompany.getIdentifier() + ""});
        } catch (Exception e) {
            e.printStackTrace();
        }

    }

    public void updateTransactionByKeyword(SearchCompanyResponse.TranCompany tranCompany, int userCateId) {
        try {
            ContentValues values = new ContentValues();
            values.put(COLUMN_USER_CATEGORY_ID, userCateId);
            values.put(COLUMN_CATEGORY_CODE, tranCompany.getCategory().getCode());
            values.put(COLUMN_FRANCHISE, tranCompany.getCompany().getName());
            values.put(COLUMN_COMPANY_ID, tranCompany.getCompany().getId());

            if(!TextUtils.isEmpty(tranCompany.getClassCode()))
                values.put(COLUMN_CLASS_CODE, tranCompany.getClassCode());

            values.put(COLUMN_SEARCH_KEYWORD, tranCompany.getKeyword().getSearch());
            values.put(COLUMN_KEYWORD, tranCompany.getKeyword().getOri());

            // 값 추가하기
            db.update(ReaderContract.TransactionsTable.TABLE_NAME, values, COLUMN_KEYWORD + "= ?", new String[]{tranCompany.getKeyword().getOri()});
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


}
