package com.tenqube.visual_third;

import android.content.Context;
import android.support.annotation.NonNull;
import android.text.TextUtils;

import com.tenqube.visual_third.api.VisualApi;
import com.tenqube.visual_third.exception.AuthException;
import com.tenqube.visual_third.exception.ParameterException;
import com.tenqube.visual_third.manager.AnswerManager;
import com.tenqube.visual_third.manager.VisualAlarmManager;
import com.tenqube.visual_third.model.api.SignUpRequest;
import com.tenqube.visual_third.model.api.SignUpResponse;
import com.tenqube.visual_third.model.js.LogRequest;
import com.tenqube.visual_third.repository.VisualRepository;
import com.tenqube.visual_third.ui.OnResultListener;
import com.tenqube.visual_third.ui.VisualWebActivity;

import tenqube.parser.core.ParserService;

import static com.tenqube.visual_third.Constants.DEV;
import static com.tenqube.visual_third.Constants.PROD;
import static com.tenqube.visual_third.manager.AnswerManager.initFabric;
import static com.tenqube.visual_third.util.Validator.notNull;
import static com.tenqube.visual_third.util.Validator.notZero;
import static tenqube.parser.core.ParserService.mIsDebug;
import static tenqube.parser.util.LogUtil.LOGI;

public class VisualServiceImpl implements VisualService {

    public static final String TAG = VisualServiceImpl.class.getSimpleName();

    private Context context;
    private VisualApi api;

    private ParserService parserService;
    private VisualRepository repository;
    private VisualAlarmManager alarmManager;

    private boolean isDoingSignup;
    private boolean isDoingStartVisual;

    public VisualServiceImpl(@NonNull Context context, @NonNull String apiKey, @NonNull String qualifier) throws ParameterException {

        if(context == null) throw new ParameterException("context is null");

        if(TextUtils.isEmpty(apiKey)) throw new ParameterException("apiKey is null");

        if(TextUtils.isEmpty(qualifier) || !(Constants.DEV.equals(qualifier) || Constants.PROD.equals(qualifier)))
            throw new ParameterException("check qualifier value (" + DEV + ", " + PROD + ")");

        this.context = context;
        initFabric(context);

        repository = VisualRepository.getInstance(context);
        repository.saveSDKInfo(apiKey, qualifier);

        api = VisualApi.getInstance(context);

        parserService = ParserService.getInstance(context);

        alarmManager= VisualAlarmManager.getInstance(context);
        alarmManager.disableDailyAlarms();
    }

    @Override
    public void signUp(@NonNull final String custId, @NonNull final OnResultListener onResultListener) throws ParameterException {
        LOGI(TAG, "가입", mIsDebug);
        if(isDoingSignup) return;
        isDoingSignup = true;
        final SignUpRequest signUpRequest = new SignUpRequest(custId);
        notNull(onResultListener);

        if(!repository.shouldSignUp()) { //가입 여부 확인
            onResultListener.onResult(Constants.SignupResult.ALREADY_JOINED, "already joined");
            isDoingSignup = false;
            return;
        }

        // log 전송
        AnswerManager.onKeyMetric(new LogRequest("signUp"));
        new Thread(new Runnable() {
            @Override
            public void run() {
                SignUpResponse response = api.signUp(signUpRequest);

                if (response == null || response.getResults() == null) {
                    onResultListener.onResult(Constants.SignupResult.FAIL, "network error");
                } else {
                    repository.saveServerInfo(response);
                    onResultListener.onResult(Constants.SignupResult.SUCCESS, "success");
                }

                isDoingSignup = false;
            }
        }).start();
    }

    @Override
    public void startVisual(@NonNull String custId, final @NonNull OnResultListener resultListener) throws ParameterException {
        LOGI(TAG, "웹 가계부 호출", mIsDebug);
        if(isDoingStartVisual) return;

        isDoingStartVisual = true;

        notNull(resultListener);

        signUp(custId, new OnResultListener() {
            @Override
            public void onResult(int signUpResult, String msg) {

                switch (signUpResult) {
                    case Constants.SignupResult.ALREADY_JOINED:
                    case Constants.SignupResult.SUCCESS:
                        try {
                            AnswerManager.onKeyMetric(new LogRequest("startVisual"));
                            VisualWebActivity.startActivity(context, "");
                        } catch (AuthException e) {
                            resultListener.onResult(Constants.SignupResult.FAIL, e.toString());
                            return;
                        } finally {
                            isDoingStartVisual = false;
                        }
                        break;
                }

                resultListener.onResult(signUpResult, msg);
                isDoingStartVisual = false;
            }
        });
    }

    @Override
    public boolean isActiveTranPopup() {
        AnswerManager.onKeyMetric(new LogRequest("isActiveTranPopup"));
        return repository.shouldShowTranPopup();
    }

    @Override
    public void setTranPopup(boolean isActive) {
        AnswerManager.onKeyMetric(new LogRequest("setTranPopup"));
        repository.setTranPopup(isActive);
    }

    @Override
    public void setLogger(boolean isActive) {
        LOGI(TAG, "setLogger", mIsDebug);

        if(isActive() && parserService != null) {
            parserService.setDebugMode(isActive);
        }
    }

    @Override
    public void initSDK() {
        LOGI(TAG, "initSDK", mIsDebug);

        if(isActive() && parserService != null) {
            parserService.initDb();
        }
    }

    @Override
    public void setEnabled(boolean enabled) {
        repository.setEnabled(enabled);

        if(!enabled) {
            repository.setActiveNoti(Constants.NotificationType.MONTHLY, false);
            repository.setActiveNoti(Constants.NotificationType.WEEKLY, false);
            repository.setActiveNoti(Constants.NotificationType.DAILY, false);
        }
        alarmManager.setAlarms();
    }

    @Override
    public void settingNotification(int smallIcon, @NonNull String channel, int color) throws ParameterException {
        notZero(smallIcon);
        notNull(channel);
        repository.settingNotification(smallIcon, channel, color);
    }

    @Override
    public void setMonthlyNoti(boolean isActive) {
        repository.setActiveNoti(Constants.NotificationType.MONTHLY, isActive);
        alarmManager.setAlarms();
    }

    @Override
    public boolean isActiveMonthlyNoti() {
        return repository.isActiveNoti(Constants.NotificationType.MONTHLY);
    }

    @Override
    public void setWeeklyNoti(boolean isActive) {
        repository.setActiveNoti(Constants.NotificationType.WEEKLY, isActive);
        alarmManager.setAlarms();

    }

    @Override
    public boolean isActiveWeeklyNoti() {
        return repository.isActiveNoti(Constants.NotificationType.WEEKLY);
    }

    @Override
    public void setDailyNoti(boolean isActive) {
        repository.setActiveNoti(Constants.NotificationType.DAILY, isActive);
        alarmManager.setAlarms();
    }

    @Override
    public boolean isActiveDailyNoti() {
        return repository.isActiveNoti(Constants.NotificationType.DAILY);
    }

    @Override
    public void setReportTest(String type, int second) {
        VisualAlarmManager.getInstance(context).setReportTest(type, second);
    }


    private boolean isActive() {
        return context != null;
    }
}
