package com.tenqube.visual_third.repository;

import androidx.annotation.Nullable;

import com.tenqube.visual_third.Callback;
import com.tenqube.visual_third.Constants;
import com.tenqube.visual_third.api.resource.ResourceApi;
import com.tenqube.visual_third.api.resource.ResourceService;
import com.tenqube.visual_third.db.dao.CategoryDao;
import com.tenqube.visual_third.db.dao.ContentsDao;
import com.tenqube.visual_third.db.dao.NotificationAppDao;
import com.tenqube.visual_third.db.dao.NotificationDao;
import com.tenqube.visual_third.entity.VisualNotification;
import com.tenqube.visual_third.manager.PrefManager;
import com.tenqube.visual_third.manager.SignatureManager;
import com.tenqube.visual_third.model.api.AnalysisResponse;
import com.tenqube.visual_third.model.api.NotificationPkgResponse;
import com.tenqube.visual_third.model.api.NotificationResponse;
import com.tenqube.visual_third.model.api.ParsingRuleKey;
import com.tenqube.visual_third.model.api.SyncCategoryResponse;
import com.tenqube.visual_third.model.api.VersionResponse;
import com.tenqube.visual_third.util.AppExecutors;
import com.tenqube.visual_third.util.VisualInjection;

import java.util.ArrayList;

import tenqube.parser.core.ParserService;
import tenqube.parser.model.ParsingRule;

import static com.tenqube.visual_third.Constants.ANALYSIS_VERSION;
import static com.tenqube.visual_third.Constants.CATEGORY_VERSION;
import static com.tenqube.visual_third.Constants.NOTIFICATION_VERSION;
import static com.tenqube.visual_third.Constants.RULE_VERSION;
import static com.tenqube.visual_third.manager.PrefManager.PARSING_RULE_VERSION;
import static com.tenqube.visual_third.manager.PrefManager.PKG_VERSION;
import static com.tenqube.visual_third.util.Utils.isEmpty;

public class ResourceRepository {

    @Nullable
    private ResourceService resourceService;

    private final PrefManager prefManager;
    private final ParserService parserService;
    private final AppExecutors appExecutors;

    private final NotificationAppDao notificationAppDao;
    private final CategoryDao categoryDao;
    private final NotificationDao notificationDao;
    private final ContentsDao contentsDao;
    private final SignatureManager signatureManager;

    private static ResourceRepository mInstance = null;

    public static ResourceRepository getInstance(PrefManager prefManager,
                                                 ParserService parserService,
                                                 AppExecutors appExecutors,
                                                 NotificationAppDao notificationAppDao,
                                                 CategoryDao categoryDao,
                                                 NotificationDao notificationDao,
                                                 ContentsDao contentsDao,
                                                 SignatureManager signatureManager

                                                 ) {

        synchronized (ResourceRepository.class) {
            if (mInstance == null) {
                mInstance = new ResourceRepository(prefManager,
                        parserService,
                        appExecutors,
                        notificationAppDao,
                        categoryDao,
                        notificationDao,
                        contentsDao,
                        signatureManager);
            }
        }
        return mInstance;
    }

    private ResourceRepository(PrefManager prefManager,
                               ParserService parserService,
                               AppExecutors appExecutors,
                               NotificationAppDao notificationAppDao,
                               CategoryDao categoryDao,
                               NotificationDao notificationDao,
                               ContentsDao contentsDao,
                               SignatureManager signatureManager
    ) {
        this.prefManager = prefManager;
        this.parserService = parserService;
        this.appExecutors = appExecutors;
        this.notificationAppDao = notificationAppDao;
        this.categoryDao = categoryDao;
        this.notificationDao = notificationDao;
        this.contentsDao = contentsDao;
        this.signatureManager = signatureManager;
    }

    private void createResource() {

        ResourceApi resourceApi = VisualInjection.provideResourceApi(prefManager);
        resourceService = VisualInjection.provideResourceService(resourceApi, signatureManager);

    }

    public void sync() {

        if(resourceService == null) {
            createResource();
        }

        appExecutors.networkIO().execute(new Runnable() {
            @Override
            public void run() {
                VersionResponse versionResponse = resourceService.syncVersion();
                if(versionResponse != null) {
                    syncCategory(versionResponse);
                    syncAnalysis(versionResponse);
                    syncNotification(versionResponse);
                    syncNotificationApps(versionResponse);
                    syncParsingRule(versionResponse);
                }
            }
        });
    }

    private void syncCategory(VersionResponse versionResponse) {

        int clientVersion = prefManager.loadIntValue(PrefManager.CATEGORY_VERSION, CATEGORY_VERSION);
        if(versionResponse != null && versionResponse.getCategory() > clientVersion) {

            if(resourceService != null) {
                SyncCategoryResponse response = resourceService.syncCategory(clientVersion, versionResponse.getCategory());
                if(response != null) {
                    prefManager.saveIntValue(PrefManager.CATEGORY_VERSION, versionResponse.getCategory());
                    if (!response.getCategories().isEmpty())
                        mergeCategory(response.getCategories());
                }
            }


        }
    }

    private void mergeCategory(ArrayList<SyncCategoryResponse.Category> categories) {

        try {
            for(SyncCategoryResponse.Category category : categories) {
                categoryDao.mergeCategory(category);
            }
        } catch (Exception e) {

        }

    }

    private void syncAnalysis(VersionResponse versionResponse) {
        int clientVersion = prefManager.loadIntValue(PrefManager.ANALYSIS_VERSION, ANALYSIS_VERSION);
        if(versionResponse != null && versionResponse.getAnalysis() > clientVersion) {

            if(resourceService != null) {
                AnalysisResponse response = resourceService.syncAnalysis(clientVersion, versionResponse.getAnalysis());
                if(response != null) {
                    prefManager.saveIntValue(PrefManager.ANALYSIS_VERSION, versionResponse.getAnalysis());
                    if(!isEmpty(response.getContents()))
                        mergeAnalysis(response.getContents());
                }
            }

        }
    }

    private void mergeAnalysis(ArrayList<AnalysisResponse.Content> contents) {
        for(AnalysisResponse.Content content : contents) {

            if(content.getEnable() == 0) {
                this.contentsDao.deleteContent(content.getId());
            } else {
                this.contentsDao.mergeContent(content);

                if(!isEmpty(content.getConditions())) {
                    for(AnalysisResponse.Condition condition : content.getConditions()) {
                        if(condition.getEnable() == 0) {
                            this.contentsDao.deleteCondition(condition.getId());
                        } else {
                            this.contentsDao.mergeCondition(condition);
                        }
                    }
                }
            }

        }
    }

    private void syncParsingRule(VersionResponse versionResponse) {

        int clientVersion = prefManager.loadIntValue(PARSING_RULE_VERSION, RULE_VERSION);
        if(versionResponse != null && versionResponse.getParsingRule() > clientVersion) {

            if(resourceService != null) {
                ParsingRule parsingRule = resourceService.syncParsingRule(clientVersion, versionResponse.getParsingRule());
                if(parsingRule != null) {
                    parserService.syncParsingRule(parsingRule);
                    prefManager.saveIntValue(PARSING_RULE_VERSION, parsingRule.ruleVersion);
                }
            }

        }
    }

    public void syncNotification() {
        if(resourceService == null) {
            createResource();
        }

        if(resourceService == null) {
            return;
        }
        appExecutors.networkIO().execute(new Runnable() {
            @Override
            public void run() {
                VersionResponse versionResponse = resourceService.syncVersion();

                int clientVersion = prefManager.loadIntValue(PrefManager.NOTIFICATION_VERSION, NOTIFICATION_VERSION);
                if(versionResponse != null && versionResponse.getNotification() > clientVersion) {
                    NotificationResponse response = resourceService.syncNotification();
                    if(response != null) {

                        prefManager.saveIntValue(PrefManager.NOTIFICATION_VERSION, versionResponse.getNotification());
                        prefManager.saveIntValue(PrefManager.NOTIFICATION_DIVIDER, response.getDivider());

                        if (response.getNotifications() != null && !response.getNotifications().isEmpty())
                            mergeNotifications(response.getNotifications());
                    }
                }
            }
        });
    }

    private void syncNotification(VersionResponse versionResponse) {

        int clientVersion = prefManager.loadIntValue(PrefManager.NOTIFICATION_VERSION, NOTIFICATION_VERSION);
        if(versionResponse != null && versionResponse.getNotification() > clientVersion) {

            if(resourceService != null) {
                NotificationResponse response = resourceService.syncNotification();
                if(response != null) {

                    prefManager.saveIntValue(PrefManager.NOTIFICATION_VERSION, versionResponse.getNotification());
                    prefManager.saveIntValue(PrefManager.NOTIFICATION_DIVIDER, response.getDivider());

                    if (response.getNotifications() != null && !response.getNotifications().isEmpty())
                        mergeNotifications(response.getNotifications());
                }
            }

        }
    }

    private void mergeNotifications(ArrayList<NotificationResponse.Notification> notifications) {

        try {
            ArrayList<VisualNotification> values = new ArrayList<>();
            for(NotificationResponse.Notification notification : notifications) {
                values.add(notification.toVisualNotification());
            }
            if(!values.isEmpty()) notificationDao.mergeNotifications(values);

        } catch (Exception e) {

        }

    }

    private void syncNotificationApps(VersionResponse versionResponse) {
        int clientVersion = prefManager.loadIntValue(PKG_VERSION, Constants.PKG_VERSION);
        if(versionResponse != null && versionResponse.getPkgVersion() > clientVersion) {

            if(resourceService != null) {
                NotificationPkgResponse response = resourceService.syncNotificationApps();
                if(response != null) {
                    notificationAppDao.mergeAll(response.getApps());
                    prefManager.saveIntValue(PrefManager.PKG_VERSION, versionResponse.getPkgVersion());
                }
            }

        }
    }

    public ParsingRuleKey syncParsingRuleKey() {
        if(resourceService == null) {
            createResource();
        }
        if(resourceService == null) {
            return null;
        }
        return resourceService.syncParsingRuleKey();
    }

    public void syncParsingRule(final Callback<ParsingRule> callback) {

        if(resourceService == null) {
            createResource();
        }

        appExecutors.networkIO().execute(new Runnable() {
            @Override
            public void run() {

                final VersionResponse versionResponse = resourceService.syncVersion();
                int clientVersion = prefManager.loadIntValue(PrefManager.PARSING_RULE_VERSION, RULE_VERSION);
                ParsingRule parsingRule = null;
                if(versionResponse != null) {
                    parsingRule = resourceService.syncParsingRule(clientVersion, versionResponse.getParsingRule());
                }

                final ParsingRule finalParsingRule = parsingRule;
                appExecutors.mainThread().execute(new Runnable() {
                    @Override
                    public void run() {
                        callback.onDataLoaded(finalParsingRule);
                    }
                });
            }
        });
    }
}
