package com.tenqube.visual_third.repository;

import android.content.Context;
import android.content.Intent;
import android.database.Cursor;
import androidx.annotation.Nullable;
import androidx.localbroadcastmanager.content.LocalBroadcastManager;
import android.text.TextUtils;

import com.tenqube.visual_third.BuildConfig;
import com.tenqube.visual_third.Callback;
import com.tenqube.visual_third.VisualCallback;
import com.tenqube.visual_third.VisualUserInfo;
import com.tenqube.visual_third.api.auth.AuthApi;
import com.tenqube.visual_third.api.auth.AuthService;
import com.tenqube.visual_third.api.auth.AuthServiceImpl;
import com.tenqube.visual_third.api.search.SearchApi;
import com.tenqube.visual_third.api.search.SearchService;
import com.tenqube.visual_third.api.search.SearchServiceImpl;
import com.tenqube.visual_third.api.visual.VisualApi;
import com.tenqube.visual_third.api.visual.VisualApiService;
import com.tenqube.visual_third.api.visual.VisualApiServiceImpl;
import com.tenqube.visual_third.db.dao.AdvertisementDao;
import com.tenqube.visual_third.db.dao.CardDao;
import com.tenqube.visual_third.db.dao.CategoryDao;
import com.tenqube.visual_third.db.dao.CurrencyDao;
import com.tenqube.visual_third.db.dao.NotificationAppDao;
import com.tenqube.visual_third.db.dao.NotificationDao;
import com.tenqube.visual_third.db.dao.TransactionDao;
import com.tenqube.visual_third.db.dao.UserCategoryDao;
import com.tenqube.visual_third.entity.Advertisement;
import com.tenqube.visual_third.entity.Card;
import com.tenqube.visual_third.entity.Category;
import com.tenqube.visual_third.entity.Currency;
import com.tenqube.visual_third.entity.JoinedTransaction;
import com.tenqube.visual_third.entity.NotificationApp;
import com.tenqube.visual_third.entity.Transaction;
import com.tenqube.visual_third.entity.UserCategory;
import com.tenqube.visual_third.entity.VisualNotification;
import com.tenqube.visual_third.exception.ParameterException;
import com.tenqube.visual_third.manager.AnswerManager;
import com.tenqube.visual_third.manager.PrefManager;
import com.tenqube.visual_third.manager.SecretKeyManager;
import com.tenqube.visual_third.model.api.AdInfoResponse;
import com.tenqube.visual_third.model.api.AdResponse;
import com.tenqube.visual_third.model.api.CurrencyResponse;
import com.tenqube.visual_third.model.api.SearchCompanyRequest;
import com.tenqube.visual_third.model.api.SearchCompanyResponse;
import com.tenqube.visual_third.model.api.UserInfoRequest;
import com.tenqube.visual_third.model.api.UserRequest;
import com.tenqube.visual_third.model.js.Cards;
import com.tenqube.visual_third.model.js.CategoryInfo;
import com.tenqube.visual_third.model.js.InsertTransactionRequest;
import com.tenqube.visual_third.model.js.LogRequest;
import com.tenqube.visual_third.model.js.TransactionByIdsRequest;
import com.tenqube.visual_third.model.js.TransactionRequest;
import com.tenqube.visual_third.model.js.Transactions;
import com.tenqube.visual_third.model.js.UpdateTransactionRequest;
import com.tenqube.visual_third.model.parser.SyncTransaction;
import com.tenqube.visual_third.ui.OnResultListener;
import com.tenqube.visual_third.util.AppExecutors;
import com.tenqube.visual_third.util.Utils;
import com.tenqube.visual_third.util.VisualInjection;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.UUID;

import retrofit2.Call;
import retrofit2.Response;
import tenqube.parser.OnNetworkResultListener;
import tenqube.parser.constants.Constants;

import static com.tenqube.visual_third.Constants.DEFAULT_AD_RATIO;
import static com.tenqube.visual_third.Constants.NOTIFICATION_DIVIDER;
import static com.tenqube.visual_third.manager.PrefManager.ACCESS_TOKEN;
import static com.tenqube.visual_third.manager.PrefManager.NOTI_CHANNEL;
import static com.tenqube.visual_third.manager.PrefManager.NOTI_COLOR;
import static com.tenqube.visual_third.manager.PrefManager.NOTI_ICON_RES;
import static com.tenqube.visual_third.manager.PrefManager.REFRESH_TOKEN;
import static com.tenqube.visual_third.manager.PrefManager.SDK_ENABLED;
import static com.tenqube.visual_third.manager.PrefManager.START_DAY;
import static com.tenqube.visual_third.manager.PrefManager.USER_ID;
import static com.tenqube.visual_third.ui.VisualWebActivity.ACTION_BROADCAST_TRANSACTION;
import static com.tenqube.visual_third.util.Mapper.toAnalysisTran;
import static com.tenqube.visual_third.util.Mapper.toApiTransactions;
import static com.tenqube.visual_third.util.Mapper.toJsCards;
import static com.tenqube.visual_third.util.Mapper.toJsCategories;
import static com.tenqube.visual_third.util.Mapper.toJsTransactions;
import static com.tenqube.visual_third.util.Mapper.toJsUserCategories;
import static com.tenqube.visual_third.util.Utils.getAdId;
import static com.tenqube.visual_third.util.Utils.isEmpty;

public class VisualRepository {

    private Context context;
    private static VisualRepository mInstance;

    private final PrefManager prefManager;
    private final CardDao cardDao;
    private final CategoryDao categoryDao;
    private final TransactionDao transactionDao;
    private final UserCategoryDao userCategoryDao;
    private final CurrencyDao currencyDao;
    private final NotificationDao notificationDao;
    private final NotificationAppDao notificationAppDao;

    private final SecretKeyManager secretKeyManager;

    @Nullable
    private VisualCallback visualCallback;


    @Nullable
    private AuthService authService;

    @Nullable
    private VisualApiService visualService;

    @Nullable
    private SearchService searchService; //가입 후에 생성하기

    private final AdvertisementDao advertisementDao;
    private AppExecutors appExecutors;

    private boolean isVisualActive;

    public static VisualRepository getInstance(Context context,
                                               PrefManager prefManager,
                                               CardDao cardDao,
                                               CategoryDao categoryDao,
                                               TransactionDao transactionDao,
                                               UserCategoryDao userCategoryDao,
                                               CurrencyDao currencyDao,
                                               NotificationDao notificationDao,
                                               AdvertisementDao advertisementDao,
                                               NotificationAppDao notificationAppDao,
                                               AppExecutors appExecutor,
                                               SecretKeyManager secretKeyManager
                                               ){
        synchronized (VisualRepository.class) {
            if(mInstance == null){
                mInstance = new VisualRepository(context,
                        prefManager,
                        cardDao,
                        categoryDao,
                        transactionDao,
                        userCategoryDao,
                        currencyDao,
                        notificationDao,
                        advertisementDao,
                        notificationAppDao,
                        appExecutor,
                        secretKeyManager);
            }
        }
        return mInstance;
    }

    private VisualRepository(Context context, PrefManager prefManager,
                             CardDao cardDao,
                             CategoryDao categoryDao,
                             TransactionDao transactionDao,
                             UserCategoryDao userCategoryDao,
                             CurrencyDao currencyDao,
                             NotificationDao notificationDao,
                             AdvertisementDao advertisementDao,
                             NotificationAppDao notificationAppDao,
                             AppExecutors appExecutors,
                             SecretKeyManager secretKeyManager) {
        this.context = context;
        this.prefManager = prefManager;
        this.cardDao = cardDao;
        this.categoryDao = categoryDao;
        this.transactionDao = transactionDao;
        this.userCategoryDao = userCategoryDao;
        this.currencyDao = currencyDao;
        this.notificationDao = notificationDao;
        this.advertisementDao = advertisementDao;
        this.notificationAppDao = notificationAppDao;
        this.appExecutors = appExecutors;
        this.secretKeyManager = secretKeyManager;
    }

    public String getSyncStatus() {
        return prefManager.loadStringValue(PrefManager.RESTORE_STATUS, Constants.NONE);
    }

    public void setVisualCallback(VisualCallback callback) {
        this.visualCallback = callback;
    }

    public void setVisualActive(boolean isVisualActive) {
        this.isVisualActive = isVisualActive;
    }

    public void syncAd(final Callback<Boolean> callback) {
        appExecutors.networkIO().execute(new Runnable() {
            @Override
            public void run() {

                boolean shouldLoad = true;
                final AdInfoResponse versionResponse = visualService == null ? null : visualService.getAdInfo();
                if(versionResponse != null && !versionResponse.shouldShowAd()) {

                    shouldLoad = false;

                } else {
                    int version = prefManager.loadIntValue(PrefManager.AD_VERSION, 0);
                    if(versionResponse != null && versionResponse.getVersion() > version) {
                        final AdResponse adResponse = visualService == null ? null : visualService.getAds(versionResponse.getVersion());
                        if(adResponse != null)
                            mergeAdvertisement(adResponse, versionResponse.getVersion());

                    }
                }

                final boolean finalShouldLoad = shouldLoad;
                appExecutors.mainThread().execute(new Runnable() {
                    @Override
                    public void run() {
                        callback.onDataLoaded(finalShouldLoad);
                    }
                });
            }
        });

    }

    public double getCurrencyRate(String from, String to, double rate) {

        CurrencyResponse currencyResponse =  visualService == null ? null : visualService.getCurrencyRate(from, to);
        if(currencyResponse != null) {
            currencyDao.mergeCurrency(new Currency(0, from, to, rate, Utils.getStringDateAsYYYYMMddHHmmss(Calendar.getInstance())));
            return currencyResponse.getRate();
        }
        return rate;
    }

    public void onTransactionReceived(int tranId) {
        Intent intent = new Intent(ACTION_BROADCAST_TRANSACTION);
        intent.putExtra("tranId", tranId);
        LocalBroadcastManager.getInstance(context).sendBroadcast(intent);

        appExecutors.mainThread().execute(new Runnable() {
            @Override
            public void run() {

                if(visualCallback != null) {
                    visualCallback.onTransactionReceived();
                }
            }
        });


    }

    public void syncTransactions(final OnNetworkResultListener callback) {
        if(visualService == null) {
            if(callback != null) callback.onResult(false);
            return;
        }

        final com.tenqube.visual_third.model.api.TransactionRequest transactionRequest = loadNotSyncedTransactions();
        if(transactionRequest != null && !isEmpty(transactionRequest.getTransactions())) {
            visualService.saveTransactions(transactionRequest, new retrofit2.Callback<Void>() {
                @Override
                public void onResponse(Call<Void> call, Response<Void> response) {

//                    if(response.isSuccessful() || response.code() == 400) {
                        updateSyncedTransactions(transactionRequest);
                        deleteTransactions();
//                    }
                    if(callback != null) callback.onResult(true);
                }

                @Override
                public void onFailure(Call<Void> call, Throwable t) {
                    updateSyncedTransactions(transactionRequest);
                    deleteTransactions();

                    if(callback != null) callback.onResult(true);
                }
            });
        } else {
            if(callback != null) callback.onResult(true);
        }
    }

    public SearchCompanyResponse searchCompany(SearchCompanyRequest request) {
        if(searchService == null) {
            createSearchService();
        }

        return searchService.searchCompany(request);
    }


    public void signUp(final String uid,
                       final VisualUserInfo visualUserInfo,
                       final OnResultListener listener) {
        Utils.LOGD("signUp", "uid: " + uid);

        appExecutors.diskIO().execute(new Runnable() {
            @Override
            public void run() {

                int resultCode;
                String msg;
                try {
                    if(isJoined()) { //가입 여부 확인
                        resultCode = com.tenqube.visual_third.Constants.SignupResult.ALREADY_JOINED;
                        msg = "already joined";
                    } else {

                        String adId = getAdId(context);
                        final UserRequest signUpRequest = new UserRequest(uid, adId);
                        AnswerManager.onKeyMetric(new LogRequest("signUp"));

                        boolean success = authService != null && authService.signUp(signUpRequest);

                        if(success) {
                            saveUserInfo(visualUserInfo);
                            setTranPopup(false);
                            setActiveNoti(com.tenqube.visual_third.Constants.ReportAlarmType.WEEKLY.name().toLowerCase(), true);
                            setActiveNoti(com.tenqube.visual_third.Constants.ReportAlarmType.MONTHLY.name().toLowerCase(), true);
                            resultCode = com.tenqube.visual_third.Constants.SignupResult.SUCCESS;
                            msg = "success";
                        } else {
                            resultCode = com.tenqube.visual_third.Constants.SignupResult.SERVER_ERROR;
                            msg = "server error";
                        }
                    }

                } catch (IOException e) {
                    resultCode = com.tenqube.visual_third.Constants.SignupResult.NETWORK_ERROR;
                    msg = "network error";
                } catch (ParameterException e) {
                    resultCode = com.tenqube.visual_third.Constants.SignupResult.SERVER_ERROR;
                    msg = "parameter error";
                } catch (Exception e) {
                    resultCode = com.tenqube.visual_third.Constants.SignupResult.SERVER_ERROR;
                    msg = e.toString();
                }

                final int finalResultCode = resultCode;
                final String finalMsg = msg;
                appExecutors.mainThread().execute(new Runnable() {
                    @Override
                    public void run() {
                        Utils.LOGD("signUp", "onResult finalResultCode: " + finalResultCode);
                        Utils.LOGD("signUp", "onResult finalMsg: " + finalMsg);

                        listener.onResult(finalResultCode, finalMsg);
                    }
                });
            }
        });
    }

    private void saveUserInfo(final VisualUserInfo visualUserInfo) {
        try {

            UserInfoRequest userInfoRequest = new UserInfoRequest(visualUserInfo.getBirth(), visualUserInfo.getGender().name().toLowerCase());
            visualService.saveUserInfo(userInfoRequest);
        } catch (Exception e) {
        }

    }

    private void createApiService() {
        AuthApi authApi = VisualInjection.provideApiService(AuthApi.class, getUrl(BuildConfig.BASE_URL), null,
                VisualInjection.getAuthInterceptors(prefManager));

        authService = AuthServiceImpl.getInstance(context, authApi, prefManager, secretKeyManager);

        VisualApi visualApi = VisualInjection.provideApiService(VisualApi.class,
                getUrl(BuildConfig.BASE_URL), authService,
                VisualInjection.getVisualInterceptors(prefManager, authService));

        visualService = VisualApiServiceImpl.getInstance(context, visualApi, prefManager);
    }

    private String getUrl(String url) {
        return url + prefManager.loadStringValue(PrefManager.QUALIFIER, "dev") + "/";
    }

    private void createSearchService() {
        SearchApi searchApi = VisualInjection.provideApiService(SearchApi.class,
                prefManager.loadStringValue(PrefManager.SEARCH_URL, ""), authService,
                VisualInjection.getSearchInterceptors(prefManager, secretKeyManager));

        searchService = SearchServiceImpl.getInstance(context, searchApi, prefManager);
    }

    public void initialize(final Callback<Boolean> callback) {
        appExecutors.diskIO().execute(new Runnable() {
            @Override
            public void run() {

                deleteCard();
                insertCash();
                deleteTransactions();
                appExecutors.mainThread().execute(new Runnable() {
                    @Override
                    public void run() {

                        callback.onDataLoaded(true);
                    }
                });
            }
        });
    }

    public void signOut(final Callback<Boolean> callback) {
        appExecutors.diskIO().execute(new Runnable() {
            @Override
            public void run() {

                secretKeyManager.save(ACCESS_TOKEN, "");
                secretKeyManager.save(REFRESH_TOKEN, "");
                prefManager.saveBoolean(PrefManager.TRAN_POPUP, false);
                notificationDao.updateAllNotiEnable(false);

                prefManager.saveBoolean(PrefManager.BULK_EXECUTED, false);

                initialize(new Callback<Boolean>() {
                    @Override
                    public void onDataLoaded(Boolean value) {
                        if(visualCallback != null && isVisualActive) {
                            visualCallback.onSignOut();
                        }
                        callback.onDataLoaded(value);
                    }
                });

            }
        });
    }

    public String getUid() {
        return prefManager.loadStringValue(USER_ID, UUID.randomUUID().toString());
    }

    public int getDivider() {
        return prefManager.loadIntValue(PrefManager.NOTIFICATION_DIVIDER, NOTIFICATION_DIVIDER);
    }

    public boolean isSatisfied(String queryAll) {
        String[] queries = queryAll.split(";");
        boolean isSatisfied = true;
        for(String query : queries) {
            isSatisfied = !isInvalid(query) && advertisementDao.isSatisfied(query);
            if(!isSatisfied) break;
        }
        return isSatisfied;
    }

    private boolean isInvalid(String query) {
        return TextUtils.isEmpty(query) ||
                query.toLowerCase().contains("delete") ||
                query.toLowerCase().contains("update") ||
                query.toLowerCase().contains("insert") ||
                query.toLowerCase().contains("`") ||
                !query.toLowerCase().contains("select");

    }

    public ArrayList<Advertisement> loadAds() {
        return advertisementDao.loadAds();
    }

    private void mergeAdvertisement(AdResponse ad, int version) {
        prefManager.saveIntValue(PrefManager.AD_VERSION, version);

        String ratio = TextUtils.isEmpty(ad.getRatio()) ? DEFAULT_AD_RATIO : ad.getRatio();
        prefManager.saveStringValue(PrefManager.AD_RATIO, ratio);


        if(ad.getAdmob() != null) {
            prefManager.saveStringValue(PrefManager.ADMOB_BG_COLOR, ad.getAdmob().getBgColor());
            prefManager.saveStringValue(PrefManager.ADMOB_TITLE_COLOR, ad.getAdmob().getTitleColor());
            prefManager.saveStringValue(PrefManager.ADMOB_CONTENT_COLOR, ad.getAdmob().getContentColor());
            prefManager.saveStringValue(PrefManager.ADMOB_LABEL_COLOR, ad.getAdmob().getLabelColor());
            prefManager.saveStringValue(PrefManager.ADMOB_LINK_COLOR, ad.getAdmob().getLinkToColor());
        }

        if(!isEmpty(ad.getAds())) {

            for(Advertisement a : ad.getAds()) {
                advertisementDao.merge(a);
            }
        }
    }

    public String getAdmobColor(String key, String defaultColor) {
        return prefManager.loadStringValue(key, defaultColor);
    }

    // provider
    public Cursor getSum(String fromAt, String toAt, int groupBy) {
        return transactionDao.getSum(fromAt, toAt, groupBy);
    }

    public void settingNotification(int smallIcon, String channel, int color) {
        prefManager.saveIntValue(NOTI_ICON_RES, smallIcon);
        prefManager.saveStringValue(NOTI_CHANNEL, channel);
        prefManager.saveIntValue(NOTI_COLOR, color);
    }

    public void setEnabled(boolean enabled) {
        prefManager.saveBoolean(SDK_ENABLED, enabled);
    }

    public void saveSDKInfo(String apiKey, String qualifier, String authority) {
        prefManager.saveStringValue(PrefManager.API_KEY, apiKey);
        prefManager.saveStringValue(PrefManager.QUALIFIER, qualifier);
        prefManager.saveStringValue(PrefManager.AUTHORITY, authority);
        createApiService();
    }

    public double getSum(String fromAt, String toAt) {
        return transactionDao.getSum(fromAt, toAt);
    }

    public boolean isActiveNoti(String name) {
        return notificationDao.isActiveNoti(name);
    }

    public void setNotiAll(boolean isActive) {
        notificationDao.updateAllNotiEnable(isActive);

    }

    public void setActiveNoti(String name, boolean isActive) {
        notificationDao.setActiveNoti(name, isActive);
        onSettingChange(name, isActive);
    }

    private void onSettingChange(String name, boolean isActive) {
        if(!TextUtils.isEmpty(name) && visualCallback != null) {
            com.tenqube.visual_third.Constants.ReportAlarmType type = null;
            if(name.contains("daily")) {
                type = com.tenqube.visual_third.Constants.ReportAlarmType.DAILY;
            } else if(name.contains("weekly")) {
                type = com.tenqube.visual_third.Constants.ReportAlarmType.WEEKLY;
            } else if(name.contains("monthly")) {
                type = com.tenqube.visual_third.Constants.ReportAlarmType.MONTHLY;
            }
        }
    }

    public void updateNotiHour(String name, int hour) {
        notificationDao.updateNotiHour(name, hour);
    }

    public ArrayList<VisualNotification> loadNotifications() {
        return notificationDao.loadNotifications();
    }

    public VisualNotification loadNotification(int id) {
        return notificationDao.loadNotification(id);
    }

    public VisualNotification loadDailyNoti() {
        return notificationDao.loadNotification(com.tenqube.visual_third.Constants.ReportAlarmType.DAILY.name().toLowerCase());
    }

    public void updateNotiStatus(int id, boolean enabled) {
        notificationDao.updateNotiStatus(id, enabled);
    }

    public boolean isActiveTranPopup() {
        return prefManager.isEnabled(PrefManager.TRAN_POPUP, false);
    }

    public void setTranPopup(boolean shouldShow) {
        prefManager.saveBoolean(PrefManager.TRAN_POPUP, shouldShow);
    }

    public Currency loadCurrencyInfo(String from, String to) {
        return currencyDao.loadCurrencyInfo(from, to);
    }

    public void updateDeletedTransactionById(Integer[] ids) {
        transactionDao.updateDeletedTransactionById(ids);
    }

    private void deleteTransactions() {
        ArrayList<Integer> ids = transactionDao.loadDeletedTranIds();
        if(!isEmpty(ids))
            transactionDao.deleteTransactions(ids);
    }

    public int insertTransaction(InsertTransactionRequest insertTransactionRequest) {
        return transactionDao.insertTransaction(insertTransactionRequest);
    }

    public JoinedTransaction loadJoinedTransaction(int tranId) {
        return transactionDao.loadJoinedTransaction(tranId);
    }

    public ArrayList<JoinedTransaction> loadJoinedTransactions() {
        return transactionDao.loadJoinedTransactions();
    }

    public int loadTranId(String identifier) {
        return transactionDao.loadTranId(identifier);
    }

    public Transaction loadApplyAllTran(String keyword) {
        if(TextUtils.isEmpty(keyword)) return  null;
        return transactionDao.loadApplyAllTran(keyword);
    }

    // load
    public ArrayList<Transactions.Transaction> loadTransactions(TransactionRequest transactionRequest) {
        if(transactionRequest.hasStartDay()) {
            transactionRequest.setStartDay(getStartDay());
        }
        ArrayList<JoinedTransaction> transactions = transactionDao.loadJoinedTransactions(transactionRequest);
        return toJsTransactions(context, transactions);
    }

    public ArrayList<com.tenqube.visual_third.model.analysis.Transaction> loadAnalysisTransactions(TransactionRequest transactionRequest) {
        if(transactionRequest.hasStartDay()) {
            transactionRequest.setStartDay(getStartDay());
        }
        ArrayList<JoinedTransaction> transactions = transactionDao.loadJoinedTransactions(transactionRequest);
        return toAnalysisTran(transactions);
    }

    public ArrayList<Transactions.Transaction> loadTransactions(TransactionByIdsRequest transactionRequest) {
        ArrayList<JoinedTransaction> transactions = transactionDao.loadJoinedTransactions(transactionRequest);
        return toJsTransactions(context, transactions);
    }

    private com.tenqube.visual_third.model.api.TransactionRequest loadNotSyncedTransactions() {
        ArrayList<JoinedTransaction> transactions = transactionDao.loadNotSyncedTransactions();
        if(!transactions.isEmpty())
            return new com.tenqube.visual_third.model.api.TransactionRequest(toApiTransactions(transactions));
        else
            return null;
    }

    public ArrayList<Cards.Card> loadCards() {
        ArrayList<Card> cards = cardDao.loadCards();
        return toJsCards(cards);
    }

    private void deleteCard() {
        cardDao.deleteAll();
    }

    private void insertCash() {
        cardDao.insertCash();
    }

    public ArrayList<CategoryInfo.ServerCategory> loadCategories() {
        ArrayList<Category> categories = categoryDao.loadCategories();
        return toJsCategories(categories);
    }

    public Category loadCategory(String categoryCode) {
        return categoryDao.loadCategory(categoryCode);
    }

    public ArrayList<CategoryInfo.UserCategory> loadUserCategories() {
        ArrayList<UserCategory> categories = userCategoryDao.loadUserCategories();
        return toJsUserCategories(categories);
    }

    public ArrayList<NotificationApp> loadNotiCatchedApps() {

        ArrayList<NotificationApp> results = notificationAppDao.findAll();
        if(results.isEmpty()) {
            notificationAppDao.insertNotificationApp(context);
        }
        return notificationAppDao.findAll();
    }

    private void updateSyncedTransactions(com.tenqube.visual_third.model.api.TransactionRequest transactionRequest) {
        ArrayList<String> ids = new ArrayList<>();
        ArrayList<com.tenqube.visual_third.model.api.TransactionRequest.Transaction> transactions = transactionRequest.getTransactions();

        for(com.tenqube.visual_third.model.api.TransactionRequest.Transaction tran : transactions) {
            ids.add(tran.getIdentifier());
        }
        transactionDao.updateSyncedTransactions(ids);
    }

    public void updateTransaction(UpdateTransactionRequest updateTransactionRequest) {
        transactionDao.updateTransaction(updateTransactionRequest);
    }


    private int loadCardId(String cardName, int cardType, int cardSubType) {
        int cardId = cardDao.getCardId(cardName, cardType, cardSubType);
        if(cardId == -1) {
            cardId = cardDao.insert(cardName, cardType, cardSubType);
        }
        return cardId;
    }

    /**
     * 파싱된 결과 로컬 디비에 저장하기
     * 1. 카드 아이디 구하기 없으면 저장
     * 2. 사용자 카테고리 아이디
     * 3. 내역 저장하기
     * @param transactions 파싱된 내역
     */
    public void mergeTransactions(ArrayList<com.tenqube.visual_third.model.parser.Transaction> transactions) {

        try {
            for(com.tenqube.visual_third.model.parser.Transaction parsedTran : transactions) {

                // getCardId
                int cardId = loadCardId(parsedTran.getParsedTransaction().cardName, parsedTran.getParsedTransaction().cardType, parsedTran.getParsedTransaction().cardSubType);
                parsedTran.setCardId(cardId);

                // getUserCateId
                int userCateId = userCategoryDao.getCategoryId(parsedTran.getParsedTransaction().dwType == Constants.DWType.DEPOSIT.ordinal() ?
                        "901010" : "101010");

                parsedTran.setUserCateId(userCateId);

                // merge transaction
                transactionDao.mergeTransaction(parsedTran);
            }
        } catch (Exception e) {

        }

    }

    public void insertTransaction(SyncTransaction syncTransaction) {
        try {

            // getCardId
            int cardId = loadCardId(syncTransaction.cardName, syncTransaction.cardType, syncTransaction.cardSubType);

            // getUserCateId
            String repCode = (syncTransaction.categoryCode/10000)+ "1010";
            int userCateId = userCategoryDao.getCategoryId(repCode);
            if(cardId != -1 && userCateId != -1)
                transactionDao.insertTransaction(syncTransaction, cardId, userCateId);

        } catch (Exception e) {
        }
    }

    /**
     * 검색 후 업데이트 할 부분
     * @param response 검색된 결과 값
     */
    public void updateTransactions(SearchCompanyResponse response) {

        try {
            // userCateId 구하기
            for(SearchCompanyResponse.TranCompany tranCompany : response.getResults()) {
                int userCateId = userCategoryDao.getCategoryId(tranCompany.getCategory().getRepCode());
                transactionDao.updateTransaction(tranCompany, userCateId);
            }
        } catch (Exception e) {
        }

    }

    public void updateTransaction(SearchCompanyResponse.TranCompany tranCompany, boolean isAll) {

        try {
            // userCateId 구하기
            int userCateId = userCategoryDao.getCategoryId(tranCompany.getCategory().getRepCode());

            if(isAll) {
                transactionDao.updateTransactionByKeyword(tranCompany, userCateId);
            } else {
                transactionDao.updateTransaction(tranCompany, userCateId);

            }
        } catch (Exception e) {

        }

    }

    public void saveStartDay(int startDay) {
        prefManager.saveIntValue(START_DAY, startDay);
    }

    public int getStartDay() {
        return prefManager.loadIntValue(START_DAY, 1);
    }

    public int insertCard(String name, int type) {
        return cardDao.insert(name, type, 0);
    }

    public boolean isDev() {
        return com.tenqube.visual_third.Constants.DEV.equals(prefManager.loadStringValue(PrefManager.QUALIFIER, com.tenqube.visual_third.Constants.DEV));
    }

    public boolean shouldBulk() {
        return !prefManager.isEnabled(PrefManager.BULK_EXECUTED, false);
    }

    public void saveBulk() {
        prefManager.saveBoolean(PrefManager.BULK_EXECUTED, true);
    }

    public String getAuthority() {
        return prefManager.loadStringValue(PrefManager.AUTHORITY,
                                context.getApplicationContext().getPackageName());
    }

    public String getAdType() {
        return prefManager.loadStringValue(PrefManager.AD_RATIO, DEFAULT_AD_RATIO);
    }

    public void setAppNoti(boolean isActive) {
        prefManager.saveBoolean(PrefManager.APP_NOTI, isActive);
    }

    public boolean isAppNoti() {
        return prefManager.isEnabled(PrefManager.APP_NOTI, true);
    }

    public boolean isJoined() {
        return !TextUtils.isEmpty(secretKeyManager.getKey(ACCESS_TOKEN));
    }
}
