package com.tenqube.visual_third.ui;

import android.Manifest;
import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.res.Configuration;
import android.graphics.Color;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.os.Message;
import android.text.TextUtils;
import android.view.View;
import android.webkit.WebResourceRequest;
import android.webkit.WebView;
import android.webkit.WebViewClient;
import android.widget.LinearLayout;
import android.widget.Toast;

import androidx.annotation.NonNull;
import androidx.appcompat.app.AppCompatActivity;
import androidx.localbroadcastmanager.content.LocalBroadcastManager;
import androidx.swiperefreshlayout.widget.SwipeRefreshLayout;

import com.tenqube.visual_third.Constants;
import com.tenqube.visual_third.R;
import com.tenqube.visual_third.VisualServiceImpl;
import com.tenqube.visual_third.analysis.AnalysisServiceImpl;
import com.tenqube.visual_third.exception.AuthException;
import com.tenqube.visual_third.exception.ParameterException;
import com.tenqube.visual_third.manager.ExcelManager;
import com.tenqube.visual_third.manager.PrefManager;
import com.tenqube.visual_third.manager.SecretKeyManager;
import com.tenqube.visual_third.manager.VisualAlarmManager;
import com.tenqube.visual_third.model.parser.PopupData;
import com.tenqube.visual_third.parser.loader.BulkLoader;
import com.tenqube.visual_third.repository.RepositoryHolder;
import com.tenqube.visual_third.repository.ResourceRepository;
import com.tenqube.visual_third.repository.VisualRepository;
import com.tenqube.visual_third.util.PathUtil;
import com.tenqube.visual_third.util.Utils;
import com.tenqube.visual_third.util.permission.AfterPermissionGranted;
import com.tenqube.visual_third.util.permission.PermissionUtil;
import com.tenqube.visual_third.web.ActionImpl;
import com.tenqube.visual_third.web.Completion;
import com.tenqube.visual_third.web.ErrorImpl;
import com.tenqube.visual_third.web.LogImpl;
import com.tenqube.visual_third.web.RepoImpl;
import com.tenqube.visual_third.web.SystemImpl;
import com.tenqube.visual_third.web.UiImpl;
import com.tenqube.visual_third.web.VisualInterface;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.File;
import java.util.List;

import static com.tenqube.visual_third.util.Utils.isNotiEnabled;
import static com.tenqube.visual_third.util.Utils.isSDKDisabled;
import static com.tenqube.visual_third.util.Validator.notNull;
import static com.tenqube.visual_third.util.WebViewHelper.getBaseUrl;
import static com.tenqube.visual_third.util.WebViewHelper.getProgressUrl;
import static com.tenqube.visual_third.util.WebViewHelper.settings;

public class VisualWebActivity extends AppCompatActivity implements TimePickerFragment.Callback, DatePickerFragment.Callback, PermissionUtil.PermissionCallbacks{

    public static final String ARG_PATH = "ARG_PATH";
    public static final String ACTION_BROADCAST_TRANSACTION = "ACTION_BROADCAST_TRANSACTION";

    public static final String TAG = VisualServiceImpl.class.getSimpleName();

    public static final int NOTIFICATION_REQUEST_CODE = 10;
    public static final int VISUAL_REQUEST_CODE = 512;
    public static final int PERMISSION_REQUEST_CODE = 0;

    public static final int FINISH = 10;
    public static final int PROGRESS = 11;
    public static final int SHOW_PROGRESS_BAR = 12;
    public static final int ERROR = 13;
    public static final int SYNC_ERROR = 14;
    public static final int SYNC_PROGRESS = 15;
    public static final int SHOW_SYNC_PROGRESS_BAR = 16;
    public static final int CLOSE_SYNC_PROGRESS_BAR = 17;

    private BulkLoader bulkLoader;

    private VisualRepository repository;
    private ResourceRepository resourceRepository;
    private VisualAlarmManager alarmManager;
    private boolean isPageLoaded;

    private String mFailingUrl;

    private String baseUrl = "";
    private String webUrl = "";

    private String path;
    private WebView webview;

    private VisualInterface.Repo repoInterface;
    private VisualInterface.UI uiInterface;
    private VisualInterface.System systemInterface;
    private VisualInterface.Error errorInterface;
    private VisualInterface.Log logInterface;
    private VisualInterface.Action actionInterface;

    private SwipeRefreshLayout swipeRefreshLayout;
    private LinearLayout errorView;
    private boolean isError;

    private PrefManager prefManager;

    private boolean isLoading;
    private long startTime;

    @SuppressLint({"JavascriptInterface", "AddJavascriptInterface", "SetJavaScriptEnabled"})
    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        try {
            startTime = System.currentTimeMillis();
            prefManager = PrefManager.getInstance(getApplicationContext());
             Utils.LOGD("onCreate", "start VisualActivity");

            if(isSDKDisabled(getApplicationContext())) {
                throw new Exception("SDK disabled");
            }

            repository = RepositoryHolder.getInstance(getApplicationContext()).getVisualRepository();

            resourceRepository = RepositoryHolder.getInstance(getApplicationContext()).getResourceRepository();

            alarmManager = VisualAlarmManager.getInstance(getApplicationContext());

            path = getIntent().getStringExtra(ARG_PATH);
            if(path == null) {
                path = "";
            }

            setContentView(R.layout.activity_visual_web);

            swipeRefreshLayout = findViewById(R.id.swipe_refresh_layout);
            swipeRefreshLayout.setEnabled(false);
            swipeRefreshLayout.setColorSchemeResources(
                    R.color.colorPopupRed,
                    R.color.colorPopupRed,
                    R.color.colorPopupRed);

            swipeRefreshLayout.setOnRefreshListener(new SwipeRefreshLayout.OnRefreshListener() {
                @Override
                public void onRefresh() {
                    swipeRefreshLayout.setRefreshing(false);
                    AnalysisServiceImpl.shouldRefresh = true;
                    webview.reload();

                }
            });

            //error view 처리
            errorView = findViewById(R.id.error_container);
            errorView.setVisibility(View.GONE);

            findViewById(R.id.retry).setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    retry();
                }
            });

            webview = findViewById(R.id.visual_web_view);

            // javascriptInterface
            errorInterface = new ErrorImpl(this, webview);
            logInterface = new LogImpl(this, webview, errorInterface);

            repoInterface = new RepoImpl(this, webview, errorInterface, repository, new AnalysisServiceImpl(this, ""), alarmManager);
            uiInterface = new UiImpl(this, webview, errorInterface, new BottomDialog(this));
            systemInterface = new SystemImpl(this, webview, errorInterface);
            actionInterface = new ActionImpl(this, webview);

            webview.setBackgroundColor(Color.TRANSPARENT);

            webview.addJavascriptInterface(errorInterface, "visualError");
            webview.addJavascriptInterface(repoInterface, "visualRepo");
            webview.addJavascriptInterface(uiInterface, "visualUI");
            webview.addJavascriptInterface(systemInterface, "visualSystem");
            webview.addJavascriptInterface(logInterface, "visualLog");
            webview.addJavascriptInterface(actionInterface, "visualAction");

            // settings
            settings(webview, repository.isDev());

            // error
            webview.setWebViewClient(new WebViewClient() {

                @Override
                public boolean shouldOverrideUrlLoading(WebView view, WebResourceRequest request) {
                    Utils.LOGD("shouldOverrideUrlLoading", "수행시간: " + ((System.currentTimeMillis() - startTime) / 1000) + "초");

                    if(!view.getUrl().contains(webUrl)) {
                        finish();
                        return  true;
                    }
                    return super.shouldOverrideUrlLoading(view, request);
                }

                @Override
                public boolean shouldOverrideUrlLoading(WebView view, String url) {

                    if(!url.contains(webUrl)) {
                        finish();
                        return  true;
                    }
                    return super.shouldOverrideUrlLoading(view, url);
                }

                public void onReceivedError(WebView view, int errorCode, String description, String failingUrl) {

                    isError = true;
                    mFailingUrl = failingUrl;
                    loadErrorUrl();
                }

                @Override
                public void onPageFinished(WebView view, String url) {
                    if (!isError) errorView.setVisibility(View.GONE);
                    isError = false;
                    Utils.LOGD("onPageFinished", "수행시간: " + ((System.currentTimeMillis() - startTime) / 1000) + "초, url:" + url);
                    super.onPageFinished(view, url);
                }
            });

            if(shouldRequestOverlayPermission()) {
                VisualPermissionActivity.startActivity(this, PERMISSION_REQUEST_CODE);
            } else {
                startVisual();
            }

        } catch (Exception e) {

            finish();
        }
    }

    private BroadcastReceiver transactionReceiver = new BroadcastReceiver() {

        @Override
        public void onReceive(Context context, final Intent intent) {

            try {
                final int tranId = intent.getIntExtra("tranId", 0);
                webview.post(new Runnable() {
                    @Override
                    public void run() {
                        if(tranId != 0)
                            webview.loadUrl("javascript:window.onTransactionReceived" + "(" + tranId + ");");
                    }
                });

            } catch (Exception e) {
            }
        }
    };

    private void startVisual() {
        loadUrl(path);

        if(repository.shouldBulk()) { // 벌크 파싱 진행
            // 벌크 파싱 시작
            startBulkParsing(true);
        } else {

            // 알람 설정
            alarmManager.setAlarms();

            // 리소스 동기화
            resourceRepository.sync();
        }
    }

    //bulk
    private Handler handler = new Handler(new Handler.Callback() { // 핸들러로 전달받은 값 progress 값 세팅해주기
        @Override
        public boolean handleMessage(final Message msg) {


            try {
                switch (msg.what) {
                    case PROGRESS:
                        if(webview != null) {
                            Bundle bundle = msg.getData();
                            if(bundle != null) {
                                webview.loadUrl("javascript:window.onProgress" + "(" + bundle.getInt("now") + "," + bundle.getInt("total") + ");");
                            }
                        }
                        break;

                    case SYNC_PROGRESS:
                        break;

                    case SHOW_PROGRESS_BAR:
                        isLoading = true;
                        Utils.LOGD("BulkHandler", "startBulk handleMessage SHOW_PROGRESS_BAR");
                        loadProgressUrl(Constants.ProgressType.BULK);
                        break;
                    case SHOW_SYNC_PROGRESS_BAR:
                        loadProgressUrl(Constants.ProgressType.SYNC);
                        isLoading = true;

                        break;

                    case CLOSE_SYNC_PROGRESS_BAR:
                        isLoading = false;
                        break;

                    case SYNC_ERROR:
                        onSyncError();
                        onBulkFinish(false, false);
                        break;
                    case FINISH:
                        onBulkFinish(true, true);
                        break;

                    case ERROR:
                    default:
                        onBulkFinish(false, true);
                        break;
                }
            } catch (Exception e) {
                onBulkFinish(false, true);
            }

            return false;
        }
    });

    private void onSyncError() {
        if(isLoading) {
            webview.postDelayed(new Runnable() {
                @Override
                public void run() {
                    //onSyncError
                    if(webview != null)
                        webview.loadUrl("javascript:window.onSyncError();");
                }
            }, 200);
        }
    }

    private void onBulkFinish(final boolean isSuccess,final boolean shouldReload) {
        Utils.LOGD("BulkHandler", "onBulkFinish");
        isLoading = false;
        repository.saveBulk();

        if(webview != null && shouldReload) {
            startVisual();
        }
        if(completion != null) {
            completion.onDataLoaded(isSuccess);
            completion = null;
        }
    }

    /**
     * 1. 퍼미션 체크
     * 2. 프로그레스 다이얼로그 show
     * 3. 지난 6개월 문자 파싱 시작
     */
    private void startBulkParsing(final boolean isInit) {

        new Thread(new Runnable() {
            @Override
            public void run() {
                try {
                    // 1. 프로그레스 다이얼로그 show
                    if(isInit) {
                        handler.sendEmptyMessage(SHOW_SYNC_PROGRESS_BAR);
                    } else {
                        handler.sendEmptyMessage(SHOW_PROGRESS_BAR);
                    }

                    bulkLoader = new BulkLoader(VisualWebActivity.this, handler, new BulkLoader.OnSmsLoadFinished() {
                        @Override
                        public void onFinished(boolean isError) {
                            try {
                                if(handler != null)
                                    handler.sendEmptyMessage(FINISH);
                            } catch (Exception e) {
                            }
                        }
                    });

                    // 3. 지난 6개월 문자 파싱 시작
                    bulkLoader.doParsing(isInit);

                } catch (Exception e) {
                    handler.sendEmptyMessage(ERROR);
                }

            }
        }).start();
    }

    public void changeStatusBarColor(String color) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            try {
                getWindow().setStatusBarColor(Color.parseColor(color));
            } catch (Exception e) {
            }
        }
    }

    public void onPagedLoaded() {
        runOnUiThread(new Runnable() {
            @Override
            public void run() {
                isPageLoaded = true;
            }
        });
    }

    private void loadUrl(String path) {
        if(webview != null && !isLoading) {
            isPageLoaded = false;
            if(path == null) path = "";
            webUrl = prefManager.loadStringValue(PrefManager.WEB_URL, "");
            if(TextUtils.isEmpty(webUrl)){
                finish();
                return;
            }
            baseUrl = getBaseUrl(webUrl);

            Utils.LOGD("loadUrl", "수행시간: " + ((System.currentTimeMillis() - startTime) / 1000) + "초, url:" + baseUrl);

            webview.loadUrl(baseUrl + path);
        }
    }

    private void loadProgressUrl(Constants.ProgressType progressType) {
        if(webview != null) {
            isPageLoaded = false;
            webUrl = prefManager.loadStringValue(PrefManager.WEB_URL, "");
            if(TextUtils.isEmpty(webUrl)){
                finish();
                return;
            }
            baseUrl = getProgressUrl(webUrl, progressType);
            webview.loadUrl(baseUrl);
        }
    }

    private void loadErrorUrl() {
        isPageLoaded = false;
        if(swipeRefreshLayout != null) swipeRefreshLayout.setEnabled(false);
        errorView.setVisibility(View.VISIBLE);
    }

    @Override
    protected void onActivityResult(int requestCode, int resultCode, final Intent data) {
        super.onActivityResult(requestCode, resultCode, data);

        if (requestCode == NOTIFICATION_REQUEST_CODE) {
            boolean isEnabled = isNotiEnabled(getApplicationContext());

            JSONObject jsonObject = new JSONObject();
            try {
                jsonObject.put("isNotiEnabled", isEnabled);
                String json = jsonObject.toString();
                if (systemInterface != null) systemInterface.onNotiCallback(json);

            } catch (JSONException e) {

            }
        } else if (requestCode == PERMISSION_REQUEST_CODE) {
            if(shouldRequestOverlayPermission()) {
                finish();
            } else {
                startVisual();
            }
        }
    }

    @Override
    public void onBackPressed() {

        try {
            if(isPageLoaded && uiInterface != null) {
                uiInterface.onFinish();
            } else {
                super.onBackPressed();
            }
        } catch (Exception e) {
            super.onBackPressed();
        }
    }

    @Override
    public void onResume() {
        super.onResume();
        LocalBroadcastManager.getInstance(this).registerReceiver(transactionReceiver, new IntentFilter(ACTION_BROADCAST_TRANSACTION));
        repository.setVisualActive(true);
        overridePendingTransition(0, 0);
    }

    @Override
    public void onPause() {
        super.onPause();
        LocalBroadcastManager.getInstance(this).unregisterReceiver(transactionReceiver);
        repository.setVisualActive(false);
        overridePendingTransition(0, 0);
    }

    @Override
    public void onCalendar(String date, String callback) {
        uiInterface.onCalendar(date, callback);
    }

    public void onScrollChanged(final int pos) {
    }

    public void setRefreshEnabled(final boolean enabled) {
        swipeRefreshLayout.post(new Runnable() {
            @Override
            public void run() {
                swipeRefreshLayout.setEnabled(enabled);
            }
        });
    }

    public void reload() {
        if(webview != null)  {
            webview.post(new Runnable() {
                @Override
                public void run() {
                    webview.reload();
                }
            });
        }
    }

    public void retry() {
        if(webview != null)  {
            webview.post(new Runnable() {
                @Override
                public void run() {
                    webview.loadUrl(mFailingUrl);
                }
            });
        }
    }

    public static final String[] PERMISSIONS_STORAGE = {Manifest.permission.READ_EXTERNAL_STORAGE, Manifest.permission.WRITE_EXTERNAL_STORAGE};
    public static final int STORAGE = 1;

    public Completion<Boolean> completion;

    public void exportExcel(Completion<Boolean> completion) {
        this.completion = completion;
        exportExcel();
    }

    public void restore(Completion<Boolean> completion) {
        this.completion = completion;
        startBulkParsing(false);
    }

    @AfterPermissionGranted(STORAGE)
    public void exportExcel() {
        if (PermissionUtil.hasPermissions(
                this, PERMISSIONS_STORAGE)) {

            ExcelManager.getInstance(VisualWebActivity.this).export(new ExcelManager.Callback() {
                @Override
                public void onCompleted(File file) {

                    boolean isSuccess = file != null;
                    if(isSuccess) {
                        // 인텐트 공유
                        String authority = repository.getAuthority();
                        try {
                            Intent intent = new Intent(Intent.ACTION_SEND);
                            intent.setType("application/octet-stream");
                            intent.putExtra(Intent.EXTRA_STREAM, Utils.getUri(getApplicationContext(), authority, file));
                            startActivity(Intent.createChooser(intent, "공유"));
                            Toast.makeText(getApplicationContext(),getString(R.string.visual_export_excel_success_msg, file.getName()), Toast.LENGTH_SHORT).show();

                        } catch (Exception e){

                            String log = "authority:" + authority  + "\n FileProvider Error";
                            Toast.makeText(getApplicationContext(), log, Toast.LENGTH_SHORT).show();
                            completion.onDataLoaded(false);
                            return;
                        }
                    }
                    completion.onDataLoaded(isSuccess);
                }
            });

        } else {
            PermissionUtil.requestPermissions(
                    this,
                    STORAGE,
                    PERMISSIONS_STORAGE);
        }
    }

    @Override
    public void onPermissionsGranted(int requestCode, List<String> list) {

        if (requestCode == STORAGE) {
            exportExcel();
        }
    }

    @Override
    public void onPermissionsDenied(int requestCode, List<String> list) {
        Toast.makeText(getApplicationContext(), "엑셀 파일 내보내기를 위해 외부 저장소에 대한 권한이 필요합니다. 권한이 없는 경우 엑셀 파일 내보내기 사용이 불가합니다.", Toast.LENGTH_SHORT).show();
    }

    @Override
    public void onRequestPermissionsResult(int requestCode, String[] permissions, int[] grantResults) {
        super.onRequestPermissionsResult(requestCode, permissions, grantResults);

        // Forward results to EasyPermissions
        PermissionUtil.onRequestPermissionsResult(requestCode, permissions, grantResults, this);
    }

    public static void startActivity(Activity activity, String path) throws AuthException {
        if(TextUtils.isEmpty(SecretKeyManager.getInstance(activity).getKey(PrefManager.ACCESS_TOKEN)))
            throw new AuthException("Need to sign up");

        Intent intent = new Intent(activity, VisualWebActivity.class);
        intent.putExtra(ARG_PATH, path);
        intent.addFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP |
                Intent.FLAG_ACTIVITY_REORDER_TO_FRONT);
        activity.startActivityForResult(intent, VISUAL_REQUEST_CODE);
    }

    public static void startActivity(Context context, String path) throws AuthException {
        if(TextUtils.isEmpty(SecretKeyManager.getInstance(context).getKey(PrefManager.ACCESS_TOKEN)))
            throw new AuthException("Need to sign up");

        Intent intent = new Intent(context, VisualWebActivity.class);
        intent.putExtra(ARG_PATH, path);
        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK |
                Intent.FLAG_ACTIVITY_CLEAR_TOP |
                Intent.FLAG_ACTIVITY_REORDER_TO_FRONT);
        context.startActivity(intent);
    }

    public static void startActivity(@NonNull Context context, @NonNull PopupData popupData) throws AuthException, ParameterException {

        notNull(context);
        notNull(popupData);

        if(TextUtils.isEmpty(SecretKeyManager.getInstance(context).getKey(PrefManager.ACCESS_TOKEN)))
            throw new AuthException("Need to sign up");

        Intent intent = new Intent(context, VisualWebActivity.class);
        intent.putExtra(ARG_PATH, PathUtil.makePath(popupData));
        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK |
                Intent.FLAG_ACTIVITY_CLEAR_TOP |
                Intent.FLAG_ACTIVITY_REORDER_TO_FRONT);
        context.startActivity(intent);
    }

    public boolean shouldRequestOverlayPermission() {
        return PermissionUtil.shouldRequestOverlayPermission(getApplication().getApplicationContext()) && repository.isActiveTranPopup();
    }

    @Override
    public void onConfigurationChanged(Configuration newConfig) {
        super.onConfigurationChanged(newConfig);
        Utils.LOGD("onConfigurationChanged", "");
        finish();
    }


}
