package com.tenqube.visual_third.util;

import android.content.Context;

import com.google.gson.Gson;
import com.tenqube.visual_third.BuildConfig;
import com.tenqube.visual_third.api.ApiKeyInterceptor;
import com.tenqube.visual_third.api.ForbiddenInterceptor;
import com.tenqube.visual_third.api.HeaderInterceptor;
import com.tenqube.visual_third.api.TokenAuthenticator;
import com.tenqube.visual_third.api.auth.AuthService;
import com.tenqube.visual_third.api.resource.ResourceApi;
import com.tenqube.visual_third.api.resource.ResourceHeaderInterceptor;
import com.tenqube.visual_third.api.resource.ResourceService;
import com.tenqube.visual_third.api.resource.ResourceServiceImpl;
import com.tenqube.visual_third.api.search.SearchHeaderInterceptor;
import com.tenqube.visual_third.db.DatabaseHelper;
import com.tenqube.visual_third.db.Db;
import com.tenqube.visual_third.db.dao.AdvertisementDao;
import com.tenqube.visual_third.db.dao.CardDao;
import com.tenqube.visual_third.db.dao.CategoryDao;
import com.tenqube.visual_third.db.dao.ContentsDao;
import com.tenqube.visual_third.db.dao.CurrencyDao;
import com.tenqube.visual_third.db.dao.NotificationAppDao;
import com.tenqube.visual_third.db.dao.NotificationDao;
import com.tenqube.visual_third.db.dao.TransactionDao;
import com.tenqube.visual_third.db.dao.UserCategoryDao;
import com.tenqube.visual_third.manager.PrefManager;
import com.tenqube.visual_third.manager.SecretKeyManager;
import com.tenqube.visual_third.manager.SignatureManager;
import com.tenqube.visual_third.repository.AnalysisRepository;
import com.tenqube.visual_third.repository.ResourceRepository;
import com.tenqube.visual_third.repository.VisualRepository;

import java.util.concurrent.TimeUnit;

import okhttp3.Interceptor;
import okhttp3.OkHttpClient;
import okhttp3.logging.HttpLoggingInterceptor;
import retrofit2.Retrofit;
import retrofit2.converter.gson.GsonConverterFactory;
import tenqube.parser.core.ParserService;

public final class VisualInjection {

    public static VisualRepository provideVisualRepository(Context context,
                                                           PrefManager prefManager,
                                                           CardDao cardDao,
                                                           CategoryDao categoryDao,
                                                           TransactionDao transactionDao,
                                                           UserCategoryDao userCategoryDao,
                                                           CurrencyDao currencyDao,
                                                           NotificationDao notificationDao,
                                                           AdvertisementDao advertisementDao,
                                                           NotificationAppDao notificationAppDao,
                                                           AppExecutors appExecutor,
                                                           SecretKeyManager secretKeyManager) {

        return VisualRepository.getInstance(context,
                prefManager,
                cardDao,
                categoryDao,
                transactionDao,
                userCategoryDao,
                currencyDao,
                notificationDao,
                advertisementDao,
                notificationAppDao,
                appExecutor,
                secretKeyManager);

    }

    public static ResourceRepository provideResourceRepository(PrefManager prefManager,
                                                               ParserService parserService,
                                                               AppExecutors appExecutors,
                                                               NotificationAppDao notificationAppDao,
                                                               CategoryDao categoryDao,
                                                               NotificationDao notificationDao,
                                                               ContentsDao contentsDao,
                                                               SignatureManager signatureManager) {
        return ResourceRepository.getInstance(prefManager,
                parserService,
                appExecutors,
                notificationAppDao,
                categoryDao,
                notificationDao,
                contentsDao,
                signatureManager);
    }

    public static AnalysisRepository provideAnalysisRepository(ContentsDao contentsDao) {
        return AnalysisRepository.getInstance(contentsDao);
    }

    public static PrefManager providePrefManager(Context context) {
        return PrefManager.getInstance(context);
    }

    public static Db provideDb(Context context) {
        DatabaseHelper dbHelper = DatabaseHelper.getInstance(context);
        return new Db(dbHelper.getWritableDatabase(), dbHelper.getReadableDatabase());
    }

    public static SecretKeyManager provideSecretKeyManager(Context context) {
        return SecretKeyManager.getInstance(context);
    }

    public static AppExecutors provideAppExecutors() {
        return new AppExecutors();
    }

    public static NotificationAppDao provideNotificationAppDao(Db db) {
        return new NotificationAppDao(db);
    }

    public static AdvertisementDao provideAdvertisementDao(Db db) {
        return new AdvertisementDao(db);
    }

    public static NotificationDao provideNotificationDao(Db db) {
        return new NotificationDao(db);
    }

    public static CurrencyDao provideCurrencyDao(Db db) {
        return new CurrencyDao(db);
    }

    public static UserCategoryDao provideUserCategoryDao(Db db) {
        return new UserCategoryDao(db);
    }

    public static TransactionDao provideTransactionDao(Db db) {
        return new TransactionDao(db);
    }

    public static CategoryDao provideCategoryDao(Db db) {
        return new CategoryDao(db);
    }

    public static CardDao provideCardDao(Db db) {
        return new CardDao(db);
    }

    public static ContentsDao provideContentsDao(Db db) {
        return new ContentsDao(db);
    }

    public static SignatureManager provideSignatureManager() {
        return SignatureManager.getInstance();
    }
    public static  ParserService provideParserService(Context context) {
        return ParserService.getInstance(context);
    }

    public static ResourceApi provideResourceApi(PrefManager prefManager) {
        return VisualInjection.provideApiService(ResourceApi.class, prefManager.loadStringValue(PrefManager.RESOURCE_URL, ""), null,
                VisualInjection.getResourceInterceptors(prefManager));

    }

    public static ResourceService provideResourceService(ResourceApi resourceApi, SignatureManager signatureManager) {
        return ResourceServiceImpl.getInstance(resourceApi, signatureManager);
    }

    public static <T> T provideApiService(Class<T> tClass, String url, AuthService authService, Interceptor... interceptors) {

        OkHttpClient okHttpClient = provideOkHttpClient(authService, interceptors);
        Retrofit retrofit = provideRetrofit(okHttpClient, url);
        return retrofit.create(tClass);
    }

    private static Retrofit provideRetrofit(OkHttpClient okHttpClient, String url) {
        if(!"/".equals(url.substring(url.length() - 1))) {
            url += "/";
        }
        return new Retrofit.Builder()
                .baseUrl(url)
                .client(okHttpClient)
                .addConverterFactory(GsonConverterFactory.create(new Gson()))
                .build();
    }

    private static OkHttpClient provideOkHttpClient(AuthService authService, Interceptor... interceptors) {

        OkHttpClient.Builder okHttpClientBuilder = new OkHttpClient.Builder();
        okHttpClientBuilder.connectTimeout(5, TimeUnit.SECONDS);
        okHttpClientBuilder.readTimeout(5, TimeUnit.SECONDS);
        okHttpClientBuilder.writeTimeout(5, TimeUnit.SECONDS);

        if(BuildConfig.DEBUG) {
            HttpLoggingInterceptor httpLoggingInterceptor = new HttpLoggingInterceptor();
            httpLoggingInterceptor.setLevel(HttpLoggingInterceptor.Level.BODY);
            okHttpClientBuilder.addInterceptor(httpLoggingInterceptor);
        }

        if(authService != null) {
            okHttpClientBuilder.authenticator(new TokenAuthenticator(authService));
        }

        for(Interceptor interceptor : interceptors) {
            okHttpClientBuilder.addInterceptor(interceptor);
        }

        return okHttpClientBuilder.build();
    }

    public static Interceptor[] getSearchInterceptors(PrefManager prefManager, SecretKeyManager secretKeyManager) {

        Interceptor[] interceptors = new Interceptor[1];
        SearchHeaderInterceptor searchHeaderInterceptor = new SearchHeaderInterceptor(prefManager, secretKeyManager);
        interceptors[0] = searchHeaderInterceptor;

//        ForbiddenInterceptor forbiddenInterceptor = new ForbiddenInterceptor(authService);
//        interceptors[1] = forbiddenInterceptor;

        return interceptors;
    }

    static Interceptor[] getResourceInterceptors(PrefManager prefManager) {

        Interceptor[] interceptors = new Interceptor[1];
        ResourceHeaderInterceptor resourceHeaderInterceptor = new ResourceHeaderInterceptor(prefManager);
        interceptors[0] = resourceHeaderInterceptor;

        return interceptors;
    }

    public static Interceptor[] getVisualInterceptors(PrefManager prefManager, AuthService authService) {

        Interceptor[] interceptors = new Interceptor[3];
        HeaderInterceptor headerInterceptor = new HeaderInterceptor(authService);
        interceptors[0] = headerInterceptor;

        ForbiddenInterceptor forbiddenInterceptor = new ForbiddenInterceptor(authService);
        interceptors[1] = forbiddenInterceptor;

        ApiKeyInterceptor apiKeyInterceptor = new ApiKeyInterceptor(prefManager);
        interceptors[2] = apiKeyInterceptor;

        return interceptors;
    }

    public static Interceptor[] getAuthInterceptors(PrefManager prefManager) {

        Interceptor[] interceptors = new Interceptor[1];
        ApiKeyInterceptor headerInterceptor = new ApiKeyInterceptor(prefManager);
        interceptors[0] = headerInterceptor;

        return interceptors;
    }

}
