package com.tenqube.visual_third.manager;

import android.content.Context;
import android.content.res.AssetManager;
import android.os.Environment;
import android.text.TextUtils;

import com.tenqube.visual_third.entity.JoinedTransaction;
import com.tenqube.visual_third.repository.RepositoryHolder;
import com.tenqube.visual_third.repository.VisualRepository;
import com.tenqube.visual_third.util.AppExecutors;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;

import jxl.CellView;
import jxl.Workbook;
import jxl.format.Border;
import jxl.format.BorderLineStyle;
import jxl.format.Colour;
import jxl.format.VerticalAlignment;
import jxl.write.Blank;
import jxl.write.DateFormat;
import jxl.write.DateTime;
import jxl.write.Label;
import jxl.write.Number;
import jxl.write.NumberFormat;
import jxl.write.WritableCellFormat;
import jxl.write.WritableFont;
import jxl.write.WritableSheet;
import jxl.write.WritableWorkbook;

public class ExcelManager {

    WritableCellFormat titleFormat;

    private static final String DATE = "날짜";
    private static final String STATE = "상태";
    private static final String AMOUNT = "금액";
    private static final String KEYWORD = "내역";
    private static final String PAYMENT = "결제수단";
    private static final String CATEGORY = "카테고리";
    private static final String MEMO = "메모";

    private static final String[] COLUMS = new String[]{"날짜", "상태", "금액", "내역", "결제수단", "카테고리", "메모"};

    private static final String FILE_NAME = "hnm-visual.xls";
    private static ExcelManager mInstance = null;
    private Context mContext;
    private VisualRepository repository;
    private AppExecutors appExecutors;

    public static ExcelManager getInstance(Context ctx) {

        synchronized (ExcelManager.class) {
            if (mInstance == null) {
                mInstance = new ExcelManager(ctx.getApplicationContext());
            }
        }
        return mInstance;
    }

    public void destroy() {
        mInstance = null;
    }

    private ExcelManager(Context context) {
        this.mContext = context;
        repository = RepositoryHolder.getInstance(context).getVisualRepository();
        appExecutors = new AppExecutors();
        try {
            titleFormat = createTitleFormat();
        } catch (Exception e) {

        }
    }

    private File createFile() throws IOException {
        //
        File path = Environment.getExternalStoragePublicDirectory(Environment.DIRECTORY_DOWNLOADS);
        if (!path.exists()) {
            path.mkdirs();
        }
        File deviceFile = new File(path, FILE_NAME);
        if (!deviceFile.exists()) {
            deviceFile.createNewFile();
        }
        return deviceFile;
    }

    private WritableCellFormat createTitleFormat() throws Exception {
        WritableFont times10pt = new WritableFont(WritableFont.ARIAL, 10);

        WritableCellFormat titleFormat = new WritableCellFormat(times10pt);
        titleFormat.setWrap(false);
        titleFormat.setVerticalAlignment(VerticalAlignment.CENTRE);
        titleFormat.setBorder(Border.ALL, BorderLineStyle.THIN, Colour.WHITE);

        return titleFormat;
    }

    private WritableCellFormat createNumberFormat() throws Exception {

        WritableFont times10pt = new WritableFont(WritableFont.ARIAL, 10);

        jxl.write.NumberFormat numberFormat = new NumberFormat("###,###");

        WritableCellFormat integerFormat = new WritableCellFormat(times10pt, numberFormat);
        integerFormat.setWrap(false);
        integerFormat.setVerticalAlignment(VerticalAlignment.CENTRE);
        integerFormat.setBorder(Border.ALL, BorderLineStyle.THIN, Colour.WHITE);

        return integerFormat;
    }

    private WritableCellFormat createDateTimeFormat() throws Exception {
        WritableFont times10pt = new WritableFont(WritableFont.ARIAL, 10);
        DateFormat userdatafmt = new DateFormat("yyyy-MM-DD hh:mm:ss");
        WritableCellFormat dataformat = new WritableCellFormat(times10pt, userdatafmt);
        dataformat.setWrap(false);
        dataformat.setVerticalAlignment(VerticalAlignment.CENTRE);
        dataformat.setBorder(Border.ALL, BorderLineStyle.THIN, Colour.WHITE);

        return dataformat;
    }

    public interface Callback {
        void onCompleted(File file);
    }

    public void export(final Callback completion) {

        appExecutors.diskIO().execute(new Runnable() {
            @Override
            public void run() {
                AssetManager am = mContext.getAssets();
                InputStream inStream;
                File deviceFile = null;
                try {
                    inStream = am.open(FILE_NAME);
                    Workbook tempWorkbook = Workbook.getWorkbook(inStream);

                    deviceFile = createFile();

                    // WritableWorkbook 을 visual.xls 파일을 기반으로 동일하게 생성합니다.
                    WritableWorkbook writableWorkbook = Workbook.createWorkbook(deviceFile, tempWorkbook);
                    WritableSheet excelSheet = writableWorkbook.getSheet(0);

                    addHeaders(excelSheet);

                    ArrayList<JoinedTransaction> transactions = repository.loadJoinedTransactions();

                    // 각 조건에 맞게 텍스 트변형 후 엑셀 인서트
                    int row = 1;
                    for(JoinedTransaction transaction : transactions) {

                        addLines(excelSheet, transaction, row);

                        row++;
                    }
                    writableWorkbook.write();
                    writableWorkbook.close();

                } catch (Exception e) {

                } finally {
                    final File finalDeviceFile = deviceFile;
                    appExecutors.mainThread().execute(new Runnable() {
                        @Override
                        public void run() {
                            completion.onCompleted(finalDeviceFile);
                        }
                    });

                }
            }
        });


    }

    private void addLabel(WritableSheet sheet, int column, int row, String s, WritableCellFormat titleFormat, CellView cv)  {

        try {
            cv.setFormat(titleFormat);
            if(s==null|| TextUtils.isEmpty(s)){
                Blank blank = new Blank(column, row, titleFormat); // 빈 셀(열,행,포멧)
                sheet.addCell(blank);
                sheet.setColumnView(row,cv);
            } else {
                Label label;
                label = new Label(column, row, s, titleFormat);
                sheet.addCell(label);
                sheet.setColumnView(row,cv);
            }
        } catch (Exception e) {

        }
    }

    private void addNumber(WritableSheet sheet, int column, int row, int spentMoney,CellView cv, WritableCellFormat integerFormat) {

        try {
            cv.setFormat(integerFormat);
            Number number = new Number(column, row, spentMoney, integerFormat);
            sheet.addCell(number);
            sheet.setColumnView(row,cv);
        } catch (Exception e) {

        }
    }

    private void addDate(WritableSheet sheet, int column, int row, Calendar s, WritableCellFormat titleFormat, CellView cv, WritableCellFormat dataformat) {


        try {
            if(s == null){
                cv.setFormat(titleFormat);
                Blank blank = new Blank(column, row, titleFormat); // 빈 셀(열,행,포멧)
                sheet.addCell(blank);
                sheet.setColumnView(row,cv);

            }else{
//            hh:mm:ss

                Date now = s.getTime();
                cv.setFormat(dataformat);
                DateTime datetime = new DateTime(column, row,now,dataformat);
                sheet.addCell(datetime);
                sheet.setColumnView(row,cv);

            }
        } catch (Exception e) {

        }

    }

    private void addHeaders(WritableSheet sheet) {
        int size = COLUMS.length;
        CellView cv = new CellView();
        cv.setAutosize(true);
        for(int column = 0 ; column < size ; column++){
            addLabel(sheet, column, 0, COLUMS[column], titleFormat, cv);
        }
    }

    private void addLines(WritableSheet sheet, JoinedTransaction transaction, int row){

        try {

            CellView cv = new CellView();
            cv.setAutosize(true);
            int size = COLUMS.length;
            for(int column = 0 ; column < size ; column++){

                String value = COLUMS[column];

                switch (value){
                    case DATE:
                        addLabel(sheet, column, row, "" + transaction.getTransaction().getSpentDate(), titleFormat, cv);
                        break;
                    case STATE:
                        String state = transaction.getTransaction().getSpentMoney() < 0 ? "취소" : "승인";
                        addLabel(sheet, column, row, state, titleFormat, cv);

                        break;
                    case AMOUNT:
                        int spentMoney = (int) transaction.getTransaction().getSpentMoney();
                        addLabel(sheet, column, row, "" + spentMoney, titleFormat, cv);
                        break;

                    case KEYWORD:
                        addLabel(sheet, column, row, "" + transaction.getTransaction().getKeyword(), titleFormat, cv);
                        break;

                    case PAYMENT:
                        addLabel(sheet, column, row, "" + transaction.getCard().getName(), titleFormat, cv);
                        break;

                    case CATEGORY:
                        addLabel(sheet, column, row, "" + transaction.getCategory().getLarge(), titleFormat, cv);
                        break;

                    case MEMO:
                        addLabel(sheet, column, row, "" + transaction.getTransaction().getMemo(), titleFormat, cv);
                        break;

                }

            }
        } catch (Exception e){

        }
    }





}
