package com.tenqube.visual_third;

import android.app.Activity;
import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.tenqube.visual_third.analysis.AnalysisHelper;
import com.tenqube.visual_third.analysis.AnalysisService;
import com.tenqube.visual_third.analysis.AnalysisServiceImpl;
import com.tenqube.visual_third.exception.AuthException;
import com.tenqube.visual_third.exception.ParameterException;
import com.tenqube.visual_third.manager.AnswerManager;
import com.tenqube.visual_third.manager.VisualAlarmManager;
import com.tenqube.visual_third.model.analysis.Analysis;
import com.tenqube.visual_third.model.analysis.Transaction;
import com.tenqube.visual_third.model.js.LogRequest;
import com.tenqube.visual_third.model.js.TransactionRequest;
import com.tenqube.visual_third.repository.RepositoryHolder;
import com.tenqube.visual_third.repository.VisualRepository;
import com.tenqube.visual_third.ui.OnResultListener;
import com.tenqube.visual_third.ui.VisualWebActivity;
import com.tenqube.visual_third.util.AppExecutors;
import com.tenqube.visual_third.util.Validator;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;

import tenqube.parser.core.ParserService;

import static com.tenqube.visual_third.util.Utils.sumBy;
import static com.tenqube.visual_third.util.Validator.notNull;
import static com.tenqube.visual_third.util.Validator.notZero;

public class VisualServiceImpl implements VisualService {

    public static final String TAG = VisualServiceImpl.class.getSimpleName();

    private Context context;
    private AnalysisService service;
    private ParserService parserService;
    private VisualRepository repository;
    private VisualAlarmManager alarmManager;
    private AppExecutors appExecutors = new AppExecutors();

    public VisualServiceImpl(@NonNull Context context,
                             @NonNull String apiKey,
                             @NonNull String qualifier,
                             @NonNull String authority) throws ParameterException {

        Validator.notNull(context);

        Validator.isNotEmpty(apiKey);

        Validator.isNotEmpty(authority);

        Validator.in(qualifier, Constants.DEV, Constants.PROD);

        this.context = context;

        repository = RepositoryHolder.getInstance(context).getVisualRepository();

        repository.saveSDKInfo(apiKey, qualifier, authority);

        parserService = ParserService.getInstance(context);

        alarmManager = VisualAlarmManager.getInstance(context);

        service = new AnalysisServiceImpl(context, qualifier);

    }

    @Override
    public void startVisual(final @Nullable Activity activity,
                            final @NonNull String uid,
                            final @NonNull String path,
                            final @NonNull String adId,
                            final @NonNull Constants.Gender gender,
                            final @NonNull OnResultListener resultListener) throws ParameterException {

        Validator.isNotEmpty(uid);
        Validator.notNull(resultListener);
        Validator.notNull(path);

        repository.signUp(uid, new OnResultListener() {
            @Override
            public void onResult(int signUpResult, String msg) {
                try {
                    //가입 성공시에만 isAgree 값에 맞게 설정정보 업데이트 처리
                    if (signUpResult == Constants.SignupResult.SUCCESS ||
                    signUpResult == Constants.SignupResult.ALREADY_JOINED) {
                        AnswerManager.onKeyMetric(new LogRequest("startVisual"));
                        if (activity == null) {
                            VisualWebActivity.startActivity(context, path);
                        } else {
                            VisualWebActivity.startActivity(activity, path);
                        }
                    }

                } catch (AuthException e) {
                    signUpResult = Constants.SignupResult.SERVER_ERROR;
                    msg = e.toString();
                } finally {
                    resultListener.onResult(signUpResult, msg);
                }
            }
        });
    }

    @Override
    public void setVisualCallback(VisualCallback callback) {
        repository.setVisualCallback(callback);
    }

    @Override
    public void signOut(Callback<Boolean> callback) {
        repository.signOut(new Callback<Boolean>() {
            @Override
            public void onDataLoaded(Boolean value) {
                alarmManager.setAlarms();
            }
        });
    }

    @Override
    public void settingNotification(int smallIcon, @NonNull String channel, int color) throws ParameterException {
        notZero(smallIcon);
        notNull(channel);
        repository.settingNotification(smallIcon, channel, color);
    }

    @Override
    public boolean isActiveTranPopup() {
        AnswerManager.onKeyMetric(new LogRequest("isActiveTranPopup"));
        return repository.isActiveTranPopup();
    }

    @Override
    public void setTranPopup(boolean isActive) {
        AnswerManager.onKeyMetric(new LogRequest("setTranPopup"));
        repository.setTranPopup(isActive);
    }

    @Override
    public void setReportAlarm(Constants.ReportAlarmType type, boolean isActive) {
        repository.setActiveNoti(type.name().toLowerCase(), isActive);
        alarmManager.setAlarms();
    }

    @Override
    public boolean isActiveReportAlarm(Constants.ReportAlarmType type) {
        return repository.isActiveNoti(type.name().toLowerCase());
    }

    @Override
    public void setReportTest(Constants.ReportAlarmType type, int second) {
        alarmManager.setReportTest(type.name().toLowerCase(), second);
    }

    @Override
    public void setLogger(boolean isActive) {
        if(isActive() && parserService != null) {
//            parserService.setDebugMode(isActive);
        }
    }

    @Override
    public void initSDK() {
        if(isActive() && parserService != null) {
            parserService.initDb();
        }
    }

    @Override
    public void setEnabled(boolean enabled) {
        repository.setEnabled(enabled);
        if(!enabled) {
            repository.setNotiAll(false);
            alarmManager.setAlarms();
        }
    }

    @Override
    public boolean isJoined() {
        return repository.isJoined();
    }

    @Override
    public void getVisualSummary(final Callback<VisualSummary> callback) {

        // 초기화 체크
        appExecutors.diskIO().execute(new Runnable() {
            @Override
            public void run() {
                if (!repository.isJoined()) { // 가입되지 않은 경우 null 을 리턴합니다.
                    appExecutors.mainThread().execute(new Runnable() {
                        @Override
                        public void run() {
                            callback.onDataLoaded(null);
                        }
                    });

                } else {

                    Calendar calendar = Calendar.getInstance();
                    int year = calendar.get(Calendar.YEAR);
                    int month = calendar.get(Calendar.MONTH) + 1;

                    TransactionRequest request = new TransactionRequest(year, month, 3, true, "");
                    request.setDwType(tenqube.parser.constants.Constants.DWType.WITHDRAW.ordinal());
                    request.setExceptType(0);

                    ArrayList<Transaction> transactions = repository.loadAnalysisTransactions(request);

                    // empty 허용하지 않습니다.
                    final ArrayList<Analysis> monthly = service.loadAnalysisList(transactions, repository.getStartDay());
                    final VisualSummary visualSummary;

                    double sum = getCurrentMonthSum(year, month);
                    if (monthly.isEmpty()) {
                        visualSummary = new VisualSummary(sum, String.format("이번 달 지출은 %1$s 입니다.", AnalysisHelper.getLv0Currency(sum)));
                    } else {
                        Collections.shuffle(monthly);
                        Analysis analysis = monthly.get(0);
                        visualSummary = new VisualSummary(sum, analysis.getlContent());
                    }

                    // 랜덤으로 하나 뽑아서 visual summary 에 할당합니다.
                    appExecutors.mainThread().execute(new Runnable() {
                        @Override
                        public void run() {
                            callback.onDataLoaded(visualSummary);
                        }
                    });
                }
            }
        });
    }

    private double getCurrentMonthSum(int year, int month) {

        TransactionRequest currentRequest = new TransactionRequest(year, month, 0, true, "");
        currentRequest.setDwType(tenqube.parser.constants.Constants.DWType.WITHDRAW.ordinal());
        currentRequest.setExceptType(0);
        ArrayList<Transaction> currentMonthTransactions = repository.loadAnalysisTransactions(currentRequest);

        return sumBy(currentMonthTransactions);
    }

    @Override
    public void setAppNoti(boolean isActive) {
        repository.setAppNoti(isActive);
        alarmManager.setAlarms();
    }

    @Override
    public boolean isActiveAppNoti() {
        return repository.isAppNoti();
    }

    private boolean isActive() {
        return context != null;
    }
}
